# Written by Bill Bumgarner and Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: parseargs.py 119 2007-06-20 07:40:58Z camrdale-guest $

"""Functions for dealing with arguments, defaults and options."""

from types import *
from cStringIO import StringIO


def splitLine(line, COLS=80, indent=10):
    """Split an output line to a screen width (with indenting).
    
    @type line: C{string}
    @param line: the output line to split
    @type COLS: C{int}
    @param COLS: the number of columns in the screen width
        (optional, defaults to 80)
    @type indent: C{int}
    @param indent: the number of columns to indent the output lines with spaces
        (optional, defaults to 10)
    @rtype: C{string}
    @return: the multiple lines formatted for output
    
    """
    
    indent = " " * indent
    width = COLS - (len(indent) + 1)
    if indent and width < 15:
        width = COLS - 2
        indent = " "
    s = StringIO()
    i = 0
    for word in line.split():
        if i == 0:
            s.write(indent+word)
            i = len(word)
            continue
        if i + len(word) >= width:
            s.write('\n'+indent+word)
            i = len(word)
            continue
        s.write(' '+word)
        i += len(word) + 1
    return s.getvalue()

def formatDefinitions(options, COLS, presets = {}):
    """Format configuration options for printing to the user.
    
    @type options: C{list} of (C{string}, unknown, C{string})
    @param options: the name, default value, and description for each of 
        the configuration options
    @type COLS: C{int}
    @param COLS: the width of the screen to format the output for
    @type presets: C{dictionary}
    @param presets: other default values to use instead (optional)
    @rtype: C{string}
    @return: the formatted options for output
    
    """
    
    s = StringIO()
    for (longname, default, doc) in options:
        s.write('--' + longname + ' <arg>\n')
        default = presets.get(longname, default)
        if type(default) in (IntType, LongType):
            try:
                default = int(default)
            except:
                pass
        if default is not None:
            doc += ' (defaults to ' + repr(default) + ')'
        s.write(splitLine(doc,COLS,10))
        s.write('\n\n')
    return s.getvalue()


def usage(str):
    """Raise an error to indicate a problem.
    
    @type str: C{string}
    @param str: the error that occurred
    @raise ValueError: always
    
    """
    
    raise ValueError(str)


def defaultargs(options):
    """Make a dictionary of all the default options.
    
    @type options: C{list} of (C{string}, unknown, C{string})
    @param options: the name, default value, and description for each of 
        the configuration options
    @rtype: C{dictionary}
    @return: the default options, keys are the option names
    
    """
    
    l = {}
    for (longname, default, doc) in options:
        if default is not None:
            l[longname] = default
    return l
        

def parseargs(argv, options, minargs = None, maxargs = None, presets = {}):
    """Parse a list of input arguments and set the configuration options variable.
    
    All configuration options must begin with '--', must be followed by a space 
    and then the new setting to use, and must appear in the L{options} variable 
    keys. Non-options can appear anywhere in the arguments (except between an
    option and it's setting), and there must be between minargs and maxargs of
    them.
    
    @type argv: C{list} of C{string}
    @param argv: the input command-line arguments
    @type options: C{list} of (C{string}, unknown, C{string})
    @param options: the name, default value, and description for each of 
        the configuration options
    @type minargs: C{int}
    @param minargs: the minimum number of non-option arguments required 
        (optional, defaults to no minimum)
    @type maxargs: C{int}
    @param maxargs: the maximum number of non-option arguments required 
        (optional, defaults to no maximum)
    @type presets: C{dictionary}
    @param presets: other default values to use instead (optional)
    @rtype: (C{dictionary}, C{list})
    @return: the configuration options, and any remaining non-option arguments
    @raise ValueError: by calling L{usage} if something goes wrong
    
    """
    
    config = {}
    longkeyed = {}
    for option in options:
        longname, default, doc = option
        longkeyed[longname] = option
        config[longname] = default
    for longname in presets.keys():        # presets after defaults but before arguments
        config[longname] = presets[longname]
    options = []
    args = []
    pos = 0
    while pos < len(argv):
        if argv[pos][:2] != '--':
            args.append(argv[pos])
            pos += 1
        else:
            if pos == len(argv) - 1:
                usage('parameter passed in at end with no value')
            key, value = argv[pos][2:], argv[pos+1]
            pos += 2
            if not longkeyed.has_key(key):
                usage('unknown key --' + key)
            longname, default, doc = longkeyed[key]
            try:
                t = type(config[longname])
                if t is NoneType or t is StringType:
                    config[longname] = value
                elif t in (IntType, LongType):
                    config[longname] = long(value)
                elif t is FloatType:
                    config[longname] = float(value)
                else:
                    assert 0
            except ValueError, e:
                usage('wrong format of --%s - %s' % (key, str(e)))
    for key, value in config.items():
        if value is None:
            usage("Option --%s is required." % key)
    if minargs is not None and len(args) < minargs:
        usage("Must supply at least %d args." % minargs)
    if maxargs is not None and len(args) > maxargs:
        usage("Too many args - %d max." % maxargs)
    return (config, args)

def test_parseargs():
    """Test the L{parseargs} function for errors."""
    
    assert parseargs(('d', '--a', 'pq', 'e', '--b', '3', '--c', '4.5', 'f'), (('a', 'x', ''), ('b', 1, ''), ('c', 2.3, ''))) == ({'a': 'pq', 'b': 3, 'c': 4.5}, ['d', 'e', 'f'])
    assert parseargs([], [('a', 'x', '')]) == ({'a': 'x'}, [])
    assert parseargs(['--a', 'x', '--a', 'y'], [('a', '', '')]) == ({'a': 'y'}, [])
    try:
        parseargs([], [('a', 'x', '')])
    except ValueError:
        pass
    try:
        parseargs(['--a', 'x'], [])
    except ValueError:
        pass
    try:
        parseargs(['--a'], [('a', 'x', '')])
    except ValueError:
        pass
    try:
        parseargs([], [], 1, 2)
    except ValueError:
        pass
    assert parseargs(['x'], [], 1, 2) == ({}, ['x'])
    assert parseargs(['x', 'y'], [], 1, 2) == ({}, ['x', 'y'])
    try:
        parseargs(['x', 'y', 'z'], [], 1, 2)
    except ValueError:
        pass
    try:
        parseargs(['--a', '2.0'], [('a', 3, '')])
    except ValueError:
        pass
    try:
        parseargs(['--a', 'z'], [('a', 2.1, '')])
    except ValueError:
        pass

