# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Francois.Ingelrest@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import pygst
pygst.require('0.10')
import gobject, gst, modules, os.path

from tools import consts


MOD_INFO = ('GStreamer Player', 'GStreamer Player', '', [], True, False)


class GSTPlayer(modules.Module):
    """ This module is the 'real' GStreamer player """

    def __init__(self):
        """ Constructor """
        modules.Module.__init__(self, (consts.MSG_CMD_PLAY, consts.MSG_CMD_STOP,   consts.MSG_CMD_TOGGLE_PAUSE,
                                       consts.MSG_CMD_SEEK, consts.MSG_CMD_BUFFER, consts.MSG_EVT_APP_STARTED, consts.MSG_CMD_SET_VOLUME))


    def onAppStarted(self):
        """ This is the real initialization function, called when this module has been loaded """
        self.timer       = None    # Used to monitor the current track
        self.bufPos      = None    # Used when the user wants to seek a position while paused
        self.bufURI      = None    # If not None, the resource that the second player is going to play
        self.eosSignaled = False   # Prevent duplicated EOS signals
        # We use two players to avoid gaps between songs
        self.player  = gst.element_factory_make('playbin', consts.appName + '1')
        self.player2 = gst.element_factory_make('playbin', consts.appName + '2')
        # Connect the primary player to our handler
        self.bus = self.player.get_bus()
        self.bus.add_signal_watch()
        self.bus.connect('message', self.onGSTMsg)


    # Helper functions
    def isPaused(self):  return self.player.get_state()[1] == gst.STATE_PAUSED
    def isPlaying(self): return self.player.get_state()[1] == gst.STATE_PLAYING


    def timerFunc(self):
        """ Regularly called while playing """
        if not self.isPlaying():
            return True

        if self.bufPos is not None:
            self.player.seek_simple(gst.FORMAT_TIME, gst.SEEK_FLAG_FLUSH, self.bufPos*1000000000)
            self.bufPos = None
            return True

        currPosition = self.player.query_position(gst.FORMAT_TIME)[0]
        modules.postMsg(consts.MSG_EVT_TRACK_POSITION, {'seconds': int(currPosition / 1000000000)})

        # If the current track is close to its end, we must ask to buffer the next one, if any
        # If the track is *really* close to its end, start playing the buffered resource, if any
        remaining = self.player.query_duration(gst.FORMAT_TIME)[0] - currPosition
        if remaining <= 4000000000 and self.bufURI is None:
            modules.postMsg(consts.MSG_EVT_NEED_BUFFER)
        elif remaining <= 250000000 and self.bufURI is not None and self.player2.get_state()[1] != gst.STATE_PLAYING:
            self.player2.set_state(gst.STATE_PLAYING)

        return True


    def bufferResource(self, uri):
        """ Buffer the given resource with the second player """
        if self.bufURI is None:
            self.bufURI = uri
            self.player2.set_property('uri', uri)
            self.player2.set_state(gst.STATE_READY)
            # Increase the frequency of the timer, to start playing the buffered resource just when needed
            if self.timer is not None:
                gobject.source_remove(self.timer)
            self.timer = gobject.timeout_add(100, self.timerFunc)


    def onGSTMsg(self, bus, msg):
        """ GStreamer message handler """
        if msg.type in (gst.MESSAGE_EOS, gst.MESSAGE_ERROR) and not self.eosSignaled:
            self.eosSignaled = True

            if msg.type == gst.MESSAGE_EOS: modules.postMsg(consts.MSG_EVT_TRACK_ENDED_OK)
            else:                           modules.postMsg(consts.MSG_EVT_TRACK_ENDED_ERROR)

            return True

        return False


    def play(self, uri):
        """ Play the given resource """
        self.bufPos      = None
        self.eosSignaled = False
        self.player.set_state(gst.STATE_NULL)
        # If the given resource is already buffered, we're lucky and we just need to be sure that the second player is playing it
        if uri == self.bufURI:
            if self.player2.get_state()[1] != gst.STATE_PLAYING:
                self.player2.set_state(gst.STATE_PLAYING)
            # Now we need to swap the two players
            self.bus.remove_signal_watch()
            self.player, self.player2 = self.player2, self.player
            self.bus = self.player.get_bus()
            self.bus.add_signal_watch()
            self.bus.connect('message', self.onGSTMsg)
        # No luck, the buffered resource, if any, is not the correct one
        else:
            if self.player2.get_state()[1] != gst.STATE_NULL:
                self.player2.set_state(gst.STATE_NULL)
            self.player.set_property('uri', uri)
            self.player.set_state(gst.STATE_PLAYING)
        # Clear the buffer in any case
        self.bufURI = None
        # Ensure that the timer frequency is back to normal
        if self.timer is not None:
            gobject.source_remove(self.timer)
        self.timer = gobject.timeout_add(500, self.timerFunc)


    def stop(self):
        """ Stop playing """
        if self.timer is not None:
            gobject.source_remove(self.timer)
            self.timer = None

        self.player.set_state(gst.STATE_NULL)
        self.player2.set_state(gst.STATE_NULL)
        self.bufURI = None
        modules.postMsg(consts.MSG_EVT_STOPPED)


    def togglePause(self):
        """ Switch between play/pause """
        if self.isPaused():
            self.player.set_state(gst.STATE_PLAYING)
            modules.postMsg(consts.MSG_EVT_UNPAUSED)
        elif self.isPlaying():
            self.player.set_state(gst.STATE_PAUSED)
            modules.postMsg(consts.MSG_EVT_PAUSED)


    def seek(self, seconds):
        """ Jump to the given position if playing, or buffer it if paused """
        if   self.isPaused():  self.bufPos = seconds
        elif self.isPlaying(): self.player.seek_simple(gst.FORMAT_TIME, gst.SEEK_FLAG_FLUSH, seconds*1000000000)


    def setVolume(self, value):
        """ Change the volume """
        if   value < 0: value = 0
        elif value > 1: value = 1

        self.player.set_property('volume',  value)
        self.player2.set_property('volume', value)
        modules.postMsg(consts.MSG_EVT_VOLUME_CHANGED, {'value': value})


    # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if   msg == consts.MSG_CMD_STOP:         self.stop()
        elif msg == consts.MSG_CMD_PLAY:         self.play(params['uri'])
        elif msg == consts.MSG_CMD_SEEK:         self.seek(params['seconds'])
        elif msg == consts.MSG_CMD_BUFFER:       self.bufferResource(params['uri'])
        elif msg == consts.MSG_CMD_SET_VOLUME:   self.setVolume(params['value'])
        elif msg == consts.MSG_EVT_APP_STARTED:  self.onAppStarted()
        elif msg == consts.MSG_CMD_TOGGLE_PAUSE: self.togglePause()
