/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_DAEMON_CLIENT_CHANNELHANDLER_H_
#define _DECIBEL_DAEMON_CLIENT_CHANNELHANDLER_H_

#include <QtCore/QObject>
#include <QtCore/QString>

#include <Decibel/Types>

class QDBusObjectPath;

namespace QtTapioca
{
class Connection;
class Channel;
}

namespace Decibel
{

class ChannelHandlerPrivate;

/**
 * @brief A ChannelHandler is called by Decibel whenever it
 * activates a external service to handle a incoming information
 * channel.
 *
 * The ChannelHandler implements an D-Bus interface that is used
 * by the Decibel daemon to activate external components. This
 * interface is used to inform the application about which channel
 * it is supposed to handle.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ChannelHandler : public QObject
{
    Q_OBJECT
    Q_DECLARE_PRIVATE(ChannelHandler)

public:
    /** @brief Constructor. */
    explicit ChannelHandler(QObject * parent = 0);
    /** @brief Destructor. */
    virtual ~ChannelHandler();

    /**
     * @brief Decide whether this ChannelHandler is interested in
     * the given channel.
     * @param connection The Connection the Channel is part of
     * @param channel A pointer to the Channel proxy object.
     * @return true if this ChannelHandler wants to handle the new
     * Channel and false otherwise.
     *
     * This method provides a way for a ChannelHandler to decline
     * responsibility for a incoming channel. If it does accept the
     * incoming channel this method is used to setup the Channel.
     */
    virtual bool handleChannel(QtTapioca::Connection * connection,
                               QtTapioca::Channel * channel) = 0;

public slots:
    /**
     * @brief Notification slot for the D-Bus signal.
     * @param channel_info The D-Bus information required to set up
     * a Channel.
     * @return true if this ChannelHandler wants to handle the new
     * Channel and false otherwise.
     *
     * This method is called by D-Bus and sets up QtTapioca Connection
     * and Channel objects before calling the virtual HandleChannel
     * method and returning its result.
     */
    bool handleChannel(const Decibel::ChannelInfo & channel_info);

private:
    ChannelHandlerPrivate * d;
};

} // namespace

#endif
