/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redisQObject::tribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is disQObject::tributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin SQObject::treet, Fifth Floor, Boston, MA  02110-1301  USA
 */

extern "C"
{
#include <signal.h>
}

#include <QtCore/QCoreApplication>
#include <QtCore/QDebug>

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMetaType>

#include "accountmanager.h"
#include "componentmanager.h"
#include "connectionfacade.h"
#include "contactmanager.h"
#include "policyengine.h"
#include "protocolmanager.h"

#include <Decibel/DBusNames>

/**
 * @mainpage The Decibel Realtime Communication Framework
 *
 * Welcome to Decibel!
 */

namespace
{
    static void signal_handler(int signal)
    {
        if (signal == SIGTERM || signal == SIGINT)
        {
            QCoreApplication * qapp = QCoreApplication::instance();
            if (0 != qapp) { qapp->quit(); }
            return;
        }
    }
}

int main(int argc, char ** argv)
{
    QCoreApplication app(argc, argv);
    app.setOrganizationName(Decibel::organisation_name);
    app.setApplicationName(Decibel::system_name);

    // install signal handler:
    if (signal(SIGTERM, signal_handler) == SIG_ERR)
    { qWarning() << QObject::tr("Failed to setup handler for SIGTERM!"); }
    if (signal(SIGINT, signal_handler) == SIG_ERR)
    { qWarning() << QObject::tr("Failed to setup handler for SIGINT!"); }

    ProtocolManager protocol_mgr(&app);
    ConnectionFacade connection_facade(&protocol_mgr, &app);
    AccountManager account_mgr(&connection_facade, &app);
    ComponentManager component_mgr(&app);
    PolicyEngine policy_engine(&component_mgr, &app);
    ContactManager contact_mgr(&account_mgr, &app);

    QObject::connect(&connection_facade, SIGNAL(channelOpened(QtTapioca::Connection *,
                                                              QtTapioca::Channel *)),
                     &policy_engine, SLOT(onChannelOpened(QtTapioca::Connection *,
                                                          QtTapioca::Channel *)));
    QObject::connect(&connection_facade, SIGNAL(connectionOpened(QtTapioca::Connection *)),
                     &contact_mgr, SLOT(onConnectionOpened(QtTapioca::Connection *)));
    QObject::connect(&connection_facade, SIGNAL(connectionClosed(QtTapioca::Connection *)),
                     &contact_mgr, SLOT(onConnectionClosed(QtTapioca::Connection *)));

    Decibel::registerTypes();

    // register on D-BUS:
    QDBusConnection::sessionBus().registerService(Decibel::daemon_service);
    qDebug(qPrintable(QObject::tr("Service %1 registered with session bus.").
               arg(Decibel::daemon_service)));
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_protocolmanager_path,
                                                 &protocol_mgr);
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_accountmanager_path,
                                                 &account_mgr);
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_componentmanager_path,
                                                 &component_mgr);
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_contactmanager_path,
                                                 &contact_mgr);
    qDebug(qPrintable(QObject::tr("All objects registered.")));

    account_mgr.bringUpAccounts();

    qDebug(qPrintable(QObject::tr("Enternig main loop.")));

    return app.exec();
}
