/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtTest/QtTest>
#include <QtCore/QDebug>

#include "accountmanager.h"
#include "connectionfacade.h"
#include "protocolmanager.h"
#include "accountmanagertest.h"

#include <Decibel/Errors>
#include <Decibel/AccountData>
#include <Decibel/DBusNames>

#include <QtTapioca/ContactBase>

namespace
{
    const QString application_name_postfix("_TESTSUITE");
}

void AccountManagerTest::init()
{
    QCoreApplication::instance()->setOrganizationName(Decibel::organisation_name);
    QCoreApplication::instance()->setApplicationName(Decibel::system_name +
                                                     application_name_postfix);

    m_protocolMgr = new ProtocolManager();
    m_connectionFacade = new ConnectionFacade(m_protocolMgr);
}

void AccountManagerTest::cleanup()
{
    delete m_protocolMgr; m_protocolMgr = 0;
    delete m_connectionFacade; m_connectionFacade = 0;
}

void AccountManagerTest::createAccountWithError()
{
    AccountManager account_mgr(m_connectionFacade);

    // Empty Account:
    QVariantMap error_data;
    int error = account_mgr.addAccount(error_data);
    QVERIFY(error == Decibel::ERROR_EMPTY_LIST);

    // No protocol:
    error_data.insert("name", QVariant("value"));
    error = account_mgr.addAccount(error_data);
    QVERIFY(error == Decibel::ERROR_NO_PROTOCOL);

    // Wrong protocol type:
    error_data.insert(Decibel::name_protocol, QVariant(1));
    error = account_mgr.addAccount(error_data);
    QVERIFY(error == Decibel::ERROR_WRONG_VALUE_TYPE_FOR_PROTOCOL);

    // Duplicate name:
    error_data.clear();
    error_data.insert("name", QVariant("value"));
    error_data.insert(Decibel::name_protocol, QVariant("testproto"));
    error_data.insertMulti("name", QVariant("some other value"));
    error = account_mgr.addAccount(error_data);
    QVERIFY(error == Decibel::ERROR_DUPLICATE_NAME);
}

void AccountManagerTest::simplePersistenceTest()
{
    QVariantMap account_data;
    account_data.insert(Decibel::name_protocol, "testproto");
    account_data.insert(Decibel::name_presence, 3);
    account_data.insert(Decibel::name_presence_message, "");
    account_data.insert("name", "value");
    account_data.insert("int", 1);
    account_data.insert("float", 2.0);

    int account;
    {
        AccountManager account_mgr(m_connectionFacade);
        account = account_mgr.addAccount(account_data);
        QVERIFY(account >= 0);
    }
    {
        AccountManager account_mgr(m_connectionFacade);
        QVariantMap query_data = account_mgr.queryAccount(account);
        QVERIFY(account_data == query_data);
        account_mgr.deleteAccount(account);
    }
}

void AccountManagerTest::createQueryUpdateQueryDeleteQuery()
{
    AccountManager account_mgr(m_connectionFacade);

    // Create:
    QVariantMap account_data;
    account_data.insert(QString(Decibel::name_protocol), QVariant(QString("testproto")));
    account_data.insert(QString("Test"), QVariant("Data"));
    account_data.insert(QString("remove_me"), QVariant("remove_me"));
    account_data.insert(Decibel::name_presence, QVariant(1));
    account_data.insert(Decibel::name_presence_message, QVariant("no message"));
    int new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account >= 0);

    // Query:
    QVariantMap query_data = account_mgr.queryAccount(new_account);
    QVERIFY(query_data == account_data);

    // Update:
    QVariantMap updated_data;
    updated_data.insert(QString(Decibel::name_protocol), QVariant(QString("testproto")));
    updated_data.insert(QString("Test"), QVariant("new data"));
    updated_data.insert(QString("just_added"), QVariant("just_added"));
    updated_data.insert(Decibel::name_presence, QVariant(2));
    updated_data.insert(Decibel::name_presence_message, QVariant("some message"));
    int updated_account = account_mgr.updateAccount(new_account, updated_data);
    QVERIFY(updated_account == new_account);

    // Query:
    query_data = account_mgr.queryAccount(updated_account);
    QVERIFY(query_data == updated_data);

    // Delete:
    account_mgr.deleteAccount(new_account);

    // Query:
    query_data = account_mgr.queryAccount(new_account);
    QVERIFY(query_data.isEmpty());
}

void AccountManagerTest::listCreateListDeleteList()
{
    AccountManager account_mgr(m_connectionFacade);

    // List:
    QList<uint> original_list = account_mgr.listAccounts();

    // Create:
    QVariantMap account_data;
    account_data.insert(QString(Decibel::name_protocol), QVariant(QString("testproto")));
    account_data.insert(QString("Test"), QVariant("Data"));
    int new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account >= 0);

    // List:
    QList<uint> new_list = account_mgr.listAccounts();
    QVERIFY(original_list.size() == new_list.size() - 1);
    int hdl;
    foreach (hdl, original_list)
    { QVERIFY(new_list.indexOf(hdl) >= 0); }
    QVERIFY(new_list.indexOf(new_account) >= 0);

    // Delete:
    account_mgr.deleteAccount(new_account);

    // List:
    new_list = account_mgr.listAccounts();
    QVERIFY(original_list.size() == new_list.size());
    foreach (hdl, original_list)
    { QVERIFY(new_list.indexOf(hdl) >= 0); }
    QVERIFY(new_list.indexOf(new_account) < 0);
}

void AccountManagerTest::createSearch()
{
    AccountManager account_mgr(m_connectionFacade);

    QVariantMap account_data;
    account_data.insert(Decibel::name_protocol, QVariant(QString("search_me")));
    account_data.insert(QString("search"), QVariant(42));
    account_data.insert(QString("random"), QVariant(23.0));
    uint new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account > 0);
    uint duplicate_entry = account_mgr.addAccount(account_data);
    QVERIFY(new_account == duplicate_entry);

    QVariantMap account_data2;
    account_data2.insert(Decibel::name_protocol, QVariant(QString("not_me")));
    account_data2.insert(QString("search"), QVariant(42));
    account_data2.insert(QString("random"), QVariant("23.0"));
    uint new_account2 = account_mgr.addAccount(account_data2);
    QVERIFY(new_account2 > 0);

    QList<uint> search_results;
    QVariantMap search;
    search.insert(Decibel::name_protocol, QVariant(QString("search_me")));
    search_results = account_mgr.findAccounts(search);
    QVERIFY(search_results.size() == 1);
    QVERIFY(search_results[0] == new_account);

    search.clear();
    search.insert(QString("search"), QVariant(42));
    search_results = account_mgr.findAccounts(search);
    QVERIFY(search_results.size() == 2);
    QVERIFY(search_results.indexOf(new_account) >= 0);
    QVERIFY(search_results.indexOf(new_account2) >= 0);

    search.clear();
    search.insert(QString("random"), QVariant(23.0));
    search_results = account_mgr.findAccounts(search);
    QVERIFY(search_results.size() == 1);
    QVERIFY(search_results[0] == new_account);

    search.clear();
    search_results = account_mgr.findAccounts(search);
    QVERIFY(search_results == account_mgr.listAccounts());

    search.clear();
    search.insert(QString("XXX-undefined-XXX"), QVariant(QString("UNSET")));
    search_results = account_mgr.findAccounts(search);
    QVERIFY(search_results.isEmpty());
}

void AccountManagerTest::presenceSetup()
{
    AccountManager account_mgr(m_connectionFacade);

    // default presence:
    QVariantMap account_data;
    account_data.insert(QString(Decibel::name_protocol), QVariant(QString("search_me")));
    account_data.insert(QString("search"), QVariant(42));
    account_data.insert(QString("random"), QVariant(23.0));
    int new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account >= 0);

    QVariantMap new_data;
    new_data = account_mgr.queryAccount(new_account);
    QVERIFY(new_data.contains(Decibel::name_presence));
    QVERIFY(new_data[Decibel::name_presence] == QtTapioca::ContactBase::Offline);
    QVERIFY(account_mgr.presence(new_account) == QtTapioca::ContactBase::Offline);

    // invalid presence:
    account_data.insert(QString(Decibel::name_protocol), QVariant(QString("search_me")));
    account_data.insert(QString("search"), QVariant(42));
    account_data.insert(QString("random"), QVariant(23.0));
    account_data.insert(Decibel::name_presence, QVariant(0));
    new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account >= 0);

    new_data = account_mgr.queryAccount(new_account);
    QVERIFY(new_data.contains(Decibel::name_presence));
    QVERIFY(new_data[Decibel::name_presence] == QtTapioca::ContactBase::Offline);
    QVERIFY(account_mgr.presence(new_account) == QtTapioca::ContactBase::Offline);

    // valid presence:
    account_data.insert(QString(Decibel::name_protocol), QVariant(QString("search_me")));
    account_data.insert(QString("search"), QVariant(42));
    account_data.insert(QString("random"), QVariant(23.0));
    account_data.insert(Decibel::name_presence, QVariant(4));
    new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account >= 0);

    new_data = account_mgr.queryAccount(new_account);
    QVERIFY(new_data.contains(Decibel::name_presence));
    QVERIFY(new_data[Decibel::name_presence] == 4);
    QVERIFY(account_mgr.presence(new_account) == 4);
}

void AccountManagerTest::goOnline()
{
    // Set gabble as default CM for jabber:
    ProtocolManager protocol_mgr;

    QStringList jabber_cms =
        protocol_mgr.connectionManagersForProtocol("jabber");
    if (jabber_cms.indexOf("gabble") < 0)
    {
        QSKIP("The gabble Connection Manager is required for this test.",
              SkipSingle);
    }

    QVERIFY(protocol_mgr.setDefaultConnectionManagerForProtocol("jabber", "gabble"));

    // Get data for account:
    QString jabber_user(qgetenv("DECIBEL_TS_JABBER_USER"));
    QString jabber_passwd(qgetenv("DECIBEL_TS_JABBER_PASSWD"));

    if (jabber_user.isEmpty() || jabber_passwd.isEmpty())
    {
        QSKIP("You need to set DECIBEL_TS_JABBER_USER and DECIBEL_TS_JABBER_PASSWD!",
              SkipSingle);
    }

    // Register account with AccountManager:
    ConnectionFacade connection_mgr(&protocol_mgr);
    AccountManager account_mgr(&connection_mgr);
    QVariantMap account_data;
    account_data.insert(Decibel::name_protocol, QString("jabber"));
    account_data.insert("account", jabber_user);
    account_data.insert("password", jabber_passwd);
    int new_account = account_mgr.addAccount(account_data);
    QVERIFY(new_account >= 0);
    QVERIFY(account_mgr.presence(new_account) == QtTapioca::ContactBase::Offline);

    QTest::qWait(1000);

    // set account to online:
    qDebug() << "going online...";
    int presence = account_mgr.setPresence(new_account, QtTapioca::ContactBase::Available);
    if (presence < 0)
    {
        QSKIP("The account is already connected! Please go offline before running this test",
              SkipSingle);
    }
    QVERIFY(presence > 0);

    QTest::qWait(5000);

    presence = account_mgr.presence(new_account);
    QVERIFY(presence == QtTapioca::ContactBase::Available);

    qDebug() << "Setting message...";
    QString test_message("test message");
    presence = account_mgr.setPresenceMessage(new_account, test_message);
    QTest::qWait(5000);

    QVERIFY(presence > 0);
    QString returned_test_message = account_mgr.presenceMessage(new_account);
    QVERIFY(test_message == returned_test_message);

    qDebug() << "getting busy...";
    presence = account_mgr.setPresence(new_account, QtTapioca::ContactBase::Busy);
    QVERIFY(presence > 0);

    QTest::qWait(5000);

    presence = account_mgr.presence(new_account);
    QVERIFY(presence == QtTapioca::ContactBase::Busy);

    qDebug() << "Getting available with new message";
    QString new_message("Available again!");
    presence = account_mgr.setPresenceAndMessage(new_account,
                                                 QtTapioca::ContactBase::Available,
                                                 new_message);
    QVERIFY(presence == QtTapioca::ContactBase::Available);
    QTest::qWait(5000);

    presence = account_mgr.presence(new_account);
    QString real_message = account_mgr.presenceMessage(new_account);
    QVERIFY(presence == QtTapioca::ContactBase::Available);
    QVERIFY(real_message == new_message);

    qDebug() << "going offline...";
    presence = account_mgr.setPresence(new_account, QtTapioca::ContactBase::Offline);
    QVERIFY(presence > 0);

    QTest::qWait(5000);

    QVERIFY(QtTapioca::ContactBase::Offline == account_mgr.presence(new_account));
}

// ---------------------- Helper methods ---------------------- //

QTEST_MAIN(AccountManagerTest)
#include "accountmanagertest.moc"
