/*
 * A Decibel Demo
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtCore/QStringList>
#include <QtCore/QDebug>
#include <QtCore/QCoreApplication>
#include <QtCore/QTimer>
#include <QtDBus/QDBusConnection>

#include <QtTapioca/ContactBase>

#include <Decibel/AccountManager>
#include <Decibel/DBusNames>

// Simplistic demo to set the presence information on an account
// (bringing it online/offline when necessary).
// Usage: $0 accounthandle presencelevel [presencemessage]

int main(int argc, char ** argv)
{
    QCoreApplication app(argc, argv);

    Q_ASSERT(argc >= 3 && argc <= 4);

    QString account_handle_string(argv[1]);
    QString presence_level_string(argv[2]);
    QString presence_message;
    if (argc == 4) { presence_message = QString(argv[3]); }

    int account_handle = account_handle_string.toInt();
    if (account_handle <= 0)
    {
        qDebug("Invalid account handle given (not a number?).");
        return 10;
    }
    int presence_level = presence_level_string.toInt();
    if (presence_level < QtTapioca::ContactBase::Offline ||
        presence_level > QtTapioca::ContactBase::Busy)
    {
        qDebug() << "Invalid presence level given (must be a number between"
                 << QtTapioca::ContactBase::Offline << "(Offline) and"
                 << QtTapioca::ContactBase::Busy << "(Busy).";
        qDebug() << "see Presence header file for an enum of valid states.";
        return 11;
    }

    // Connect to session bus:
    if (!QDBusConnection::sessionBus().isConnected())
    {
        qDebug("Cannot connect to the D-BUS session bus.\n"
               "To start it, run:\n"
               "\teval `dbus-launch --auto-syntax`\n");
        return 1;
    }

    // Register custom types:
    Decibel::registerTypes();

    // connect to the Decibel daemon:
    de::basyskom::Decibel::AccountManager
    account_mgr(Decibel::daemon_service, Decibel::daemon_accountmanager_path,
                QDBusConnection::sessionBus(), &app);

    QDBusReply<int> reply;
    if (presence_message.isEmpty())
    { reply = account_mgr.setPresence(account_handle, presence_level); }
    else
    {
        reply = account_mgr.setPresenceAndMessage(account_handle,
                                                  presence_level,
                                                  presence_message);
    }

    if(!reply.isValid())
    {
        QDBusError error = reply.error();
        qDebug() << "DBus Error while setting presence:"
                 << error.type() << ":"
                 << error.message();
        return 3;
    }
    if (reply.value() < 0)
    {
        qDebug() << "Failed to set presence: Returned code was"
                 << reply.value();
        return 12;
    }

    return 0;
}

