/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_DAEMON_CONNECTIONFACADE_H_
#define _DECIBEL_DAEMON_CONNECTIONFACADE_H_

#include <QtCore/QObject>

#include <QtTapioca/Connection>
#include <QtTapioca/ContactBase>

#include <Decibel/Types>

class ProtocolManager;
class ConnectionFacadePrivate;

/**
 * @brief The ConnectionFacade is used to initiate connections.
 *
 * The ConnectionFacade is the object used to initiate connections
 * to contacts and to bring accounts online.
 *
 * Objects of this class are not accessible over DBus!
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ConnectionFacade : public QObject
{
    Q_OBJECT

public:
    /** @brief Constructor. */
    explicit ConnectionFacade(ProtocolManager*, QObject* parent = 0);
    /** @brief Destructor. */
    ~ConnectionFacade();

    /**
     * @brief Connects an account to an service using a given connection manager.
     * @param account_data The account data (as a list of name value pairs).
     * @param cm_name The name of the connection manager to use.
     * @param presence_state The presence state to set.
     * @param message The presence message to set.
     * @return This method returns a connection handle > 0 (or a negative error
     *         code).
     *
     * This method brings the given account online using the given connection
     * manage. The presence is set to the value defined in the account.
     */
    QtTapioca::Connection * connectUsingAccountAndConnectionManager(
                                const QVariantMap& account_data,
                                const QString cm_name,
                                const int presence_state,
                                const QString message);

    /**
     * @brief Connects an accunt to a service using the default connection
     *        manager defined for the protocol of the account.
     * @param account_data The account data (as a list of name value pairs).
     * @param presence_state The presence state to set.
     * @param message The presence message to set.
     * @return This method returns a connection handle > 0 (or a negative error
     *         code).
     *
     * This method brings the given account online. It asks the ProtocolManager
     * for the default connection manager for the protocol of the account and
     * then makes the account become online. The presence is set to the value
     * defined in the account.
     */
    QtTapioca::Connection * connectUsingAccount(const QVariantMap& account_data,
                                                const int presence_state,
                                                const QString message);

    /**
     * @brief Update the presence state on a connection.
     * @param connection The connection to update the presence state on.
     * @param presence_state The presence state to set.
     * @param message The presence message to set.
     * @return Returns the presence state set or a negative error code.
     *
     * Updates the presence state on a connection.
     */
    int updatePresence(const QtTapioca::Connection * connection,
                       const int presence_state,
                       const QString & message);

signals:
    /**
     * @brief Emited whenever a presence info of the user was updated on one of his connections.
     * @param connection A pointer to the connection.
     * @param presence_state The new presence value.
     * @param message The new presence message.
     */
    void ownPresenceUpdated(QtTapioca::Connection * connection,
                            const int presence_state, const QString message);

    /**
     * @brief Emitted whenever a new connection is opened.
     * @param connection A pointer to the new connection.
     */
    void connectionOpened(QtTapioca::Connection * connection);

    /**
     * @brief Emitted whenever a connection is closed.
     * @param connection A pointer to the connection that is to be closed.
     */
    void connectionClosed(QtTapioca::Connection * connection);

    /**
     * @brief Emitted whenever a new channel is opened.
     * @param connection A pointer to the connection the channel belongs to.
     * @param channel A pointer to the channel.
     */
    void channelOpened(QtTapioca::Connection * connection,
                       QtTapioca::Channel * channel);

    /**
     * @brief Emitted whenever a channel is closed.
     * @param connection A pointer to the connection the channel belongs to.
     * @param channel A pointer to the channel.
     */
    void channelClosed(QtTapioca::Connection * connection,
                       QtTapioca::Channel * channel);

    /**
     * @brief Emitted before we are bringing down all connections.
     */
    void shuttingDown();

private slots:
    /** @brief Called by Tapioca whenever the status of a connection changes. */
    void onStatusChanged(QtTapioca::Connection* connection,
                         QtTapioca::Connection::Status new_state,
                         QtTapioca::Connection::Reason reason);

    /** @brief Called by Tapioca whenever a presence was updated. */
    void onPresenceUpdated(QtTapioca::ContactBase *,
                           QtTapioca::ContactBase::Presence,
                           const QString &);

    /** @brief Called by Tapioca whenever a Channel was opened. */
    void onChannelOpened(QtTapioca::Connection *, QtTapioca::Channel *, bool);

    /** @brief Called just before Decibel will quit. */
    void onAboutToQuit();

    /** @brief Called whenever a service name on the D-Bus changes. We use this
     *         to detect crashed CMs. */
    void onServiceOwnerChanged(const QString &, const QString &, const QString &);

private:
    /** @brief Register a new connection and emit the required signals. */
    void addConnection(QtTapioca::Connection * const);
    /** @brief Deregister a connection, emit the required signals and free it. */
    void rmConnection(QtTapioca::Connection * const);

    /** @brief Private data structure. */
    ConnectionFacadePrivate * const d;
};

#endif
