/*
 * This file is licensed under the terms of the GNU General Public License,
 * version 2. See the file COPYING in the main directory for details.
 * 
 * Slooow, but small string operations, so that we don't have to link lib in.
 * Originally from linux/lib/string.c, which is
 *	Copyright (C) 1991, 1992  Linus Torvalds
 *
 * Added I/O functions for libcom_err.
 *
 *  Copyright (C) 2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 */

#include "delo.h"
#include "stringops.h"

void *memset(void *s, int c, size_t count)
{
	char *xs = (char *) s;
	while (count--)
		*xs++ = c;
	return s;
}

void *memcpy(void *dest, const void *src, size_t count)
{
	char *tmp = (char *) dest, *s = (char *) src;
	while (count--)
		*tmp++ = *s++;
	return dest;
}

int memcmp(const void *cs, const void *ct, size_t count)
{
	const unsigned char *su1, *su2;
	signed char res = 0;

	for( su1 = cs, su2 = ct; 0 < count; ++su1, ++su2, count--)
		if ((res = *su1 - *su2) != 0)
			break;
	return res;
}

char *strcpy(char *dest, const char *src)
{
	char *tmp = dest;
	while ((*dest++ = *src++) != '\0');
	return tmp;
}

char *strncpy(char *dest, const char *src, int count)
{
	char *tmp = dest;
	while (count-- && (*dest++ = *src++) != '\0');
	return tmp;
}

char *strcat(char *dest, const char *src)
{
	char *tmp = dest;
	while (*dest) dest++;
	while ((*dest++ = *src++) != '\0');
	return tmp;
}

char *strncat(char *dest, const char *src, int n)
{
	char *tmp = dest;
	while (*dest) dest++;
	while (n && (*dest++ = *src++) != '\0') n--;
	if (!n) *dest = 0;
	return tmp;
}

int strcmp(const char *cs, const char *ct)
{
	int res;
	while (1)
		if ((res = *cs - *ct++) != 0 || !*cs++)
			break;
	return res;
}

int strncmp(const char *cs, const char *ct, int count)
{
	int res = 0;
	while (count) {
		if ((res = *cs - *ct++) != 0 || !*cs++)
			break;
		count--;
	}
	return res;
}

char *strchr(const char *s, int c)
{
	for(; *s != (char) c; ++s)
		if (*s == '\0')
			return 0;
	return (char *) s;
}

char *strrchr(const char *s, int c)
{
	const char *p = s + strlen(s);
	do {
		if (*p == (char)c)
			return (char *)p;
	} while (--p >= s);
	return 0;
}

unsigned int strlen(const char *s)
{
	const char *sc;
	for (sc = s; *sc != '\0'; ++sc);
	return sc - s;
}

char *strdup(const char *str)
{
	char *ret;
	ret = malloc(strlen(str) + 1);
	strcpy(ret, str);
	return ret;
}

int tolower(int c)
{
	if (c >= 'A' && c <= 'Z') return c - 'A' + 'a';
	return c;
}

int strcasecmp(const char *cs, const char *ct)
{
	register signed char __res;
	while (1)
		if ((__res = tolower(*cs) - tolower(*ct++)) != 0 || !*cs++)
			break;
	return __res;
}

int strncasecmp(const char *cs, const char *ct, size_t n)
{
	register signed char __res = 0;
	while (n--)
		if ((__res = tolower(*cs) - tolower(*ct++)) != 0 || !*cs++)
			break;
	return __res;
}

char *strstr(const char *s1, const char *s2)
{
	int l1, l2;

	l2 = strlen(s2);
	if (!l2)
		return (char *) s1;
	l1 = strlen(s1);
	while (l1 >= l2) {
		l1--;
		if (!memcmp(s1,s2,l2))
			return (char *) s1;
		s1++;
	}
	return 0;
}

int isdigit(char c)
{
	return (c >= '0' && c <= '9') ? 1 : 0;
}

typedef unsigned long time_t;

/* Not a string op, and we don't support it. */
time_t time(time_t *result)
{
	if (result)
		*result = -1;
	return -1;
}

/* libcom_err wants to have some stdio functions. */
typedef int FILE;
FILE *stderr;

int fputc(int c, UNUSED FILE *stream)
{
	printf("%c", c);
	return c;
}

int fputs(const char *s, UNUSED FILE *stream)
{
	printf("%s", s);
	return 1;
}

size_t fwrite(const void *ptr, size_t size, size_t nmemb, FILE *stream)
{
	size_t i;

	for (i = 0; i < (size * nmemb); i++) {
		char c = *((char *)ptr)++;

		fputc(c, stream);
	}
	return nmemb;
}

typedef int va_list;

int vfprintf(UNUSED FILE *stream, UNUSED const char *format,
	     UNUSED va_list ap)
{
	/* Link dummy, unused as long as com_err() gets provided a NULL
	   pointer as format arg. */
	return 0;
}

int fflush(UNUSED FILE *stream)
{
	return 0;
}

/* We don't support this */
char *strerror(UNUSED int errno)
{
	return NULL;
}
