/*
 * This file is licensed under the terms of the GNU General Public License,
 * version 2. See the file COPYING in the main directory for details.
 * 
 *  Copyright (C) 2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 */

#include <elf.h>

#include "dec_prom.h"
#include "../t-rex/t-rex.h"

#ifndef NULL
#define NULL ((void *)0)
#endif

#define PROGRAM   "t-rex-loader"
#define VERSION   "0.1"
#define COPYRIGHT "Copyright 2003"
#define AUTHOR    "Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>"

extern struct loader_data _start_data;
extern char _loader_address;

/* The PROM callback vector. */
const struct callback *callv = NULL;

/* The DECstation system id. */
int dec_sysid = 0;

#define printf callv->_printf

#ifdef DEBUG
#define dprintf(fmt, arg...) do { printf(fmt , ##arg); } while (0)
#else
#define dprintf(fmt, arg...) do { } while (0)
#endif

static inline int puts(char *s)
{
	return callv->_puts(s);
}

static inline int getsysid(void)
{
	return callv->_getsysid();
}

static inline int clearcache(void)
{
	return callv->_clearcache();
}

static inline void halt(int a, int b)
{
	callv->_halt(a, b);
}

static void *memset(void *s, int c, int count)
{
	char *xs = (char *) s;

	while (count--)
		*xs++ = c;

	return s;
}

static void *memcpy(void *dest, const void *src, int count)
{
	char *tmp = (char *) dest;
	char *s = (char *) src;

	while (count--)
		*tmp++ = *s++;

	return dest;
}

static void *loadelf32(struct loader_data *ld, Elf32_Ehdr *ehdr)
{
	int i;

	if (!(ehdr->e_type == ET_EXEC
	      && ehdr->e_machine == EM_MIPS
	      && ehdr->e_version == EV_CURRENT))
		return NULL;

	for(i = 0; i < ehdr->e_phnum; i++) {
		Elf32_Phdr *phdr;

		phdr = (Elf32_Phdr *)(&_loader_address + ld->kernel_off
				      + ehdr->e_phoff + i * sizeof(*phdr));

		if (phdr->p_type == PT_LOAD) {
			memcpy((void *)phdr->p_vaddr,
			       (void *)(&_loader_address + ld->kernel_off
					+ phdr->p_offset),
				phdr->p_filesz);
		}
	}

	return (void *)ehdr->e_entry;
}

static void *loadelf64(struct loader_data *ld, Elf64_Ehdr *ehdr)
{
	int i;

	if (!(ehdr->e_type == ET_EXEC
	      && ehdr->e_machine == EM_MIPS
	      && ehdr->e_version == EV_CURRENT))
		return NULL;

	for(i = 0; i < ehdr->e_phnum; i++) {
		Elf64_Phdr *phdr;

		phdr = (Elf64_Phdr *)(unsigned int)
			(&_loader_address + ld->kernel_off
			 + ehdr->e_phoff + i * sizeof(*phdr));

		if (phdr->p_type == PT_LOAD) {
			memcpy((void *)(unsigned int)phdr->p_vaddr,
			       (void *)(unsigned int)(&_loader_address
						      + ld->kernel_off
						      + phdr->p_offset),
				phdr->p_filesz);
		}
	}

	return (void *)(unsigned int)ehdr->e_entry;
}

static void *loadelf(struct loader_data *ld)
{
	union {
		Elf32_Ehdr s;
		Elf64_Ehdr b;
	} ehdr;

	memcpy(&ehdr, (void *)(&_loader_address + ld->kernel_off),
	       sizeof(ehdr));

	if (!(ehdr.s.e_ident[EI_MAG0] == ELFMAG0
	      && ehdr.s.e_ident[EI_MAG1] == ELFMAG1
	      && ehdr.s.e_ident[EI_MAG2] == ELFMAG2
	      && ehdr.s.e_ident[EI_MAG3] == ELFMAG3
	      && (ehdr.s.e_ident[EI_CLASS] == ELFCLASS32
		  || ehdr.s.e_ident[EI_CLASS] == ELFCLASS64)
	      && ehdr.s.e_ident[EI_DATA] == ELFDATA2LSB
	      && ehdr.s.e_ident[EI_VERSION] == EV_CURRENT))
		return NULL;

	if (ehdr.s.e_ident[EI_CLASS] == ELFCLASS32)
		return loadelf32(ld, &ehdr.s);
	else if (ehdr.s.e_ident[EI_CLASS] == ELFCLASS64)
		return loadelf64(ld, &ehdr.b);
	else
		return NULL;
}

static void uint_to_string(char *str, unsigned int val)
{
	char buf[11];
	const char num[] = "0123456789";
	int i;

	memset(buf, 0, 11);
	for (i = 0; i < 10 && val; i++) {
		buf[9 - i] = num[val % 10];
		val /= 10;
	}
	memcpy(str, buf + 10 - i, i + 1);
}

int t_rex_main(int argc, char **argv, int magic, struct callback *cv)
{
	int rex_prom = 0;
	int cargc;
	char **cargv;
	void (*entry)(int argc, char **argv, int magic, void *cv);
	int i;
	char rd_start[25];
	char rd_size[25];

	if (magic == DEC_REX_MAGIC) {
		/* Store Call Vector. */
		callv = cv;
		rex_prom = 1;
		clearcache();
	} else {
		/*
		 * We can't even issue a message here because we know
		 * nothing about the cv.
		 */
		EMERGENCY_RESET();
	}

	printf("%s V%s %s  %s\n", PROGRAM, VERSION, COPYRIGHT, AUTHOR);

	dprintf("callv addr %p\n", callv);
	for(i = 0; i < argc; i++)
		dprintf("clo: %d %s\n", i, argv[i]);

	if (rex_prom) {
		dec_sysid = getsysid();
		switch (DEC_FIRMREV(dec_sysid)) {
		case FIRM_TFC0:
			dprintf("REX Firmware revision TFC0\n");
			break;
		case FIRM_TFC1:
			dprintf("REX Firmware revision TFC1\n");
			break;
		default:
			printf("REX Firmware revision unknown (%d)\n",
			       DEC_FIRMREV(dec_sysid));
			break;
		}
	}

	/* Append ramdisk parameters "rd_start=" and "rd_size=". */
	memcpy(rd_start, "rd_start=", sizeof("rd_start="));
	uint_to_string(rd_start + sizeof("rd_start=") - 1,
		       (uint32_t)&_loader_address + _start_data.ramdisk_off);
	argv[argc++] = rd_start;
	memcpy(rd_size, "rd_size=", sizeof("rd_size="));
	uint_to_string(rd_size + sizeof("rd_size=") - 1,
		       _start_data.ramdisk_len);
	argv[argc] = rd_size;
	argv[argc + 1] = NULL;

	/*
	 * Boot command line will look like this:
	 *
	 * boot 3/tftp root=/dev/sda5 console=ttyS2
	 */

	/* Do we have parms on the prom command line? */
	cargv = argv;
	if (argc <= 3) {
		/*
		 * FIXME: This is completely stupid to work around
		 * a kernel stupidity - Probably the kernel should not
		 * skip anything and/or check for existance of "boot"
		 * at the beginning and skip only if this is the case.
		 */
		if (rex_prom) 
			/*
			 * Copy pointer - We take 0 as the kernel
			 * will skip 2 parms anyway for REX so it doesn't
			 * care about our store usage.
			 */
			cargv = argv;
		else
			/*
			 * For non REX proms the kernel skips 1 arg.
 			 */
			cargv = &argv[1];
	}

	/* Count cargv values. */
	for(cargc = 0; cargv[cargc]; cargc++)
		dprintf("configargv: %d %s\n", cargc, cargv[cargc]);

	/* Begin of progress indicator. */
	printf("Loading ...");

	if (!(entry = loadelf(&_start_data))) {
		printf("\nThis is not a little endian MIPS kernel\n");
		return 1;
	}

	/* End of progress indicator. */
	printf(" ok\n");

	entry(cargc, cargv, magic, cv);

	/* The kernel start failed */
	return 1;
}
