import unittest
from tempfile import mkstemp
from time import sleep

from feed import *
from database import *

from test.framework import DemocracyTestCase

class FeedURLValidationTest(DemocracyTestCase):
    def test(self):
        self.assertEqual(validateFeedURL("http://foo.bar.com"), True)
        self.assertEqual(validateFeedURL("https://foo.bar.com"), True)
        self.assertEqual(validateFeedURL("feed://foo.bar.com"), True)
                         
        self.assertEqual(validateFeedURL("http:foo.bar.com"), False)
        self.assertEqual(validateFeedURL("https:foo.bar.com"), False)
        self.assertEqual(validateFeedURL("feed:foo.bar.com"), False)
        self.assertEqual(validateFeedURL("http:/foo.bar.com"), False)
        self.assertEqual(validateFeedURL("https:/foo.bar.com"), False)
        self.assertEqual(validateFeedURL("feed:/foo.bar.com"), False)
        self.assertEqual(validateFeedURL("http:///foo.bar.com"), False)
        self.assertEqual(validateFeedURL("https:///foo.bar.com"), False)
        self.assertEqual(validateFeedURL("feed:///foo.bar.com"), False)

        self.assertEqual(validateFeedURL("foo.bar.com"), False)
        self.assertEqual(validateFeedURL("crap:foo.bar.com"), False)
        self.assertEqual(validateFeedURL("crap:/foo.bar.com"), False)
        self.assertEqual(validateFeedURL("crap://foo.bar.com"), False)
        self.assertEqual(validateFeedURL("crap:///foo.bar.com"), False)

class FeedURLNormalizationTest(DemocracyTestCase):
    def test(self):
        self.assertEqual(normalizeFeedURL("http://foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("https://foo.bar.com"), "https://foo.bar.com")
        self.assertEqual(normalizeFeedURL("feed://foo.bar.com"), "feed://foo.bar.com")

        self.assertEqual(normalizeFeedURL("http:foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("https:foo.bar.com"), "https://foo.bar.com")
        self.assertEqual(normalizeFeedURL("feed:foo.bar.com"), "feed://foo.bar.com")
        self.assertEqual(normalizeFeedURL("http:/foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("https:/foo.bar.com"), "https://foo.bar.com")
        self.assertEqual(normalizeFeedURL("feed:/foo.bar.com"), "feed://foo.bar.com")
        self.assertEqual(normalizeFeedURL("http:///foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("https:///foo.bar.com"), "https://foo.bar.com")
        self.assertEqual(normalizeFeedURL("feed:///foo.bar.com"), "feed://foo.bar.com")

        self.assertEqual(normalizeFeedURL("foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("crap:foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("crap:/foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("crap://foo.bar.com"), "http://foo.bar.com")
        self.assertEqual(normalizeFeedURL("crap:///foo.bar.com"), "http://foo.bar.com")

class SimpleFeedTestCase(DemocracyTestCase):
    def setUp(self):
        DDBObject.dd = DynamicDatabase()
        self.everything = DDBObject.dd
        [handle, self.filename] = mkstemp(".xml")
        handle =file(self.filename,"wb")
#RSS 2.0 example feed
#http://cyber.law.harvard.edu/blogs/gems/tech/rss2sample.xml
        handle.write("""<?xml version="1.0"?>
<rss version="2.0">
   <channel>
      <title>Liftoff News</title>
      <link>http://liftoff.msfc.nasa.gov/</link>
      <description>Liftoff to Space Exploration.</description>
      <language>en-us</language>
      <pubDate>Tue, 10 Jun 2003 04:00:00 GMT</pubDate>

      <lastBuildDate>Tue, 10 Jun 2003 09:41:01 GMT</lastBuildDate>
      <docs>http://blogs.law.harvard.edu/tech/rss</docs>
      <generator>Weblog Editor 2.0</generator>
      <managingEditor>editor@example.com</managingEditor>
      <webMaster>webmaster@example.com</webMaster>
      <item>

         <title>Star City</title>
         <link>http://liftoff.msfc.nasa.gov/news/2003/news-starcity.asp</link>
         <description>How do Americans get ready to work with Russians aboard the International Space Station? They take a crash course in culture, language and protocol at Russia's &lt;a href="http://howe.iki.rssi.ru/GCTC/gctc_e.htm"&gt;Star City&lt;/a&gt;.</description>
         <pubDate>Tue, 03 Jun 2003 09:39:21 GMT</pubDate>
         <guid>http://liftoff.msfc.nasa.gov/2003/06/03.html#item573</guid>

      </item>
      <item>
         <description>Sky watchers in Europe, Asia, and parts of Alaska and Canada will experience a &lt;a href="http://science.nasa.gov/headlines/y2003/30may_solareclipse.htm"&gt;partial eclipse of the Sun&lt;/a&gt; on Saturday, May 31st.</description>
         <pubDate>Fri, 30 May 2003 11:06:42 GMT</pubDate>
         <guid>http://liftoff.msfc.nasa.gov/2003/05/30.html#item572</guid>

      </item>
      <item>
         <title>The Engine That Does More</title>
         <link>http://liftoff.msfc.nasa.gov/news/2003/news-VASIMR.asp</link>
         <description>Before man travels to Mars, NASA hopes to design new engines that will let us fly through the Solar System more quickly.  The proposed VASIMR engine would do that.</description>
         <pubDate>Tue, 27 May 2003 08:37:32 GMT</pubDate>
         <guid>http://liftoff.msfc.nasa.gov/2003/05/27.html#item571</guid>

      </item>
      <item>
         <title>Astronauts' Dirty Laundry</title>
         <link>http://liftoff.msfc.nasa.gov/news/2003/news-laundry.asp</link>
         <description>Compared to earlier spacecraft, the International Space Station has many luxuries, but laundry facilities are not one of them.  Instead, astronauts have other options.</description>
         <pubDate>Tue, 20 May 2003 08:56:02 GMT</pubDate>
         <guid>http://liftoff.msfc.nasa.gov/2003/05/20.html#item570</guid>

      </item>
   </channel>
</rss>""")
        handle.close()
    def testRun(self):
        myFeed = RSSFeed(self.filename)
        self.assertEqual(self.everything.len(),5)
        items = self.everything.filter(lambda x:x.__class__.__name__ == 'Item')
        self.assertEqual(items.len(),4)
        #Make sure that re-updating doesn't re-create the items
        myFeed.update()
        self.assertEqual(self.everything.len(),5)
        items = self.everything.filter(lambda x:x.__class__.__name__ == 'Item')
        self.assertEqual(items.len(),4)
        myFeed.remove()

class EnclosureFeedTestCase(DemocracyTestCase):
    def setUp(self):
        DDBObject.dd = DynamicDatabase()
        self.everything = DDBObject.dd
        [handle, self.filename] = mkstemp(".xml")
        handle =file(self.filename,"wb")
        handle.write("""<?xml version="1.0"?>
<rss version="2.0">
   <channel>
      <title>Downhill Battle Pics</title>
      <link>http://downhillbattle.org/</link>
      <description>Downhill Battle is a non-profit organization working to support participatory culture and build a fairer music industry.</description>
      <pubDate>Wed, 16 Mar 2005 12:03:42 EST</pubDate>
      <item>
         <title>Bumper Sticker</title>
         <enclosure url="http://downhillbattle.org/key/gallery/chriscA.jpg" />
         <description>I'm a musician and I support filesharing.</description>

      </item>
      <item>
         <title>T-shirt</title>
         <enclosure url="http://downhillbattle.org/key/gallery/payola_tshirt.jpg" />
      </item>
      <item>
         <enclosure url="http://downhillbattle.org/key/gallery/chriscE.jpg" />
         <description>Flyer in Yucaipa, CA</description>
      </item>
      <item>
         <enclosure url="http://downhillbattle.org/key/gallery/jalabel_nov28.jpg" />
      </item>
      
   </channel>
</rss>""")
        handle.close()
    def testRun(self):
        myFeed = RSSFeed(self.filename)
        self.assertEqual(self.everything.len(),5)
        items = self.everything.filter(lambda x:x.__class__.__name__ == 'Item')
        self.assertEqual(items.len(),4)
        #Make sure that re-updating doesn't re-create the items
        myFeed.update()
        self.assertEqual(self.everything.len(),5)
        items = self.everything.filter(lambda x:x.__class__.__name__ == 'Item')
        self.assertEqual(items.len(),4)
        for item in items:
            #Generate an exception if we didn't get one of the enclosures
            item["enclosures"][0]
            self.assertRaises(IndexError,lambda :item["enclosures"][1])
        items[0].download()
        dlers = items[0].getDownloaders()
        self.assertEqual(len(dlers),1)
        self.assertEqual(dlers[0].getURL(),items[0]["enclosures"][0]["url"])
        while dlers[0].getState() == "downloading":
            sleep(1)
        self.assertEqual(dlers[0].getState(),"finished")
        dlers[0].stop()
        myFeed.remove()
            
if __name__ == "__main__":
    unittest.main()
