# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>

"""Test cases for field mapping"""

import copy
import doctest
import os

from unittest import TestCase

import desktopcouch
from desktopcouch.records.field_registry import (
    SimpleFieldMapping, MergeableListFieldMapping, Transformer,
    ListToMergeableListFieldMapping)
from desktopcouch.records import Record

TEST_RECORD = {
    'record_type': 'http://example.com/test',
    'simple_field': 23,
    'test_fields': {
        '_order': [
            'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3',
            'e47455fb-da05-481e-a2c7-88f14d5cc163'],
        'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': {
            'the field': 'another value'},
        'e47455fb-da05-481e-a2c7-88f14d5cc163':
            {'the_field': 'the value',
             'not_the_field': 'different value'}},
    'application_annotations': {'Test App': {
        'private_application_annotations': {
            'test_field': 'e47455fb-da05-481e-a2c7-88f14d5cc163'}}}}

APP_RECORD = {
    'record_type': 'http://example.com/test',
    'simpleField': 23,
    'strawberryField': 'the value'}

TEST_RECORD2 = {
    'record_type': 'http://example.com/test',
    'record_list_field': {
        '_order': [
            'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3',
            'e47455fb-da05-481e-a2c7-88f14d5cc163'],
        'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 'foo',
        'e47455fb-da05-481e-a2c7-88f14d5cc163': 'bar'}}

TEST_RECORD3 = {
    'record_type': 'http://example.com/test',
    'record_list_field': {
        '_order': [
            'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3',
            'e47455fb-da05-481e-a2c7-88f14d5cc163'],
        'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': {
            'description': 'kwark',
            'value': 'foo'},
        'e47455fb-da05-481e-a2c7-88f14d5cc163': {
            'description': 'fnord',
            'value': 'bar'}}}

FIELD_REGISTRY = {
    'simpleField': SimpleFieldMapping('simple_field'),
    'strawberryField': MergeableListFieldMapping(
        'Test App', 'test_field', 'test_fields', 'the_field')}


class TestRecord(Record):
    """A test record type."""

    def __init__(self, data=None):
        super(TestRecord, self).__init__(
            data=data, record_type='http://example.com/test')


class AppTransformer(Transformer):
    """A test transformer class."""

    def __init__(self):
        super(AppTransformer, self).__init__(
            'Test App', FIELD_REGISTRY, record_class=TestRecord)


class TestFieldMapping(TestCase):
    """Test Case for FieldMapping objects."""

    def setUp(self):
        super(TestFieldMapping, self).setUp()
        self.test_record = copy.deepcopy(TEST_RECORD)

    def test_simple_field_mapping(self):
        """Test the simple field mapping object."""
        record = Record(self.test_record)
        mapping = SimpleFieldMapping('simple_field')
        self.assertEqual(23, mapping.get_value(record))
        mapping.set_value(record, 'Fnord')
        self.assertEqual('Fnord', mapping.get_value(record))

    def test_mergeable_list_field_mapping0(self):
        """Test the MergeableListFieldMapping object."""
        record = Record(self.test_record)
        mapping = MergeableListFieldMapping(
            'Test App', 'test_field', 'test_fields', 'the_field')
        self.assertEqual('the value', mapping.get_value(record))
        mapping.set_value(record, 'a new value')
        self.assertEqual('a new value', mapping.get_value(record))
        mapping.delete_value(record)
        self.assertEqual(None, mapping.get_value(record))

    def test_mergeable_list_field_mapping1(self):
        """Test the MergeableListFieldMapping object."""
        # pylint: disable=W0212
        record = Record(self.test_record)
        mapping = MergeableListFieldMapping(
            'Test App', 'test_field', 'test_fields', 'the_field')
        self.assertEqual('the value', mapping.get_value(record))
        del record._data['test_fields'][  # pylint: disable=W0212
            'e47455fb-da05-481e-a2c7-88f14d5cc163']
        mapping.delete_value(record)
        self.assertEqual(None, mapping.get_value(record))
        # pylint: enable=W0212

    def test_mergeable_list_field_mapping_empty_field(self):
        """Test setting empty values in the MergeableListFieldMapping
        object."""
        record = Record(self.test_record)
        mapping = MergeableListFieldMapping(
            'Test App', 'test_field', 'test_fields', 'the_field')
        mapping.set_value(record, '')
        self.assertEqual(None, mapping.get_value(record))

    def test_list_to_mergeable_list_field_mapping1(self):
        """Test the ListToMergeableListFieldMapping object."""
        record = Record(data={}, record_type='http://example.org/foo')
        mapping = ListToMergeableListFieldMapping('record_list_field')
        mapping.set_value(record, ['foo', 'bar', 'baz', 'qux'])
        self.assertEqual('foo', record['record_list_field'][0])
        self.assertEqual('bar', record['record_list_field'][1])
        self.assertEqual('baz', record['record_list_field'][2])
        self.assertEqual('qux', record['record_list_field'][3])

    # pylint: disable=W0212
    def test_list_to_mergeable_list_field_mapping2(self):
        """Test the ListToMergeableListFieldMapping object."""
        record = Record(copy.deepcopy(TEST_RECORD2))
        mapping = ListToMergeableListFieldMapping('record_list_field')
        mapping.set_value(record, ['foo1', 'bar', 'baz', 'qux'])
        self.assertEqual('foo1', record['record_list_field'][0])
        self.assertEqual('bar', record['record_list_field'][1])
        self.assertEqual('baz', record['record_list_field'][2])
        self.assertEqual('qux', record['record_list_field'][3])
        self.assertEqual(
            ['foo1', 'bar', 'baz', 'qux'], mapping.get_value(record))
        self.assertEqual(
            'foo1',
            record._data['record_list_field'][
                'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3'])
        self.assertEqual(
            'bar',
            record._data['record_list_field'][
                'e47455fb-da05-481e-a2c7-88f14d5cc163'])
    # pylint: enable=W0212

    def test_list_to_mergeable_list_with_subfield1(self):
        """Test the ListToMergeableListFieldMapping object."""
        record = Record(data={}, record_type='http://example.org/foo')
        mapping = ListToMergeableListFieldMapping(
            'record_list_field', subfield='value')
        mapping.set_value(record, ['foo', 'bar', 'baz', 'qux'])
        self.assertEqual('foo', record['record_list_field'][0]['value'])
        self.assertEqual('bar', record['record_list_field'][1]['value'])
        self.assertEqual('baz', record['record_list_field'][2]['value'])
        self.assertEqual('qux', record['record_list_field'][3]['value'])

    # pylint: disable=W0212
    def test_list_to_mergeable_list_with_subfield2(self):
        """Test the ListToMergeableListFieldMapping object."""
        record = Record(copy.deepcopy(TEST_RECORD3))
        mapping = ListToMergeableListFieldMapping(
            'record_list_field', subfield='value')
        mapping.set_value(record, ['foo1', 'bar', 'baz', 'qux'])
        self.assertEqual(
            {'value': 'foo1', 'description': 'kwark'},
            record['record_list_field'][0]._data)
        self.assertEqual(
            {'value': 'bar', 'description': 'fnord'},
            record['record_list_field'][1]._data)
        self.assertEqual(
            {'value': 'baz'}, record['record_list_field'][2]._data)
        self.assertEqual(
            {'value': 'qux'}, record['record_list_field'][3]._data)
    # pylint: enable=W0212


class TestTransformer(TestCase):
    """Test application specific transformer classes"""

    def setUp(self):
        """setup test fixtures"""
        super(TestTransformer, self).setUp()
        self.transformer = AppTransformer()

    def test_from_app(self):
        """Test transformation from app to desktopcouch."""
        record = self.transformer.from_app(APP_RECORD)
        underlying = record._data  # pylint: disable=W0212
        self.assertEqual(23, record['simple_field'])
        the_uuid = record.application_annotations['Test App'][
            'private_application_annotations']['test_field']
        self.assertEqual(
            {'the_field': 'the value'},
            underlying['test_fields'][the_uuid])

    def test_from_app_with_record(self):
        """Test transformation from app to desktopcouch when passing in
        an existing record.

        """
        record = Record(record_type="http://example.com/test")
        self.transformer.from_app(APP_RECORD, record)
        underlying = record._data  # pylint: disable=W0212
        self.assertEqual(23, record['simple_field'])
        the_uuid = record.application_annotations['Test App'][
            'private_application_annotations']['test_field']
        self.assertEqual(
            {'the_field': 'the value'},
            underlying['test_fields'][the_uuid])
        # pylint: enable=W0212

    def test_to_app(self):
        """Test transformation to app from desktopcouch."""
        record = Record(TEST_RECORD)
        data = self.transformer.to_app(record)
        self.assertEqual(
            {'simpleField': 23, 'strawberryField': 'the value'}, data)

    def test_to_app_with_dictionary(self):
        """Test transformation to app from desktopcouch when passing
        in an existing dictionary.

        """
        record = Record(TEST_RECORD)
        data = {}
        self.transformer.to_app(record, data)
        self.assertEqual(
            {'simpleField': 23, 'strawberryField': 'the value'}, data)

    def test_run_doctests(self):
        """We run doctests from here to give some context"""
        field_registry_tests_path = os.path.dirname(
            desktopcouch.__file__) + '/records/doc/field_registry.txt'
        try:
            results = doctest.testfile(field_registry_tests_path,
                                       module_relative=False)
        except IOError:
            field_registry_tests_path = \
                '../desktopcouch/records/doc/field_registry.txt'
            results = doctest.testfile(field_registry_tests_path,
                                       module_relative=False)
        self.assertEqual(0, results.failed)
