require 'test/unit'
require 'dhelp'
require 'fileutils'

class TC_DhelpDB < Test::Unit::TestCase
   TMP_DIR = 'test/tmp'

   def setup
      FileUtils.rm_r TMP_DIR if File.exists? TMP_DIR
      FileUtils.mkdir_p TMP_DIR
      @db = Dhelp::Database.open(BDB::CREATE|BDB::TRUNCATE, {}, 0644,
                                 File.join(TMP_DIR, 'dhelpbase'))
   end

   def test_insert
      @db.write(Dhelp::ItemData.new(:file    => 'foofile',
                                    :dir     => 'foodir',
                                    :name    => 'fooname',
                                    :descrip => 'Description for foo'))
      assert_equal(1, @db.keys.size,         "Number of keys")
   end

   def test_remove
      fooItem = Dhelp::ItemData.new(:file    => 'foofile',
                                    :dir     => 'foodir',
                                    :name    => 'fooname',
                                    :descrip => 'Description for foo')
      almostFooItem = Dhelp::ItemData.new(:file    => 'anotherfile',
                                          :dir     => 'foodir',
                                          :name    => 'fooname',
                                          :descrip => 'Description for foo')
      barItem = Dhelp::ItemData.new(:file    => 'barfile',
                                    :dir     => 'bardir',
                                    :name    => 'barname',
                                    :descrip => 'Description for bar')
      fooItemWithAltDesc = Dhelp::ItemData.new(:file    => 'foofile',
                                    :dir     => 'foodir',
                                    :name    => 'fooname',
                                    :descrip => 'Alternative description')
      @db.write(fooItem)
      @db.write(barItem)
      assert_equal(2, @db.keys.size,         "Number of keys")
      ret = @db.del(almostFooItem)
      assert_equal(nil, ret,                 "Deleting nothing")
      assert_equal(2, @db.keys.size,         "Number of keys after deleting nothing")
      ret = @db.del(fooItem)
      assert_not_nil(ret,                    "Really deleting one key")
      assert_equal(1, @db.keys.size,         "Number of keys after really deleting")
      # Add it again, check it's deleted with the same key, but different
      # description
      @db.write(fooItem)
      assert_equal(2, @db.keys.size,         "Number of keys after really deleting")
      ret = @db.del(fooItemWithAltDesc)
      assert_not_nil(ret,                    "Deleting same key, different data")
      assert_equal(1, @db.keys.size,         "Number of keys after really deleting")
   end

   def test_categories
      testCategories = %w(cat1 cat2 supercat supercow powers)
      numberEntries = 0
      testCategories.each_with_index do |cat, i|
         # A couple of items for each category
         0.upto(i) do |j|
            numberEntries += 1
            @db.write(Dhelp::ItemData.new(:file => "#{cat}#{j}",
                                          :dir  => cat,
                                          :name => "File #{j} for cat #{cat}",
                                          :descrip => "Long description for #{cat}/#{j}"))
         end
      end
      assert_equal(numberEntries, @db.size,  "Files in categories inserted")

      # Check that every category has the correct files
      @db.each_category do |cat, itemList|
         # If the category is in index n, then it has n+1 items
         size = testCategories.index(cat) + 1
         assert_equal(size, itemList.size,   "Number of categories in #{cat}")
      end
   end

   def teardown
      @db.close
      FileUtils.rm_rf TMP_DIR
   end
end
