/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/stream_test_data_adaptor.hpp
 *
 * @brief [LEVEL: beta] @ref
 * diagnostics::unittest::Stream_Test_Data_Adaptor class
 *
 * $Id: stream_test_data_adaptor.hpp,v 1.6 2005/06/23 09:54:26 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/stream_test_data_adaptor.t.cpp
 */
#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__STREAM_TEST_DATA_ADAPTOR_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__STREAM_TEST_DATA_ADAPTOR_HPP__INCLUDE_GUARD

// used as base class
#include <diagnostics/unittest/test_system/interactive_test_data_adaptor.hpp>

// used by value in the interface
#include <iostream>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Stream_Test_Data_Adaptor diagnostics/unittest/test_system/stream_test_data_adaptor.hpp
 *
 * @brief Adapts a @ref Test_Data_Source to a @ref Test_Data while
 * using stream-based user-interaction to optionally update the
 * testdata database in case of missing entries or mismachtes.  The
 * user-interaction can be also switched off in order to silently
 * accept or reject any missing or mismachting entiries.
 *
 * @nosubgrouping
 */
class Stream_Test_Data_Adaptor :
    public Interactive_Test_Data_Adaptor
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Stream_Test_Data_Adaptor Self;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    Stream_Test_Data_Adaptor(Self const & other);
    Self & operator=(Self const & other);

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief @a input is used for user input, @a output is used for
     * reporting to the user.
     *
     * @post @ref mode() == @ref MODE_INTERACTIVE
     *
     * @throw never
     */
    Stream_Test_Data_Adaptor(::std::istream & input,
			     ::std::ostream & output);

    /**
     * @brief noop
     *
     * @throw never
     */
    virtual ~Stream_Test_Data_Adaptor();
    // @}



    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Interactive/Noninteractive mode
     * @{
     */
public:
    /**
     * @brief The interation mode of Stream_Test_Data_Adaptor.
     */ 
    typedef enum {
	/**
	 * @brief prompts the user in case of mismatching or missing entries
	 */ 
	MODE_INTERACTIVE=0,
	/**
	 * @brief rejects all mismatching or mismatching entries silently
	 */ 
	MODE_REJECT=1,
	/**
	 * @brief accepts all mismatching or mismatching entries silently
	 */ 
	MODE_ACCEPT=2
    } Mode_t;
	
    /**
     * @brief returns the current interaction mode
     *
     * throw never
     */ 
    Mode_t mode() const;

    /**
     * @brief set the interaction mode
     *
     * post mode()==@a val
     *
     * throw never
     */
    void mode(Mode_t const val);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Interface of Interactive_Test_Data_Adaptor
     * @{
     */
protected:
    /**
     * @brief If @ref mode()== @ref MODE_INTERACTIVE, the user is
     * asked whether to accept @a id / @a data as new entry (@a data
     * can be modified). Otherwise, it returns true (false) if 
     * @ref mode()==@ref MODE_ACCEPT (@ref MODE_REJECT).
     *
     * @throw never
     */ 
    virtual bool p_missing_test_data(::std::string const & id, 
				     ::std::string & data) const;
	
    /**
     * @brief If @ref mode()== @ref MODE_INTERACTIVE, the user is
     * asked whether to accept @a new_data as new value for @a id
     * instead of @a data. @a new_data can be modified too. Otherwise,
     * it returns true (false) if @ref mode()== @ref MODE_ACCEPT 
     * (@ref MODE_REJECT).
     *
     * @throw never
     */
    virtual bool p_mismatching_test_data(::std::string const & id,
					 ::std::string const & data,
					 ::std::string & new_data) const;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name State
     * @{
     */
private:
    ::std::istream & m_input;
    ::std::ostream & m_output;
    Mode_t m_mode;
    // @}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif

// vim:ts=4:sw=4
