/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "comment.h"

Comment::Comment(PumpController *pumpController,
                 QVariantMap commentMap,
                 QWidget *parent) : QFrame(parent)
{
    this->setSizePolicy(QSizePolicy::Minimum,
                        QSizePolicy::Minimum);

    this->pController = pumpController;
    this->commentID = commentMap["id"].toString();
    this->objectType = commentMap["objectType"].toString();
    QVariantMap commentAuthorMap = commentMap["author"].toMap();
    commentIsOwn = false;


    QString commentAuthorID = commentAuthorMap["id"].toString();
    commentAuthorID.remove(0, 5); // remove "acct:" at the beginning
    if (commentAuthorID == pController->currentUserID())
    {
        commentIsOwn = true; // Comment is ours!

        // Different frame style depending on whether the comment is ours or not
        this->setFrameStyle(QFrame::Sunken | QFrame::StyledPanel);
    }
    else
    {
        this->setFrameStyle(QFrame::Raised | QFrame::StyledPanel);
    }



    // Avatar pixmap
    avatarLabel = new QLabel();
    avatarLabel->setToolTip(commentAuthorID);
    QString commentAuthorAvatarURL = commentAuthorMap["image"].toMap()["url"].toString();
    QString commentAuthorAvatarFile = MiscHelpers::getCachedAvatarFilename(commentAuthorAvatarURL);
    if (QFile::exists(commentAuthorAvatarFile))
    {
        avatarLabel->setPixmap(QPixmap(commentAuthorAvatarFile)
                               .scaledToHeight(32, Qt::SmoothTransformation));
    }
    else
    {
        // Set placeholder
        avatarLabel->setPixmap(QIcon::fromTheme("user-identity").pixmap(32, 32));
        // and download avatar for next time
        this->pController->getAvatar(commentAuthorAvatarURL);
    }


    QFont commentsFont;
    commentsFont.setPointSize(commentsFont.pointSize() - 1); // 1 point less than default


    // Name, with ID as tooltip
    commentsFont.setBold(true);

    fullNameLabel = new QLabel(commentAuthorMap["displayName"].toString());
    fullNameLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    fullNameLabel->setFont(commentsFont);
    fullNameLabel->setToolTip(commentAuthorID);


    // Timestamp
    commentsFont.setBold(false);
    commentsFont.setItalic(true);

    QString timestamp = commentMap["published"].toString();
    QString commentExactTime = Timestamp::localTimeDate(timestamp);
    QString commentFuzzyTime = Timestamp::fuzzyTime(timestamp);
    timestampLabel = new QLabel(commentFuzzyTime);
    timestampLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    timestampLabel->setFont(commentsFont);
    timestampLabel->setToolTip(commentExactTime);



    // Like and Delete "buttons"
    commentsFont.setBold(true);
    commentsFont.setItalic(false);

    likeLabel = new QLabel("*like*");
    likeLabel->setContextMenuPolicy(Qt::NoContextMenu);
    likeLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    likeLabel->setFont(commentsFont);
    likeLabel->setToolTip(tr("Like or unlike this comment"));
    connect(likeLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(likeComment(QString)));

    commentIsLiked = commentMap["liked"].toBool();
    this->fixLikeLabelText();

    deleteLabel = new QLabel("<a href=\"delete://\">" + tr("Delete") + "</a>");
    deleteLabel->setContextMenuPolicy(Qt::NoContextMenu);
    deleteLabel->setAlignment(Qt::AlignTop | Qt::AlignRight);
    deleteLabel->setFont(commentsFont);
    deleteLabel->setToolTip(tr("Delete this comment"));
    connect(deleteLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(deleteComment()));


    // The likes count
    likesCountLabel = new QLabel();
    likesCountLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    commentsFont.setBold(false);
    likesCountLabel->setFont(commentsFont);

    this->setLikesCount(commentMap["likes"].toMap()["totalItems"].toInt(),
                        commentMap["likes"].toMap()["items"].toList());

    // The comment itself
    QStringList commentImageList = MiscHelpers::htmlWithReplacedImages(commentMap["content"].toString());
    QString commentContents = commentImageList.takeAt(0); // comment HTML with images replaced

    contentLabel = new QLabel(commentContents);
    contentLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    contentLabel->setFont(commentsFont);
    contentLabel->setTextInteractionFlags(Qt::TextBrowserInteraction);
    contentLabel->setWordWrap(true);
    contentLabel->setOpenExternalLinks(true);
    contentLabel->setTextFormat(Qt::RichText);
    contentLabel->setSizePolicy(QSizePolicy::Minimum,
                                QSizePolicy::MinimumExpanding);
    connect(contentLabel, SIGNAL(linkHovered(QString)),
            this, SLOT(showUrlInfo(QString)));


    // If the image list is not empty, get them
    if (!commentImageList.isEmpty())
    {
        qDebug() << "Comment has" << commentImageList.size() << "images included...";

        foreach (QString imageUrl, commentImageList)
        {
            if (QFile::exists(MiscHelpers::getCachedImageFilename(imageUrl)))
            {
                qDebug() << "Image in comment already cached, ignoring";
            }
            else
            {
                qDebug() << "Image in comment not cached, downloading";
                this->pController->getImage(imageUrl);
            }
        }
    }


    urlInfoLabel = new QLabel();
    urlInfoLabel->setFont(commentsFont);
    urlInfoLabel->setAutoFillBackground(true);
    urlInfoLabel->setForegroundRole(QPalette::ToolTipText);
    urlInfoLabel->setBackgroundRole(QPalette::ToolTipBase);
    urlInfoLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    urlInfoLabel->hide(); // Hidden initially

    contentLabelLayout = new QHBoxLayout();
    contentLabelLayout->addWidget(urlInfoLabel, 0, Qt::AlignRight | Qt::AlignBottom);
    contentLabel->setLayout(contentLabelLayout);



    leftLayout = new QVBoxLayout();
    leftLayout->addWidget(avatarLabel,     0, Qt::AlignLeft | Qt::AlignTop);
    leftLayout->addWidget(likesCountLabel, 1, Qt::AlignHCenter | Qt::AlignTop);

    rightTopLayout = new QHBoxLayout();
    rightTopLayout->addWidget(fullNameLabel,  0, Qt::AlignLeft);
    rightTopLayout->addWidget(timestampLabel, 0, Qt::AlignLeft);
    rightTopLayout->addWidget(likeLabel,      0, Qt::AlignLeft);
    rightTopLayout->addStretch(1);
    if (commentIsOwn)
    {
        rightTopLayout->addWidget(deleteLabel, 0, Qt::AlignRight);
    }

    rightLayout = new QVBoxLayout();
    rightLayout->addLayout(rightTopLayout, 0);
    rightLayout->addSpacing(4); // 4px space
    rightLayout->addWidget(contentLabel,   2);    

    mainLayout = new QHBoxLayout();
    mainLayout->addLayout(leftLayout,  1);
    mainLayout->addLayout(rightLayout, 15);
    this->setLayout(mainLayout);

    qDebug() << "Comment created";
}


Comment::~Comment()
{
    qDebug() << "Comment destroyed";
}



void Comment::fixLikeLabelText()
{
    if (commentIsLiked)
    {
        this->likeLabel->setText("<a href=\"unlike://\">" + tr("Unlike") +"</a>");
    }
    else
    {
        this->likeLabel->setText("<a href=\"like://\">" + tr("Like") +"</a>");
    }
}


void Comment::setLikesCount(int count, QVariantList namesVariantList)
{
    if (count > 0)
    {
        QString likesString;
        foreach (QVariant likesMap, namesVariantList)
        {
            likesString.append(likesMap.toMap()["displayName"].toString() + ", ");
        }
        likesString.remove(-2, 2); // remove last comma+space: ", "
        if (count > 1)
        {
            likesString = tr("%1 like this comment",
                             "Plural: %1=list of people like John, Jane, Smith").arg(likesString);
        }
        else
        {
            likesString = tr("%1 likes this comment",
                             "Singular: %1=name of just 1 person").arg(likesString);
        }


        likesCountLabel->setText(QString::fromUtf8("\342\231\245")  // heart symbol
                                 + QString(" %1").arg(count));
        // set tooltip as HTML, so it gets wordwrapped
        likesCountLabel->setToolTip(likesString + "<b></b>");
    }
    else
    {
        likesCountLabel->clear();
        likesCountLabel->setToolTip("");
    }
}



/*
 * Ensure urlInfoLabel is hidden when the mouse leaves the comment
 *
 */
void Comment::leaveEvent(QEvent *event)
{
    this->urlInfoLabel->clear();
    this->urlInfoLabel->hide();

    event->accept();
}



/****************************************************************************/
/******************************** SLOTS *************************************/
/****************************************************************************/



void Comment::likeComment(QString clickedLink)
{

    if (clickedLink == "like://")
    {
        commentIsLiked = true;
    }
    else // unlike://
    {
        commentIsLiked = false;
    }

    this->pController->likePost(this->commentID,
                                this->objectType,
                                this->commentIsLiked);
    this->fixLikeLabelText();
}


void Comment::deleteComment()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete comment?"),
                                             tr("Are you sure you want to delete this comment?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Deleting comment" << this->commentID;
        this->pController->deletePost(this->commentID, this->objectType);

        this->setDisabled(true); // disable... maybe hide?
    }
    else
    {
        qDebug() << "Confirmation canceled, not deleting the comment";
    }
}


/*
 * Show the URL of a link hovered in a comment
 *
 */
void Comment::showUrlInfo(QString url)
{
    if (url.isEmpty())
    {
        this->urlInfoLabel->clear();
        this->urlInfoLabel->hide();
    }
    else
    {
        qDebug() << "Link hovered in comment:" << url;
        this->urlInfoLabel->setText(url);
        this->urlInfoLabel->show();
    }
}
