/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "composer.h"


Composer::Composer(bool forPublisher)
{
    this->forPublisher = forPublisher;

    this->setAcceptRichText(true);
    this->setTabChangesFocus(true);


    startConversationLabel = new QLabel(tr("Click here or press Control+N to post a note..."));

    // A menu to insert some Unicode symbols
    symbolsMenu = new QMenu(tr("Symbols"));
    symbolsMenu->setIcon(QIcon::fromTheme("character-set"));
    symbolsMenu->addAction(QString::fromUtf8("\342\230\272")); // Smiling face
    symbolsMenu->addAction(QString::fromUtf8("\342\230\271")); // Sad face
    symbolsMenu->addAction(QString::fromUtf8("\342\231\245")); // Heart
    symbolsMenu->addAction(QString::fromUtf8("\342\231\253")); // Musical note
    symbolsMenu->addAction(QString::fromUtf8("\342\230\225")); // Coffee
    symbolsMenu->addAction(QString::fromUtf8("\342\234\224")); // Check mark
    symbolsMenu->addAction(QString::fromUtf8("\342\230\205")); // Black star
    symbolsMenu->addAction(QString::fromUtf8("\342\254\205")); // Arrow to the left
    symbolsMenu->addAction(QString::fromUtf8("\342\236\241")); // Arrow to the right
    connect(symbolsMenu, SIGNAL(triggered(QAction*)),
            this, SLOT(insertSymbol(QAction*)));


    toolsMenu = new QMenu(tr("Formatting"));
    toolsMenu->addAction(QIcon::fromTheme(""),
                         tr("Normal"),
                         this,
                         SLOT(makeNormal()));
    toolsMenu->addAction(QIcon::fromTheme("format-text-bold"),
                         tr("Bold"),
                         this,
                         SLOT(makeBold()),
                         QKeySequence::Bold);
    toolsMenu->addAction(QIcon::fromTheme("format-text-italic"),
                         tr("Italic"),
                         this,
                         SLOT(makeItalic()),
                         QKeySequence::Italic);
    toolsMenu->addAction(QIcon::fromTheme("format-text-underline"),
                         tr("Underline"),
                         this,
                         SLOT(makeUnderline()),
                         QKeySequence::Underline);

    toolsMenu->addSeparator();

    toolsMenu->addAction(QIcon::fromTheme("format-justify-fill"),
                         tr("Preformatted block"),
                         this,
                         SLOT(makePreformatted()));

    toolsMenu->addSeparator();

    toolsMenu->addAction(QIcon::fromTheme("insert-link"),
                         tr("Make a link"),
                         this,
                         SLOT(makeLink()),
                         QKeySequence("Ctrl+L"));
    toolsMenu->addAction(QIcon::fromTheme("insert-image"),
                         tr("Insert an image from a web site"),
                         this,
                         SLOT(insertImage()));

    toolsMenu->addSeparator();

    toolsMenu->addMenu(symbolsMenu);

    toolsMenu->addSeparator();

    toolsMenu->addAction(QIcon::fromTheme("dialog-cancel"),
                         tr("Cancel Post"),
                         this,
                         SLOT(cancelPost()));

    toolsButton = new QPushButton(QIcon::fromTheme("format-list-ordered"), tr("&F", "F for formatting, translate accordingly"));
    toolsButton->setMenu(toolsMenu);
    toolsButton->setToolTip(tr("Text Formatting Options"));
    toolsButton->setHidden(true); // Hidden on startup (Publisher in miniMode)


    this->customContextMenu = this->createStandardContextMenu();
    this->customContextMenu->addMenu(toolsMenu);

    mainLayout = new QGridLayout();


    if (this->forPublisher)  // Publisher mode
    {
        this->setToolTip(tr("Type a message here to post it"));

        mainLayout->addWidget(startConversationLabel, 0, 0, 1, 1, Qt::AlignLeft | Qt::AlignVCenter);
    }
    else                     // or Commenter mode
    {
        this->setToolTip(tr("Type a comment here.\n"
                            "You can press Control+Enter to send the comment\n"
                            "with the keyboard, or ESC to cancel."));
    }

    mainLayout->addWidget(toolsButton,            0, 1, 1, 1, Qt::AlignRight | Qt::AlignBottom);
    this->setLayout(mainLayout);

    qDebug() << "Composer box created";
}


Composer::~Composer()
{
    qDebug() << "Composer box destroyed";
}



void Composer::erase()
{
    this->clear();

    if (this->forPublisher)
    {
        startConversationLabel->show();

        toolsButton->hide();  /// FIXME: maybe move this line to focusOutEvent
    }
}


/*
 * Send a signal when getting focus
 *
 */
void Composer::focusInEvent(QFocusEvent *event)
{
    emit focusReceived(); // inform Publisher() or Commenter() that we have focus

    if (this->forPublisher)
    {
        startConversationLabel->hide(); // hide placeholder message        
    }
    toolsButton->show();

    QTextEdit::focusInEvent(event); // process standard event: allows context menu


    qDebug() << "Composer box got focus";
}



void Composer::keyPressEvent(QKeyEvent *event)
{
    //qDebug() << "Composer::keyPressEvent()";


    // Control+Enter = Send message (post)
    if ((event->key() == Qt::Key_Enter || event->key() == Qt::Key_Return)
      && event->modifiers() == Qt::ControlModifier)
    {
        qDebug() << "Control+Enter was pressed";
        emit editingFinished();
    }
    else if (event->key() == Qt::Key_Escape)
    {
        qDebug() << "Escape was pressed";
        if (this->toPlainText().isEmpty())
        {
            qDebug() << "There was no text, canceling post";
            this->cancelPost();
        }
    }
    else
    {
        QTextEdit::keyPressEvent(event);
    }
}


/*
 * For custom context menu
 *
 */
void Composer::contextMenuEvent(QContextMenuEvent *event)
{
    customContextMenu->exec(event->globalPos());
}



/*************************************************************************/
/****************************** SLOTS ************************************/
/*************************************************************************/



/*
 * Remove text formatting from selection, bold, italic, etc.
 *
 */
void Composer::makeNormal()
{
    this->setCurrentCharFormat(QTextCharFormat());
}



/*
 * Make selected text bold
 *
 */
void Composer::makeBold()
{
    //qDebug() << this->textCursor().selectionStart() << " -> "  << this->textCursor().selectionEnd();
    QTextCharFormat charFormat;
    if (this->currentCharFormat().fontWeight() == QFont::Bold)
    {
        charFormat.setFontWeight(QFont::Normal);
    }
    else
    {
        charFormat.setFontWeight(QFont::Bold);
    }
    this->mergeCurrentCharFormat(charFormat);

    this->setFocus(); // give focus back to text editor
}


/*
 * Make selected text italic
 *
 */
void Composer::makeItalic()
{
    QTextCharFormat charFormat;
    charFormat.setFontItalic(!this->currentCharFormat().fontItalic());
    this->mergeCurrentCharFormat(charFormat);

    this->setFocus();
}



/*
 * Underline selected text
 *
 */
void Composer::makeUnderline()
{
    QTextCharFormat charFormat;
    charFormat.setFontUnderline(!this->currentCharFormat().fontUnderline());
    this->mergeCurrentCharFormat(charFormat);

    this->setFocus();
}



/*
 * Put selected text into a <pre> block
 *
 */
void Composer::makePreformatted()
{
    QString selectedText = this->textCursor().selectedText();

    if (!selectedText.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<pre>" + selectedText + "</pre> ");
    }

    this->setFocus();
}




/*
 * Convert selected text into a link
 *
 */
void Composer::makeLink()
{
    QString selectedText = this->textCursor().selectedText();
    QString link;


    if (selectedText.isEmpty())
    {
        link = QInputDialog::getText(this,
                                     tr("Insert a link"),
                                     tr("Type or paste a web address here.\n"),
                                     QLineEdit::Normal,
                                     "http://");
        selectedText = link;
    }
    else
    {
        link = QInputDialog::getText(this,
                                     tr("Make a link from selected text"),
                                     tr("Type or paste a web address here.\n"
                                        "The selected text (%1) will be converted to a link.").arg(selectedText),
                                     QLineEdit::Normal,
                                     "http://");
    }



    if (!link.isEmpty())
    {
        link = link.trimmed(); // Remove possible spaces before or after

        this->textCursor().removeSelectedText();
        this->insertHtml("<a href=\"" + link + "\">" + selectedText + "</a> ");
    }
    else
    {
        qDebug() << "makeLink(): Invalid URL";
    }

    this->setFocus();
}



/*
 * Insert an image from a URL
 *
 */
void Composer::insertImage()
{
    // FIXME: should make sure there is no text selected

    QString imageURL;
    imageURL = QInputDialog::getText(this,
                                     tr("Insert an image from a URL"),
                                     tr("Type or paste the image address here.\n"),
                                     QLineEdit::Normal,
                                     "http://");


    if (!imageURL.isEmpty())
    {
        this->insertHtml("<img src=\"" + imageURL + "\" />");
    }
    else
    {
        qDebug() << "insertImage(): Invalid image URL";
    }

    this->setFocus();
}



void Composer::insertSymbol(QAction *action)
{
    this->insertPlainText(action->text());
    this->insertPlainText(" ");

    this->setFocus();
}




/*
 * Cancel editing of the post, clear it, return to minimum mode
 *
 */
void Composer::cancelPost()
{
    int cancelConfirmed = 1;

    if (this->document()->isEmpty())
    {
        cancelConfirmed = 0; // Cancelling doesn't need confirmation if it's empty
    }
    else
    {
        cancelConfirmed = QMessageBox::question(this,
                                                tr("Cancel message?"),
                                                tr("Are you sure you want to cancel this message?"),
                                                tr("&Yes, cancel it"), tr("&No"),
                                                "", 1, 1);

    }


    if (cancelConfirmed == 0)
    {
        this->erase();

        // emit signal to make Publisher go back to minimum mode
        emit editingCancelled();

        qDebug() << "Post canceled";
    }


}
