/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "pumpcontroller.h"

PumpController::PumpController(QObject *parent) :  QObject(parent)
{
    this->userAgentString = "Dianara/0.9";

    this->postsPerPage = 20;
    this->updatesToTimelineBlocked = false;

    qoauth = new QOAuth::Interface();
    qoauth->setRequestTimeout(10000); // 10 sec timeout


    QSettings settings;
    this->clientID = settings.value("clientID", "").toString();
    this->clientSecret = settings.value("clientSecret", "").toString();
    qoauth->setConsumerKey(clientID.toLocal8Bit());
    qoauth->setConsumerSecret(clientSecret.toLocal8Bit());


    this->isApplicationAuthorized = settings.value("isApplicationAuthorized", false).toBool();

    if (isApplicationAuthorized)
    {
        qDebug() << "Dianara is already authorized for user ID:" << settings.value("userID").toString();

        this->token = settings.value("token", "").toString().toLocal8Bit();
        this->tokenSecret = settings.value("tokenSecret", "").toString().toLocal8Bit();

        qDebug() << "Using token" << token;
        qDebug() << "And token secret" << tokenSecret;
    }
    emit this->authorizationStatusChanged(isApplicationAuthorized);




    connect(&nam, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)));

    // FIXME: setting this up for now, to at least have debug messages just in case
    connect(&nam, SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)),
            this, SLOT(sslErrorsHandler(QNetworkReply*,QList<QSslError>)));


    this->initialDataStep = 0;

    initialDataTimer = new QTimer(this);
    initialDataTimer->setSingleShot(false); // Triggered constantly until stopped
    connect(initialDataTimer, SIGNAL(timeout()),
            this, SLOT(getInitialData()));



    qDebug() << "PumpController created";
}


PumpController::~PumpController()
{
    qDebug() << "PumpController destroyed";
}



void PumpController::setPostsPerPage(int ppp)
{
    this->postsPerPage = ppp;
    qDebug() << "PumpController: setting postsPerPage to" << this->postsPerPage;
}

/*
 *
 *
 */
void PumpController::setUpdatesToTimelineBlocked(bool blocked)
{
    this->updatesToTimelineBlocked = blocked;
    qDebug() << "PumpController: Updates to timeline blocked? " << this->updatesToTimelineBlocked;
}



/*
 * Set new user ID (user@domain.tld) and clear OAuth-related tokens/secrets
 *
 *
 */
void PumpController::setNewUserID(QString userID)
{
    this->userID = userID;
    QStringList splittedUserID = this->userID.split("@");
    this->userName = splittedUserID.at(0); // get username, before @
    this->serverURL = splittedUserID.at(1); // get URL, after @

    qDebug() << "Server URL to connect:" << serverURL << "; username:" << userName;

    this->clientID.clear();
    this->clientSecret.clear();
    this->token.clear();
    this->tokenSecret.clear();

    this->isApplicationAuthorized = false;
    emit this->authorizationStatusChanged(isApplicationAuthorized);
}




/*
 * Get "pumpserver.org" and "user" from "user@pumpserver.org", set OAuth token from Account dlg
 *
 */
void PumpController::setUserCredentials(QString userID)
{
    this->initialDataTimer->stop(); // Just in case it was running before


    this->userID = userID;
    QStringList splittedUserID = this->userID.split("@");
    this->userName = splittedUserID.at(0);
    this->serverURL = splittedUserID.at(1);
    qDebug() << "New userID is:" << this->userID;


    this->getUserProfile(this->userID);


    // This will call getContactList() and getMainTimeline(), etc.
    this->initialDataStep = 0;
    this->initialDataTimer->start(2000);  // start 2 seconds after setting the ID
                                          // (mainly on program startup)
}



QString PumpController::currentUserID()
{
    return this->userID;
}

QString PumpController::currentUsername()
{
    return this->userName;
}

QString PumpController::currentFollowersURL()
{
    return this->userFollowersURL;
}





/*
 * Get any user's profile (not only our own)
 *
 * GET https://pumpserver.example/api/user/username
 *
 */
void PumpController::getUserProfile(QString userID)
{
    QStringList splittedUserID = userID.split("@");

    QString url = "https://" + splittedUserID.at(1) + "/api/user/" + splittedUserID.at(0);

    QNetworkRequest userProfileRequest = this->prepareRequest(url, QOAuth::GET, UserProfileRequest);
    nam.get(userProfileRequest);

    qDebug() << "Requested user profile:" << userProfileRequest.url();
}


/*
 * Update user's profile
 *
 */
void PumpController::updateUserProfile(QString avatarURL, QString fullName,
                                       QString hometown, QString bio)
{
    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/profile";

    QNetworkRequest updateProfileRequest = this->prepareRequest(url, QOAuth::PUT,
                                                                UpdateProfileRequest);


    QVariantMap jsonVariantImage;
    jsonVariantImage.insert("url", avatarURL);

    QVariantMap jsonVariantLocation;
    jsonVariantLocation.insert("objectType", "place");
    jsonVariantLocation.insert("displayName", hometown);

    QVariantMap jsonVariant;
    jsonVariant.insert("objectType", "person");
    // TODO:
    // jsonVariant.insert("image", jsonVariantImage);
    jsonVariant.insert("displayName", fullName);
    jsonVariant.insert("location", jsonVariantLocation);
    jsonVariant.insert("summary", bio);



    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);



    nam.put(updateProfileRequest, data);
    qDebug() << "Updating user profile" << fullName << hometown;
}




void PumpController::getAvatar(QString avatarURL)
{
    if (avatarURL.isEmpty())
    {
        return;
    }

    qDebug() << "Getting avatar";

    QNetworkRequest avatarRequest(QUrl((const QString)avatarURL));
    avatarRequest.setRawHeader("User-Agent", userAgentString);
    avatarRequest.setAttribute(QNetworkRequest::User,
                               QVariant(AvatarRequest));

    nam.get(avatarRequest);
}



void PumpController::getImage(QString imageURL)
{
    if (imageURL.isEmpty())
    {
        return;
    }

    QNetworkRequest imageRequest = this->prepareRequest(imageURL, QOAuth::GET, ImageRequest);

    nam.get(imageRequest);
    qDebug() << "imageRequest sent";
}




/*
 * GET https://pumpserver.example/api/user/username/following or /followers
 *
 */
void PumpController::getContactList(QString listType)
{
    qDebug() << "Getting contact list, type" << listType;

    QString url = "https://" + this->serverURL + "/api/user/" +this->userName + "/" + listType;

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "200");  // TMP/FIXME get 200 contacts
                                      // To get more, we need pagination

    QNetworkRequest contactListRequest;
    if (listType == "following")
    {
        emit currentJobChanged(tr("Getting list of 'Following'..."));
        contactListRequest = this->prepareRequest(url, QOAuth::GET,
                                                  FollowingListRequest, paramMap);
    }
    else
    {
        emit currentJobChanged(tr("Getting list of 'Followers'..."));
        contactListRequest = this->prepareRequest(url, QOAuth::GET,
                                                  FollowersListRequest, paramMap);
    }

    nam.get(contactListRequest);
}



/*
 * GET https://pumpserver.example/api/user/username/lists/person
 *
 */
void PumpController::getListsList()
{
    qDebug() << "Getting list of lists";

    QString url = "https://" + this->serverURL + "/api/user/" +this->userName + "/lists/person";

    QNetworkRequest listsListRequest = this->prepareRequest(url, QOAuth::GET,
                                                            ListsListRequest,
                                                            QOAuth::ParamMap());

    emit currentJobChanged(tr("Getting list of lists..."));

    nam.get(listsListRequest);
}




/*
 * Get main timeline
 *
 * GET https://pumpserver.example/api/username/inbox/major
 *
 */
void PumpController::getMainTimeline(int timelineOffset)
{
    if (this->updatesToTimelineBlocked)
    {
        qDebug() << "Updating timelines requested, but updates are blocked; ignoring";
        return;
    }

    emit currentJobChanged(tr("Getting main timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPage).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           MainTimelineRequest, paramMap);


    qDebug() << "===================================";
    qDebug() << "Getting timeline" << url;
    qDebug() << "with params:" << paramMap;
    qDebug() << "\nAuthorization Header:" << timelineRequest.rawHeader("Authorization");
    qDebug() << "\n\nFinal URL to retrieve:" << timelineRequest.url();
    qDebug() << "===================================";

    nam.get(timelineRequest);
}



/*
 * Get direct timeline, posts with the user's address in the "To:" field,
 * that is, sent explicitly to the user
 *
 * GET https://pumpserver.example/api/username/inbox/direct
 *
 */
void PumpController::getDirectTimeline(int timelineOffset)
{
    emit currentJobChanged(tr("Getting direct messages timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/direct/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPage).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           DirectTimelineRequest, paramMap);

    nam.get(timelineRequest);
}



/*
 * Get activity timeline, user's own posts
 *
 * GET https://pumpserver.example/api/username/feed/major
 *
 */
void PumpController::getActivityTimeline(int timelineOffset)
{
    emit currentJobChanged(tr("Getting activity timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPage).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           ActivityTimelineRequest, paramMap);

    nam.get(timelineRequest);
}



/*
 * Get favorites timeline, posts where user clicked "like"
 *
 * GET https://pumpserver.example/api/username/favorites
 *
 */
void PumpController::getFavoritesTimeline(int timelineOffset)
{
    emit currentJobChanged(tr("Getting favorites timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/favorites";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPage).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           FavoritesTimelineRequest, paramMap);

    nam.get(timelineRequest);
}






/*
 * Get list of people who liked a specific post
 *
 */
void PumpController::getPostLikes(QString postLikesURL)
{
    qDebug() << "Getting likes for post" << postLikesURL;

    emit currentJobChanged(tr("Getting likes..."));


    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "50"); // TMP, up to 50 likes

    QNetworkRequest likesRequest = this->prepareRequest(postLikesURL, QOAuth::GET,
                                                        PostLikesRequest, paramMap);

    nam.get(likesRequest);
}



/*
 * Get comments for one specific post
 *
 * GET https://pumpserver.example/api/note/#id#/replies
 * or proxyed URL. URL is given by the post itself anyway
 *
 */
void PumpController::getPostComments(QString postCommentsURL)
{
    qDebug() << "Getting comments for post" << postCommentsURL;

    emit currentJobChanged(tr("Getting comments..."));


    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "50"); // TMP, up to 50 comments

    QNetworkRequest commentsRequest = this->prepareRequest(postCommentsURL, QOAuth::GET,
                                                           PostCommentsRequest, paramMap);

    nam.get(commentsRequest);
}


/*
 * Get list of people who shared a specific post
 *
 */
void PumpController::getPostShares(QString postSharesURL)
{

}



/*
 * Get the minor feed, used in the "Meanwhile" column
 *
 * GET https://pumpserver.example/api/username/inbox/minor
 *
 */
void PumpController::getMinorFeed()
{
    emit currentJobChanged(tr("Getting minor feed..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/minor";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "40");

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           MinorFeedRequest, paramMap);


    nam.get(timelineRequest);
}



/*
 * Prepare a QNetworkRequest with OAuth header, content type and user agent.
 *
 */
QNetworkRequest PumpController::prepareRequest(QString url, QOAuth::HttpMethod method,
                                               int requestType, QOAuth::ParamMap paramMap,
                                               QString contentTypeString)
{
    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 method,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 paramMap,
                                                                 QOAuth::ParseForHeaderArguments);

    QNetworkRequest request;

    // Don't append inline parameters if they're empty, tha can mess up things
    if (!paramMap.isEmpty())
    {
        url.append(qoauth->inlineParameters(paramMap, QOAuth::ParseForInlineQuery));
    }
    request.setUrl(QUrl(url));
    // Only add Authorization header if we're requesting something in our server
    if (request.url().host() == this->serverURL)
    {
        request.setRawHeader("Authorization", authorizationHeader);
    }
    request.setHeader(QNetworkRequest::ContentTypeHeader, contentTypeString);
    request.setRawHeader("User-Agent", userAgentString);


    request.setAttribute(QNetworkRequest::User,  QVariant(requestType));

    return request;
}



/*
 * Upload a file to the /uploads feed for the user
 *
 * Used to upload pictures, but might have more uses in the future
 *
 */
void PumpController::uploadFile(QString filename, QString contentType)
{
    qDebug() << "PumpController::uploadFile()";

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/uploads";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       UploadFileRequest,
                                                       QOAuth::ParamMap(),
                                                       contentType);

    QFile file(filename);
    file.open(QIODevice::ReadOnly);

    QByteArray data = file.readAll();
    file.close();

    nam.post(postRequest, data);
}



QList<QVariantList> PumpController::processAudience(QMap<QString, QString> audienceMap)
{
    QVariantList jsonVariantTo;
    QVariantList jsonVariantCC;

    QVariantMap jsonVariantAudienceItem;
    while (!audienceMap.isEmpty())
    {
        jsonVariantAudienceItem.clear();

        if (audienceMap.keys().contains("to|collection")) // To:
        {
            QString collectionID = audienceMap.take("to|collection");
            jsonVariantAudienceItem.insert("objectType", "collection");
            jsonVariantAudienceItem.insert("id", collectionID);

            jsonVariantTo.append(jsonVariantAudienceItem);
        }
        else if (audienceMap.keys().contains("to|person"))
        {
            QString personID = audienceMap.take("to|person");
            jsonVariantAudienceItem.insert("objectType", "person");
            jsonVariantAudienceItem.insert("id", personID);

            jsonVariantTo.append(jsonVariantAudienceItem);
        }
        else if (audienceMap.keys().contains("cc|collection")) // CC:
        {
            QString collectionID = audienceMap.take("cc|collection");
            jsonVariantAudienceItem.insert("objectType", "collection");
            jsonVariantAudienceItem.insert("id", collectionID);

            jsonVariantCC.append(jsonVariantAudienceItem);
        }
        else if (audienceMap.keys().contains("cc|person"))
        {
            QString personID = audienceMap.take("cc|person");
            jsonVariantAudienceItem.insert("objectType", "person");
            jsonVariantAudienceItem.insert("id", personID);

            jsonVariantCC.append(jsonVariantAudienceItem);
        }
    }


    QList<QVariantList> jsonVariantToAndCCList;
    jsonVariantToAndCCList.append(jsonVariantTo);
    jsonVariantToAndCCList.append(jsonVariantCC);


    return jsonVariantToAndCCList;
}



/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/*********************************** SLOTS *********************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/




void PumpController::requestFinished(QNetworkReply *reply)
{
    int httpCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    int requestType = reply->request().attribute(QNetworkRequest::User).toInt();

    qDebug() << "Request finished. HTTP code:" << httpCode;
    qDebug() << "Size:" << reply->size() << "bytes; URL:" << reply->url();
    qDebug() << "isFinished()?" << reply->isFinished() << "; Request type:" << requestType;


    // Special control after sending a post or a comment
    if (httpCode != 200) // if not OK
    {
        if (requestType == PublishPostRequest)
        {
            emit postPublishingFailed();
        }
        else if (requestType == UpdatePostRequest)
        {
            emit postPublishingFailed();  // kinda TMP
        }
        else if (requestType == CommentPostRequest)
        {
            emit commentPostingFailed();
        }
    }


    switch (httpCode)
    {
    case 503:
        // if not just and image, it's important, so popup notification
        if (requestType != ImageRequest && requestType != AvatarRequest)
        {
            emit showNotification("Service Unavailable (503)\n" + reply->url().toString());
        }
        else // just status bar
        {
            emit currentJobChanged("Service Unavailable (503): " + reply->url().toString());
        }

        qDebug() << "HTTP 503: Service Unavailable.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 500:
        // if not just and image, it's important, so popup notification
        if (requestType != ImageRequest && requestType != AvatarRequest)
        {
            emit showNotification("Internal server error (500)\n" + reply->url().toString());
        }
        else
        {
            emit currentJobChanged("Internal server error (500): " + reply->url().toString());
        }
        qDebug() << "HTTP 500: Internal server error.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 410:
        emit currentJobChanged("Gone (410): " + reply->url().toString());
        qDebug() << "HTTP 410: Gone.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 404:
        emit currentJobChanged("Not Found (404): " + reply->url().toString());
        qDebug() << "HTTP 404: Not Found.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 403:
        emit currentJobChanged("Forbidden (403): " + reply->url().toString());
        qDebug() << "HTTP 403: Forbidden.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 400:
        // if not just and image, it's important, so popup notification
        if (requestType != ImageRequest && requestType != AvatarRequest)
        {
            emit showNotification("Bad Request (400)\n" + reply->url().toString());
        }
        else
        {
            emit currentJobChanged("Bad Request (400): " + reply->url().toString());
        }
        qDebug() << "HTTP 400: Bad Request.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 301:
        emit currentJobChanged("Moved Permanently (301): " + reply->url().toString());
        qDebug() << "HTTP 301: Moved Permanently.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 0:
        emit currentJobChanged(tr("Error connecting to %1").arg(reply->url().host()) + ": " + reply->errorString());
        qDebug() << "Error connecting to" << reply->url().host();
        return;

    // Other HTTP codes
    default:
        qDebug() << "Unhandled HTTP error " << httpCode;
        qDebug() << "Data:  " << reply->readAll();
        emit currentJobChanged(tr("Unhandled HTTP error code %1").arg(httpCode)
                              + ": " + reply->url().toString());

        return;


    //////////////////////////////////////// The good one!
    case 200:
        qDebug() << "HTTP 200: OK!";
    }


    // At this point, httpCode should be 200 = OK


    // Read all received data
    QByteArray packetData = reply->readAll();

    // Prepare the JSON parser
    QJson::Parser jsonParser;
    bool jsonParsedOK = false;
    QVariantMap jsonData;
    QVariantList jsonDataList;


    // Unless it was an AvatarRequest or an ImageRequest, it should be JSON, so parse it
    if (requestType != AvatarRequest && requestType != ImageRequest)
    {
        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();
        qDebug() << "Keys:" << jsonData.keys();
    }


    //////////////////////////////////////////////////////////////////

    switch (requestType)
    {

    case ClientRegistrationRequest:
        qDebug() << "Client Registration was requested";

        qDebug() << "Raw JSON:" << jsonData;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            this->clientID = jsonData["client_id"].toString();
            this->clientSecret = jsonData["client_secret"].toString();

            // FIXME: error control, etc.
            // check if jsonData.keys().contains("client_id") !!

            QSettings settings;
            settings.setValue("clientID",     this->clientID);
            settings.setValue("clientSecret", this->clientSecret);

            this->getToken();
        }

        break;



    case UserProfileRequest:
        qDebug() << "A user profile was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            QVariantMap profileMap = jsonData["profile"].toMap();

            if (profileMap["id"].toString() == "acct:" + this->userID)
            {
                qDebug() << "Received OWN profile";

                QString profImageUrl = profileMap["image"].toMap()["url"].toString();
                QString profDisplayName = profileMap["displayName"].toString();
                QString profLocation = profileMap["location"].toMap()["displayName"].toString();
                QString profSummary = profileMap["summary"].toString();

                emit profileReceived(profImageUrl, profDisplayName,
                                     profLocation, profSummary);

                // Store also the user's followers URL, for posting to Followers
                this->userFollowersURL = profileMap["followers"].toMap()["url"].toString();
            }
        }

        break;




    ////////////////////////////////////////////////// If a timeline was requested
    case MainTimelineRequest:
        // just jump to the next
    case DirectTimelineRequest:
        // just jump to next
    case ActivityTimelineRequest:
        // just... yeah, jump
    case FavoritesTimelineRequest:
        qDebug() << "A timeline was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Timeline received. Updating post list..."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in timeline:" << jsonDataList.size();

            QString previousLink = jsonData["links"].toMap().value("prev").toMap().value("href").toString();
            QString nextLink = jsonData["links"].toMap().value("next").toMap().value("href").toString();

            if (requestType == MainTimelineRequest)
            {
                qDebug() << "It was the main timeline";
                emit mainTimelineReceived(jsonDataList, this->postsPerPage,
                                          previousLink, nextLink);
            }
            else if (requestType == DirectTimelineRequest)
            {
                qDebug() << "It was the direct messages timeline";
                emit directTimelineReceived(jsonDataList, this->postsPerPage,
                                            previousLink, nextLink);
            }
            else if (requestType == ActivityTimelineRequest)
            {
                qDebug() << "It was the own activity timeline";
                emit activityTimelineReceived(jsonDataList, this->postsPerPage,
                                              previousLink, nextLink);
            }
            else if (requestType == FavoritesTimelineRequest)
            {
                qDebug() << "It was the favorites timeline";
                emit favoritesTimelineReceived(jsonDataList, this->postsPerPage,
                                               previousLink, nextLink);
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }
        break;



    ////////////////////////////////////
    case PublishPostRequest:
        emit currentJobChanged(tr("Post published successfully."));
        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QString objectID = jsonData["object"].toMap()["id"].toString();
            qDebug() << "Image post ID:" << objectID;

            if (jsonData["object"].toMap()["objectType"].toString() == "image")
            {
                // Update the image with title and description
                this->postImageStepThree(objectID);
            }
            else
            {
                // Not an image, notify "posted OK"
                emit postPublished();
                qDebug() << "Non-image post published correctly";
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;


    ////////////////////////////////////
    case UpdatePostRequest:
        qDebug() << "**************** POST UPDATED OK!";
        emit currentJobChanged(tr("Post updated successfully."));
        emit postPublished();
        break;



    ///////////////////////////////////// If liking a post was requested
    case LikePostRequest:
        emit currentJobChanged(tr("Message liked or unliked successfully."));
        emit likeSet();
        break;


    ///////////////////////////////////// If the likes for a post were requested
    case PostLikesRequest:
        qDebug() << "Likes for a post were requested" << reply->url().toString();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Likes received."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in comments list:" << jsonDataList.size();

            emit likesReceived(jsonDataList, reply->url().toString());
        }
        else
        {
            qDebug() << "Error parsing received comment JSON data!";
            qDebug() << "Raw data:" << packetData;
        }
        break;



    ///////////////////////////////////// If commenting on a post was requested
    case CommentPostRequest:
        emit currentJobChanged(tr("Comment posted successfully."));
        emit commentPosted(); // This will be catched by Commenter()

        break;



    ///////////////////////////////////// If the comments for a post were requested
    case PostCommentsRequest:
        qDebug() << "Comments for a post were requested" << reply->url().toString();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Comments received."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in comments list:" << jsonDataList.size();

            emit commentsReceived(jsonDataList, reply->url().toString());
        }
        else
        {
            qDebug() << "Error parsing received comment JSON data!";
            qDebug() << "Raw data:" << packetData;
        }
        break;



    case SharePostRequest:
        emit currentJobChanged(tr("Post shared successfully."));
        break;

    case PostSharesRequest:
        // TODO
        break;



    case MinorFeedRequest:
        qDebug() << "The minor feed was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Minor feed received."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in minor feed:" << jsonDataList.size();

             emit minorFeedReceived(jsonDataList);
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }
        break;



    case DeletePostRequest:
        emit currentJobChanged(tr("Message deleted successfully."));
        break;


    case FollowContactRequest:
        emit currentJobChanged(tr("Following successfully."));

        // Re-request the contact list, now with a new contact (not very optimized...)
        this->getContactList("following");

        break;

    case UnfollowContactRequest:
        emit currentJobChanged(tr("Stopped following successfully."));

        // Re-request the contact list, now with one contact less (not very optimized...)
        this->getContactList("following");

        break;


    case FollowingListRequest:
        // just go to the next
    case FollowersListRequest:
        qDebug() << "A contact list was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant contactsVariant = jsonData.value("items");
            if (contactsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing contacts...";

                if (requestType == FollowingListRequest)
                {
                    emit currentJobChanged(tr("List of 'following' received."));
                    emit contactListReceived("following", contactsVariant.toList());
                }
                else // == FollowersListRequest
                {
                    emit currentJobChanged(tr("List of 'followers' received."));
                    emit contactListReceived("followers", contactsVariant.toList());
                }
            }
            else
            {
                qDebug() << "Expected a list of contacts, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;


    case ListsListRequest:
        qDebug() << "The list of lists was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant listsVariant = jsonData.value("items");
            if (listsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing lists...";

                emit currentJobChanged(tr("List of 'lists' received."));
                emit listsListReceived(listsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a list of lists, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;



    case AvatarRequest:
        qDebug() << "Received AVATAR data, from " << reply->url();
        if (reply->isFinished())
        {
            qDebug() << "Avatar received 100%";
            emit avatarPictureReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Avatar not complete yet";
        }
        break;


    case ImageRequest:
        qDebug() << "Received IMAGE data, from " << reply->url();
        if (reply->isFinished())
        {
            qDebug() << "Image received 100%";
            emit imageReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Image not complete yet";
        }
        break;


    //////////////////////////////////////// If uploading a file was requested
    case UploadFileRequest:
        qDebug() << "Uploading a file was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("File uploaded successfully. Posting message..."));

            QString uploadedFileID = jsonData["id"].toString();
            qDebug() << "Uploaded file ID:" << uploadedFileID;


            if (jsonData["objectType"].toString() == "image")
            {
                this->postImageStepTwo(uploadedFileID);
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;


    }
    // end switch (requestType)
}






void PumpController::sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList)
{
    qDebug() << "\n==== SSL errors!! ====";
    qDebug() << "At:" << reply->url();
    qDebug() << "Error list:" << errorList << "\n\n";

    // ignore completely for now
    qDebug() << "Ignoring these errors...";
    reply->ignoreSslErrors();

    /*
    // Check list of SSL errors
    foreach (QSslError sslError, errorList)
    {
        // Ignore the "certificate is self-signed" SSL error, it's expected
        if (sslError.error() == QSslError::SelfSignedCertificate)
        {
            qDebug() << "Ignoring self-signed certificate error";
            reply->ignoreSslErrors(); // FIXME, maybe ask the user
            return;
        }
    }
    */
}




void PumpController::getToken()
{
    // If we do not have client_id or client_secret, do dynamic client registration
    if (this->clientID.isEmpty() || this->clientSecret.isEmpty())
    {
        qDebug() << "PumpController::getToken()";
        qDebug() << "We do not have client_id/client_secret yet; doing Dynamic Client Registration";

        // POST to https://hotpump.net/api/client/register, f.e.
        QNetworkRequest postRequest(QUrl("https://" + this->serverURL + "/api/client/register"));


        postRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
        postRequest.setRawHeader("User-Agent", userAgentString);
        postRequest.setAttribute(QNetworkRequest::User,
                                 QVariant(ClientRegistrationRequest));

        QByteArray data("{"
                        " \"type\": \"client_associate\",  "
                        " \"application_type\": \"native\", "
                        " \"application_name\": \"Dianara\" "
                        "}");

        qDebug() << "About to POST:" << data;


        nam.post(postRequest, data); // upon receiving data (id+secret), will execute getToken() again
    }
    else
    {
        qDebug() << "Using saved client_id and client_secret:" << this->clientID << this->clientSecret;

        // OAuth stuff.....
        // 1. obtaining an unauthorized Request Token from the Service Provider,
        // 2. asking the User to authorize the Request Token,
        // 3. exchanging the Request Token for the Access Token

        qDebug() << "Doing OAuth token stuff...";

        qDebug() << "NOTE: if you see a crash here, you need QCA and its openSSL plugin:";
        qDebug() << ">> qca2-plugin-openssl, libqca2-plugin-ossl, or similar";
        qDebug() << "If you compiled Dianara from source, check the INSTALL file carefully";

        QStringList QCAsupportedFeatures = QCA::supportedFeatures();
        qDebug() << "QCA Supported Features:" << QCAsupportedFeatures;

        if (QCAsupportedFeatures.contains("hmac(sha1)"))
        {
            qDebug() << "HMAC-SHA1 support is OK";
        }
        else
        {
            qDebug() << "Warning, HMAC-SHA1 doesn't seem to be supported!";

            // TMPFIX notify the user properly, etc.
        }

        qoauth->setConsumerKey(this->clientID.toLocal8Bit());
        qoauth->setConsumerSecret(this->clientSecret.toLocal8Bit());


        QString requestTokenURL = "https://" + this->serverURL + "/oauth/request_token";
        qDebug() << "GET: " << requestTokenURL << "with" << qoauth->consumerKey() << qoauth->consumerSecret();


        QOAuth::ParamMap oAuthParams;
        oAuthParams.insert("oauth_callback", "oob");


        QOAuth::ParamMap reply = qoauth->requestToken(requestTokenURL,
                                                      QOAuth::GET,
                                                      QOAuth::HMAC_SHA1,   // or PLAINTEXT?
                                                      oAuthParams);


        if (qoauth->error() == QOAuth::NoError)
        {
            qDebug() << "requestToken OK:" << reply.keys();

            token = reply.value(QOAuth::tokenParameterName());
            tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

            qDebug() << "Token:" << token;
            qDebug() << "Token Secret:" << tokenSecret;

            QUrl oAuthAuthorizeURL("https://" + this->serverURL + "/oauth/authorize");
#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
            oAuthAuthorizeURL.addQueryItem("oauth_token", token);
#else
            QUrlQuery query;
            query.addQueryItem("oauth_token", token);
            oAuthAuthorizeURL.setQuery(query);
#endif
            QDesktopServices::openUrl(oAuthAuthorizeURL);

            // Send also a signal, so AccountDialog can show the URL in
            // a label, in case the browser didn't launch
            emit openingAuthorizeURL(oAuthAuthorizeURL);

            // Now, user should enter VERIFIER in AccountDialog to authorize the program
        }
        else
        {
            qDebug() << "QOAuth error" << qoauth->error() << "!";

            qDebug() << reply.keys();
        }

    }
}



void PumpController::authorizeApplication(QString verifierCode)
{
    qDebug() << "Verifier code entered by user:" << verifierCode;

    QOAuth::ParamMap moreParams;
    moreParams.insert("oauth_verifier", verifierCode.toUtf8()); // verifier as QByteArray

    QString requestAuthorizationURL = "https://" + this->serverURL + "/oauth/access_token";

    QOAuth::ParamMap reply = qoauth->accessToken(requestAuthorizationURL,
                                                 QOAuth::GET,
                                                 token,
                                                 tokenSecret,
                                                 QOAuth::HMAC_SHA1,
                                                 moreParams);

    if (qoauth->error() == QOAuth::NoError) // Woooohooo!!
    {
        qDebug() << "Got authorized token; Dianara is authorized to access the account";
        token = reply.value(QOAuth::tokenParameterName());
        tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

        this->isApplicationAuthorized = true;

        QSettings settings;
        settings.setValue("isApplicationAuthorized", this->isApplicationAuthorized);
        settings.setValue("token",       this->token);
        settings.setValue("tokenSecret", this->tokenSecret);

        qDebug() << "Token:" << token;
        qDebug() << "TokenSecret:" << tokenSecret;

        emit this->authorizationStatusChanged(isApplicationAuthorized);
    }
    else
    {
        qDebug() << "OAuth error while authorizing application" << qoauth->error();
    }

}




/*
 * Called by a QTimer, get initial data (contacts, timeline...) one step at a time
 *
 */
void PumpController::getInitialData()
{
    qDebug() << "PumpController::getInitialData() step" << initialDataStep;

    initialDataTimer->setInterval(3000);  // Every 3 sec


    switch (this->initialDataStep)
    {
    case 0:
        this->getMainTimeline(0);
        break;

    case 1:
        this->getDirectTimeline(0);
        break;

    case 2:
        this->getActivityTimeline(0);
        break;

    case 3:
        this->getFavoritesTimeline(0);
        break;

    case 4:
        this->getMinorFeed();
        break;

    case 5:
        this->getContactList("following");
        break;

    case 6:
        this->getContactList("followers");
        break;

    case 7:
        this->getListsList();
        break;

    case 8:
        // Do nothing, so it takes longer for the final (default) step to arrive
        break;


    default:
        emit currentJobChanged(tr("Ready."));
        initialDataTimer->stop();

        qDebug() << "--------------------------------------";
        qDebug() << "-- All initial data loaded -----------";
        qDebug() << "--------------------------------------";
    }

    ++initialDataStep;
}





/*
 * Send a NOTE to the server
 *
 */
void PumpController::postNote(QMap<QString, QString> audienceMap,
                              QString postText)
{
    qDebug() << "PumpController::postNote()";

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST, PublishPostRequest);


    qDebug() << "Should be posting to:" << audienceMap;



    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "note");
    jsonVariantObject.insert("content", postText);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QList<QVariantList> audience = processAudience(audienceMap);
    jsonVariant.insert("to", audience.at(0));
    jsonVariant.insert("cc", audience.at(1));


    QJson::Serializer serializer;
    // bool ok;
    // QByteArray data = serializer.serialize(jsonVariant, &ok);

    // This way to make it work with QJSON 0.7.x
    QByteArray data = serializer.serialize(jsonVariant);


    qDebug() << "About to POST:" << data;

    nam.post(postRequest, data);
}


/*
 * Send an IMAGE to the server
 *
 * First, upload the file.
 * Then we get it's ID in a signal, and create the post itself
 *
 */
void PumpController::postImage(QMap<QString, QString> audienceMap,
                               QString postText, QString imageTitle,
                               QString imageFilename, QString contentType)
{
    qDebug() << "PumpController::postImage()";
    qDebug() << "Uploading" << imageFilename << "with title:" << imageTitle;

    // Store postText, imageTitle, and audienceMap, then upload
    this->currentImageTitle = imageTitle;
    this->currentImageDescription = postText;
    this->currentAudienceMap = audienceMap;


    this->uploadFile(imageFilename, contentType);
}


/*
 * Post Image, step 2: after getting the ID in the file upload request,
 * create the post itself
 *
 */
void PumpController::postImageStepTwo(QString id)
{
    qDebug() << "PumpController::postImageStepTwo() image ID:" << id;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST, PublishPostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "image");
    jsonVariantObject.insert("id", id);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QList<QVariantList> audience = processAudience(this->currentAudienceMap);
    jsonVariant.insert("to", audience.at(0));
    jsonVariant.insert("cc", audience.at(1));

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);


    qDebug() << "About to POST:" << data;

    nam.post(postRequest, data);
}


/*
 * Workaround for the non-title-non-description issue
 *
 * Update the image post with the title and the description
 *
 */
void PumpController::postImageStepThree(QString id)
{
    qDebug() << "PumpController::postImageStepThree() post ID:" << id;

    // Using the ID directly as URL
    QNetworkRequest postRequest = this->prepareRequest(id, QOAuth::PUT, UpdatePostRequest);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "update"); // "post" would suffice
    jsonVariant.insert("displayName", this->currentImageTitle);
    jsonVariant.insert("content", this->currentImageDescription);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "About to PUT:" << data;
    nam.put(postRequest, data);
}



/*
 * Like (favorite) a post, by its ID (URL)
 *
 */
void PumpController::likePost(QString postID, QString postType, bool like)
{
    qDebug() << "PumpController::likePost() liking post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest likeRequest = this->prepareRequest(url, QOAuth::POST, LikePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", like ? "favorite":"unfavorite"); // like or unlike
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "about to POST:" << data;

    nam.post(likeRequest, data);
}



void PumpController::addComment(QString comment, QString postID, QString postType)
{
    qDebug() << "PumpController::addComment() sending comment to this post:" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest commentRequest = this->prepareRequest(url, QOAuth::POST, CommentPostRequest);



    QVariantMap jsonVariantInReplyTo;
    jsonVariantInReplyTo.insert("id", postID);
    jsonVariantInReplyTo.insert("objectType", postType);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "comment");
    jsonVariantObject.insert("content", comment);
    jsonVariantObject.insert("inReplyTo", jsonVariantInReplyTo);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "about to POST:" << data;

    nam.post(commentRequest, data);
}




void PumpController::sharePost(QString postID, QString postType)
{
    qDebug() << "PumpController::sharePost() sharing post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest shareRequest = this->prepareRequest(url, QOAuth::POST, SharePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "share");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "about to POST:" << data;

    nam.post(shareRequest, data);
}



void PumpController::deletePost(QString postID, QString postType)
{
    qDebug() << "PumpController::deletePost() deleting post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest deleteRequest = this->prepareRequest(url, QOAuth::POST, DeletePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "delete");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "about to POST:" << data;

    nam.post(deleteRequest, data);
}




/*
 * Add a contact to the /following list with their webfinger address
 *
 * This will trigger a re-request of the contact list, upon receiving the HTTP 200
 * confirming the addition of the contact to /following
 *
 */
void PumpController::followContact(QString address)
{
    qDebug() << "PumpController::followContact()" << address;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest followRequest = this->prepareRequest(url, QOAuth::POST, FollowContactRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", "acct:" + address);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "follow");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "about to POST:" << data;

    nam.post(followRequest, data);
}



/*
 * Remove a contact from the /following list with their webfinger address
 *
 * This will trigger a re-request of the contact list, upon receiving the HTTP 200
 * confirming the removal of the contact from /following
 *
 */
void PumpController::unfollowContact(QString address)
{
    qDebug() << "PumpController::unfollowContact()" << address;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest unfollowRequest = this->prepareRequest(url, QOAuth::POST, UnfollowContactRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", "acct:" + address);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "stop-following");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    QByteArray data = serializer.serialize(jsonVariant);

    qDebug() << "about to POST:" << data;

    nam.post(unfollowRequest, data);
}
