/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "minorfeed.h"


MinorFeed::MinorFeed(PumpController::requestTypes minorFeedType,
                     PumpController *pumpController,
                     GlobalObject *globalObject,
                     FilterChecker *filterChecker,
                     QWidget *parent) : QFrame(parent)
{
    this->feedType = minorFeedType;
    this->pController = pumpController;
    this->globalObj = globalObject;
    this->fChecker = filterChecker;
    this->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    this->setContentsMargins(0, 0, 0, 0);

    // Layout for the items
    itemsLayout = new QVBoxLayout();
    itemsLayout->setContentsMargins(0, 0, 0, 0);
    itemsLayout->setSpacing(1); // Very small spacing
    itemsLayout->setAlignment(Qt::AlignTop);

    // Separator frame, to mark where new items end
    separatorFrame = new QFrame(this);
    separatorFrame->setFrameStyle(QFrame::HLine);
    separatorFrame->setMinimumHeight(28);
    separatorFrame->setContentsMargins(0, 8, 0, 8);
    separatorFrame->hide();

    // Button to get more items (older stuff)
    getMoreButton = new QPushButton(QIcon::fromTheme("list-add",
                                                     QIcon(":/images/list-add.png")),
                                    tr("Older Activities"));
    getMoreButton->setFlat(true);
    getMoreButton->setToolTip(tr("Get previous minor activities"));
    connect(getMoreButton, SIGNAL(clicked()),
            this, SLOT(getMoreActivities()));

    // Set disabled initially; will be enabled when contents are set
    getMoreButton->setDisabled(true);


    // Main layout
    mainLayout = new QVBoxLayout();
    mainLayout->setContentsMargins(1, 1, 1, 1);
    mainLayout->setAlignment(Qt::AlignTop);
    mainLayout->addLayout(itemsLayout);
    mainLayout->addWidget(getMoreButton);
    this->setLayout(mainLayout);



    // Demo activity, stating that there's nothing to show
    QVariantMap demoGenerator;
    demoGenerator.insert("displayName", "Dianara");

    QVariantMap demoActivityMap;
    demoActivityMap.insert("published", QDateTime::currentDateTimeUtc()
                                        .toString(Qt::ISODate));
    demoActivityMap.insert("generator", demoGenerator);
    demoActivityMap.insert("content",   tr("There are no activities to show yet."));

    ASActivity *demoActivity = new ASActivity(demoActivityMap, this);

    MinorFeedItem *demoFeedItem = new MinorFeedItem(demoActivity,
                                                    false, // Not highlighted by filter
                                                    this->pController,
                                                    this->globalObj);
    this->itemsLayout->addWidget(demoFeedItem);
    this->itemsInFeed.append(demoFeedItem);

    this->feedOffset = 0;
    this->newItemsCount = 0;

    // Remember what was the newest activity last time
    QSettings settings;
    settings.beginGroup("MinorFeedStates");
    switch (this->feedType)
    {
    case PumpController::MinorFeedMainRequest:
        this->previousNewestActivityId = settings
                                         .value("previousNewestItemIdMain")
                                         .toString();
        break;

    case PumpController::MinorFeedDirectRequest:
        this->previousNewestActivityId = settings
                                         .value("previousNewestItemIdDirect")
                                         .toString();
        break;

    case PumpController::MinorFeedActivityRequest:
        this->previousNewestActivityId = settings
                                         .value("previousNewestItemIdActivity")
                                         .toString();
        break;

    default:
        qDebug() << "MinorFeed created with wrong type:" << this->feedType;
    }
    settings.endGroup();


    // Sync all avatar's follow state when there are changes in the Following list
    connect(pController, SIGNAL(followingListChanged()),
            this, SLOT(updateAvatarFollowStates()));

    qDebug() << "MinorFeed created";
}


MinorFeed::~MinorFeed()
{
    QSettings settings;
    settings.beginGroup("MinorFeedStates");

    switch (this->feedType)
    {
    case PumpController::MinorFeedMainRequest:
        settings.setValue("previousNewestItemIdMain",
                          this->previousNewestActivityId);
        break;

    case PumpController::MinorFeedDirectRequest:
        settings.setValue("previousNewestItemIdDirect",
                          this->previousNewestActivityId);
        break;

    case PumpController::MinorFeedActivityRequest:
        settings.setValue("previousNewestItemIdActivity",
                          this->previousNewestActivityId);
        break;

    default:
        qDebug() << "MinorFeed type was wrong";
    }
    settings.endGroup();

    qDebug() << "MinorFeed destroyed; Type:" << this->feedType;
}



void MinorFeed::clearContents()
{
    qDebug() << "MinorFeed::clearContents()";
    foreach (MinorFeedItem *feedItem, itemsInFeed)
    {
        this->itemsLayout->removeWidget(feedItem);
        delete feedItem;
    }

    itemsInFeed.clear();

    itemsLayout->removeWidget(separatorFrame);
    separatorFrame->hide();
}


void MinorFeed::markAllAsRead()
{
    foreach (MinorFeedItem *feedItem, itemsInFeed)
    {
        feedItem->setItemAsNew(false);
    }
}

/*
 * Update fuzzy timestamps in all items
 *
 */
void MinorFeed::updateFuzzyTimestamps()
{
    foreach (MinorFeedItem *feedItem, itemsInFeed)
    {
        feedItem->setFuzzyTimeStamp();
    }
}


/*******************************************************************************/
/*******************************************************************************/
/********************************** SLOTS **************************************/
/*******************************************************************************/
/*******************************************************************************/


/*
 * Get the latest activities
 *
 */
void MinorFeed::updateFeed()
{
    this->feedOffset = 0;
    this->pController->getMinorFeed(this->feedType);
    //this->pController->getMinorFeedProper(this->prevLink);
}

/*
 * Get additional older activities
 *
 */
void MinorFeed::getMoreActivities()
{
    int feedJumpSize = 50; // TMP, until getMinorFeedProper() is used; FIXME
    switch (this->feedType)
    {
    case PumpController::MinorFeedMainRequest:
        feedJumpSize = 50;
        break;

    case PumpController::MinorFeedDirectRequest:
    case PumpController::MinorFeedActivityRequest:
        feedJumpSize = 20;
        break;

    default:
        break;
    }

    this->feedOffset += feedJumpSize;
    this->pController->getMinorFeed(this->feedType,
                                    this->feedOffset);
    //this->pController->getMinorFeedProper(this->nextLink);
}



void MinorFeed::setFeedContents(QVariantList activitiesList,
                                QString previous, QString next)
{
    this->prevLink = previous;
    this->nextLink = next;
    qDebug() << "MinorFeed links:" << this->prevLink << this->nextLink;

    this->getMoreButton->hide();

    if (feedOffset == 0) // Didn't ask for more, but a reload
    {
        this->clearContents();
        this->newItemsCount = 0;
        this->newHighlightedItemsCount = 0;
    }

    // Process event queue, so GUI can get updated
    qApp->processEvents(); // This also acts as "scroll to top", incidentally


    bool itemHighlightedByFilter;
    bool itemIsNew;    
    bool allNewItemsCounted = false;
    QString newestActivityId; // To store the activity ID for the newest item in the feed
                              // so we can know how many new items we receive next time


    foreach (QVariant activityVariant, activitiesList)
    {
        itemIsNew = false;

        ASActivity *activity = new ASActivity(activityVariant.toMap(), this);

        int filtered = fChecker->validateActivity(activity);

        // If there is no reason to filter out the item, add it to the feed
        if (filtered != FilterChecker::FilterOut)
        {
            // Determine which activities are new
            if (newestActivityId.isEmpty()) // Only first time, for newest item
            {
                if (this->feedOffset == 0)
                {
                    newestActivityId = activity->getId();
                }
                else
                {
                    newestActivityId = this->previousNewestActivityId;
                    allNewItemsCounted = true;
                }
            }

            if (!allNewItemsCounted)
            {
                if (activity->getId() == this->previousNewestActivityId)
                {
                    allNewItemsCounted = true;
                    if (newItemsCount > 0)
                    {
                        this->itemsLayout->addWidget(separatorFrame);
                        separatorFrame->show();
                    }
                }
                else
                {
                    // If activity is not ours, add it to the count
                    if (activity->author()->getId() != pController->currentUserId())
                    {
                        ++newItemsCount;
                        itemIsNew = true;
                    }
                }
            }


            if (filtered == FilterChecker::Highlight) // kinda TMP
            {
                itemHighlightedByFilter = true;
            }
            else
            {
                itemHighlightedByFilter = false;
            }

            MinorFeedItem *newFeedItem = new MinorFeedItem(activity,
                                                           itemHighlightedByFilter,
                                                           this->pController,
                                                           this->globalObj);
            newFeedItem->setItemAsNew(itemIsNew);
            if (itemIsNew)
            {
                connect(newFeedItem, SIGNAL(itemRead(bool)),
                        this, SLOT(decreaseNewItemsCount(bool)));

                if (newFeedItem->getItemHighlightType() != -1)
                {
                    ++this->newHighlightedItemsCount;
                }
            }
            this->itemsLayout->addWidget(newFeedItem);
            this->itemsInFeed.append(newFeedItem);
        }
        else
        {
            // Since the item is not added to the feed, we need to delete the activity
            delete activity;
        }
    }


    this->previousNewestActivityId = newestActivityId;

    qDebug() << "Meanwhile feed updated";
    if (this->feedOffset == 0) // not when getting more
    {
        emit newItemsCountChanged(newItemsCount, newHighlightedItemsCount);
        emit newItemsReceived(newItemsCount, newHighlightedItemsCount);
        qDebug() << "New items:" << newItemsCount
                 << "; Highlighted:" << newHighlightedItemsCount;
    }


    this->getMoreButton->show();
    this->getMoreButton->setEnabled(true);
}


void MinorFeed::decreaseNewItemsCount(bool wasHighlighted)
{
    --newItemsCount;
    if (wasHighlighted)
    {
        --newHighlightedItemsCount;
    }

    emit newItemsCountChanged(newItemsCount, newHighlightedItemsCount);
}


void MinorFeed::updateAvatarFollowStates()
{
    foreach (MinorFeedItem *feedItem, itemsInFeed)
    {
        feedItem->syncAvatarFollowState();
    }
}
