/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "timeline.h"

TimeLine::TimeLine(int timelineType,
                   PumpController *pumpController,
                   GlobalObject *globalObject,
                   FilterChecker *filterChecker,
                   QWidget *parent) :  QWidget(parent)
{
    this->timelineType = timelineType;
    this->pController = pumpController;
    this->globalObj = globalObject;
    this->fChecker = filterChecker;

    this->setMinimumSize(180, 180); // Ensure something's always visible



    // Simulated data for demo posts
    QVariantMap demoLocationData;
    demoLocationData.insert("displayName",  "Demoville");

    QVariantMap demoAuthorData;
    demoAuthorData.insert("displayName",    "Demo User");
    demoAuthorData.insert("id",             "demo@somepump.example");
    demoAuthorData.insert("location",       demoLocationData);
    demoAuthorData.insert("summary",        "I am not a real user");

    QVariantMap demoGeneratorData;
    demoGeneratorData.insert("displayName", "Dianara");

    QVariantMap demoObjectData;
    demoObjectData.insert("objectType",     "note");
    demoObjectData.insert("id",             "demo-post-id");

    // Show date/time when Dianara v1.2.5 was released
    demoObjectData.insert("published",      "2014-12-16T18:00:00Z");


    QSettings settings; // FIXME: kinda tmp, until posts have "unread" status, etc.
    settings.beginGroup("TimelineStates");

    // Demo post content depends on timeline type
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        demoObjectData.insert("displayName", tr("Welcome to Dianara"));
        demoObjectData.insert("content",
                              tr("Dianara is a <b>pump.io</b> client.")
                              + "<br>"

                              + tr("If you don't have a Pump account yet, you can get one "
                                   "at the following address, for instance:")
                              + "<br>"
                                "<a href=\"http://pump.io/tryit.html\">http://pump.io/tryit.html</a>"
                                "<br><br>"

                              + tr("Press <b>F1</b> if you want to open the Help window.")
                              + "<br><br>"

                              + tr("First, configure your account from the "
                                   "<b>Settings - Account</b> menu.")
                              + " "
                              + tr("After the process is done, your profile "
                                   "and timelines should update automatically.")
                              + "<br><br>"

                              + tr("Take a moment to look around the menus and "
                                   "the Configuration window.")
                              + "<br><br>"

                              + tr("You can also set your profile data and picture from "
                                   "the <b>Settings - Edit Profile</b> menu.")
                              + "<br><br>"

                              + tr("There are tooltips everywhere, so if you "
                                   "hover over a button or a text field with "
                                   "your mouse, you'll probably see some "
                                   "extra information.")
                              + "<br><br>"

                              + "<a href=\"http://jancoding.wordpress.com/dianara\">"
                              + tr("Dianara's blog") + "</a><br><br>"
                                "<a href=\"https://github.com/e14n/pump.io/wiki/User-Guide\">"
                              + tr("Pump.io User Guide")
                              + "</a>"
                              + "<br><br>");

        this->previousNewestPostId = settings.value("previousNewestPostIdMain").toString();
        break;


    case TimelineTypeDirect:
        demoObjectData.insert("displayName",  tr("Direct Messages Timeline"));
        demoObjectData.insert("content",      tr("Here, you'll see posts "
                                                 "specifically directed to you.")
                                              + "<br><br><br>");
        this->previousNewestPostId = settings.value("previousNewestPostIdDirect").toString();
        break;


    case TimelineTypeActivity:
        demoObjectData.insert("displayName", tr("Activity Timeline"));
        demoObjectData.insert("content",     tr("You'll see your own posts here.")
                                             + "<br><br><br>");
        this->previousNewestPostId = settings.value("previousNewestPostIdActivity").toString();
        break;


    case TimelineTypeFavorites:
        demoObjectData.insert("displayName", tr("Favorites Timeline"));
        demoObjectData.insert("content",     tr("Posts and comments you've liked.")
                                             + "<br><br><br>");
        this->previousNewestPostId = settings.value("previousNewestPostIdFavorites").toString();
        break;



    default:
        demoObjectData.insert("content", "<h2>Empty timeline</h2>");

    }
    settings.endGroup();


    QVariantMap demoPostData;
    demoPostData.insert("actor",          demoAuthorData);
    demoPostData.insert("generator",      demoGeneratorData);
    demoPostData.insert("object",         demoObjectData);
    demoPostData.insert("id",             "demo-activity-id");


    this->unreadPostsCount = 0;
    this->timelineOffset = 0;

    this->postsPerPage = 1;    // 1 at first, so initial page number calculation makes sense
    this->totalItemsCount = 1; // For total page count, too.

    this->minMaxHeightForPosts = 400;

    firstPageButton = new QPushButton(QIcon::fromTheme("go-first"),
                                      tr("Newest"));
    connect(firstPageButton, SIGNAL(clicked()),
            this, SLOT(goToFirstPage()));


    this->pageSelector = new PageSelector(this);
    connect(pageSelector, SIGNAL(pageJumpRequested(int)),
            this, SLOT(goToSpecificPage(int)));


    currentPageButton = new QPushButton();
    currentPageButton->setFlat(true);
    currentPageButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                     QSizePolicy::Preferred);
    connect(currentPageButton, SIGNAL(clicked()),
            this, SLOT(showPageSelector()));



    previousPageButton = new QPushButton(QIcon::fromTheme("go-previous"),
                                         tr("Newer"));
    connect(previousPageButton, SIGNAL(clicked()),
            this, SLOT(goToPreviousPage()));
    nextPageButton = new QPushButton(QIcon::fromTheme("go-next"),
                                     tr("Older"));
    connect(nextPageButton, SIGNAL(clicked()),
            this, SLOT(goToNextPage()));


    ///// Layout
    postsLayout = new QVBoxLayout();
    postsLayout->setContentsMargins(0, 0, 0, 0);
    postsLayout->setAlignment(Qt::AlignTop);

    bottomLayout = new QHBoxLayout();
    bottomLayout->addSpacing(2);
    bottomLayout->addWidget(firstPageButton,    2);
    bottomLayout->addSpacing(4);
    bottomLayout->addWidget(currentPageButton,  0);
    bottomLayout->addSpacing(4);
    bottomLayout->addWidget(previousPageButton, 2);
    bottomLayout->addWidget(nextPageButton,     2);
    bottomLayout->addSpacing(2);


    mainLayout = new QVBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addLayout(postsLayout, 1);
    mainLayout->addSpacing(2);              // 2 pixel separation
    mainLayout->addLayout(bottomLayout, 0);

    this->setLayout(mainLayout);


    ////////////////////////////////////// QActions for better keyboard control

    // Single step
    scrollUpAction = new QAction(this);
    scrollUpAction->setShortcut(QKeySequence("Ctrl+Up"));
    connect(scrollUpAction, SIGNAL(triggered()),
            this, SLOT(scrollUp()));
    this->addAction(scrollUpAction);

    scrollDownAction = new QAction(this);
    scrollDownAction->setShortcut(QKeySequence("Ctrl+Down"));
    connect(scrollDownAction, SIGNAL(triggered()),
            this, SLOT(scrollDown()));
    this->addAction(scrollDownAction);

    // Pages
    scrollPageUpAction = new QAction(this);
    scrollPageUpAction->setShortcut(QKeySequence("Ctrl+PgUp"));
    connect(scrollPageUpAction, SIGNAL(triggered()),
            this, SLOT(scrollPageUp()));
    this->addAction(scrollPageUpAction);

    scrollPageDownAction = new QAction(this);
    scrollPageDownAction->setShortcut(QKeySequence("Ctrl+PgDown"));
    connect(scrollPageDownAction, SIGNAL(triggered()),
            this, SLOT(scrollPageDown()));
    this->addAction(scrollPageDownAction);

    // Top / Bottom
    scrollTopAction = new QAction(this);
    scrollTopAction->setShortcut(QKeySequence("Ctrl+Home"));
    connect(scrollTopAction, SIGNAL(triggered()),
            this, SLOT(scrollToTop()));
    this->addAction(scrollTopAction);

    scrollBottomAction = new QAction(this);
    scrollBottomAction->setShortcut(QKeySequence("Ctrl+End"));
    connect(scrollBottomAction, SIGNAL(triggered()),
            this, SLOT(scrollToBottom()));
    this->addAction(scrollBottomAction);


    // Previous/Next page in timeline
    previousPageAction = new QAction(this);
    previousPageAction->setShortcut(QKeySequence("Ctrl+Left"));
    connect(previousPageAction, SIGNAL(triggered()),
            previousPageButton, SLOT(click()));
    this->addAction(previousPageAction);

    nextPageAction = new QAction(this);
    nextPageAction->setShortcut(QKeySequence("Ctrl+Right"));
    connect(nextPageAction, SIGNAL(triggered()),
            nextPageButton, SLOT(click()));
    this->addAction(nextPageAction);


    // Add the default "demo" post
    ASActivity *demoActivity = new ASActivity(demoPostData, this);
    Post *demoPost = new Post(demoActivity,
                              false, // Not highlighted
                              false, // Not standalone
                              pController,
                              globalObj,
                              this);
    demoPost->setMinMaxHeight(this->minMaxHeightForPosts);

    postsInTimeline.append(demoPost);
    postsLayout->addWidget(demoPost);


    this->updateCurrentPageNumber();

    // Sync avatar's follow state for every post when there are changes in the Following list
    connect(pController, SIGNAL(followingListChanged()),
            this, SLOT(updateAvatarFollowStates()));

    qDebug() << "TimeLine created";
}


/*
 * Destructor stores timeline states in the settings
 *
 */
TimeLine::~TimeLine()
{
    QSettings settings;
    settings.beginGroup("TimelineStates");

    switch (timelineType)
    {
    case TimelineTypeMain:
        settings.setValue("previousNewestPostIdMain",
                          this->previousNewestPostId);
        break;

    case TimelineTypeDirect:
        settings.setValue("previousNewestPostIdDirect",
                          this->previousNewestPostId);
        break;

    case TimelineTypeActivity:
        settings.setValue("previousNewestPostIdActivity",
                          this->previousNewestPostId);
        break;

    case TimelineTypeFavorites:
        settings.setValue("previousNewestPostIdFavorites",
                          this->previousNewestPostId);
        break;
    }
    settings.endGroup();


    qDebug() << "TimeLine destroyed; Type:" << this->timelineType;
}



/*
 * Remove all widgets (Post *) from the timeline
 *
 */
void TimeLine::clearTimeLineContents()
{
    foreach (Post *oldPost, postsInTimeline)
    {
        this->mainLayout->removeWidget(oldPost);
        delete oldPost;
    }
    this->postsInTimeline.clear();

    qApp->processEvents(); // So GUI gets updated
}



void TimeLine::requestTimelinePage()
{
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        this->pController->getMainTimeline(this->timelineOffset);
        break;

    case TimelineTypeDirect:
        this->pController->getDirectTimeline(this->timelineOffset);
        break;

    case TimelineTypeActivity:
        this->pController->getActivityTimeline(this->timelineOffset);
        break;

    case TimelineTypeFavorites:
        this->pController->getFavoritesTimeline(this->timelineOffset);
        break;
    }
}


int TimeLine::getCurrentPage()
{
    if (this->postsPerPage == 0)
    {
        this->postsPerPage = 1;
    }

    return (this->timelineOffset / this->postsPerPage) + 1;
}

int TimeLine::getTotalPages()
{
    if (this->postsPerPage == 0)
    {
        this->postsPerPage = 1;
    }

    return qCeil(this->totalItemsCount / (float)this->postsPerPage);
}


/*
 *  Update the button at the bottom of the page, indicating current "page"
 *
 */
void TimeLine::updateCurrentPageNumber()
{
    int currentPage = this->getCurrentPage();
    int totalPages = this->getTotalPages();

    this->currentPageButton->setText(QString("%1 / %2")
                                     .arg(currentPage)
                                     .arg(totalPages));
    // The shortcut needs to be set each time the text is changed
    currentPageButton->setShortcut(QKeySequence("Ctrl+G"));

    this->currentPageButton->setToolTip(tr("Page %1 of %2.")
                                        .arg(currentPage)
                                        .arg(totalPages)
                                        + "<br>"
                                        + tr("Showing %1 posts per page.")
                                          .arg(this->postsPerPage)
                                        + "<br>"
                                        + tr("%1 posts in total.")
                                          .arg(this->totalItemsCount)
                                        + "<hr>"
                                          "<b><i>"
                                        + tr("Click here or press Control+G to "
                                             "jump to a specific page")
                                        + "</i></b>");

    if (currentPage == totalPages)
    {
        this->nextPageButton->setDisabled(true);
    }
    else
    {
        this->nextPageButton->setEnabled(true);
    }
}



void TimeLine::setMinMaxHeightForPosts(int newMinMaxHeightForPosts)
{
    this->minMaxHeightForPosts = newMinMaxHeightForPosts;
}


/*
 * Resize all posts in timeline
 *
 */
void TimeLine::resizePosts()
{
    foreach (Post *post, postsInTimeline)
    {
        post->setMinMaxHeight(this->minMaxHeightForPosts);

        post->resetResizesCount();

        // Force a Post() resizeEvent, which will call
        // setPostContents() and setPostHeight()
        post->resize(post->width() - 1,
                     post->height() - 1);
    }
}

void TimeLine::markPostsAsRead()
{
    foreach (Post *post, postsInTimeline)
    {
        post->setPostAsRead();
    }
}


void TimeLine::updateFuzzyTimestamps()
{
    foreach (Post *post, postsInTimeline)
    {
        post->setFuzzyTimestamps();
    }
}



/*********************************************************/
/*********************************************************/
/************************ SLOTS **************************/
/*********************************************************/
/*********************************************************/



void TimeLine::setTimeLineContents(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink,
                                   int totalItems)
{
    qDebug() << "TimeLine::setTimeLineContents()";


    // Remove all previous posts in timeline
    qDebug() << "Removing previous posts from timeline";
    this->clearTimeLineContents();

    // Ask mainWindow to scroll the QScrollArea containing the timeline to the top
    emit scrollTo(QAbstractSlider::SliderToMinimum);

    // Hide the widget while it reloads; helps performance a lot
    this->hide();

    this->postsPerPage = postsPerPage;
    this->totalItemsCount = totalItems;

    int newPostCount = 0;
    bool allNewPostsCounted = false;
    QString newestPostId; // Here we'll store the postID for the first (newest) post in the timeline
                          // With it, we can know how many new posts (if any) we receive next time
    newHighlightedPostsCount = 0;


    // Fill timeline with new contents
    foreach (QVariant singlePost, postList)
    {
        if (singlePost.type() == QVariant::Map)
        {
            this->previousPageLink = previousLink;   // Useless at the moment
            this->nextPageLink = nextLink;
            qDebug() << "Prev/Next links:";
            qDebug() << previousPageLink;
            qDebug() << nextPageLink;

            bool postIsNew = false;

            QVariantMap activityMap;

            // Since "favorites" is a collection of objects, not activities,
            // we need to put "Favorites" posts into fake activities
            if (this->timelineType != TimelineTypeFavorites)
            {
                // Data is already an activity
                activityMap = singlePost.toMap();
            }
            else
            {
                // Put object into the empty/fake VariantMap for the activity
                activityMap.insert("object", singlePost.toMap());
                activityMap.insert("actor",  singlePost.toMap().value("author").toMap());
                activityMap.insert("id",     singlePost.toMap().value("id").toString());
            }

            ASActivity *activity = new ASActivity(activityMap, this);

            // See if it's deleted
            QString postDeletedTime = activity->object()->getDeletedTime();

            // See if we have to filter it out (or highlight it)
            int filtered = this->fChecker->validateActivity(activity);

            if (newestPostId.isEmpty()) // only first time, for newest post
            {
                if (this->timelineOffset == 0)
                {
                    newestPostId = activity->getId();
                }
                else
                {
                    newestPostId = this->previousNewestPostId;
                    allNewPostsCounted = true;
                }
            }


            if (!allNewPostsCounted)
            {
                if (activity->getId() == this->previousNewestPostId)
                {
                    allNewPostsCounted = true;
                }
                else
                {
                    // If post is NOT deleted or filtered, and not ours, add it to the count
                    if (postDeletedTime.isEmpty()
                        && filtered != FilterChecker::FilterOut
                        && activity->author()->getId() != pController->currentUserId()
                        && activity->object()->author()->getId() != pController->currentUserId())
                    {
                        ++newPostCount;

                        // Mark current post as new
                        postIsNew = true;
                    }
                }
            }



            // If post was NOT deleted, and not filtered out
            if (postDeletedTime.isEmpty() && filtered != FilterChecker::FilterOut)
            {
                bool highlightedByFilter = false;
                if (filtered == FilterChecker::Highlight)
                {
                    highlightedByFilter = true;
                }

                Post *newPost = new Post(activity,
                                         highlightedByFilter,
                                         false,  // NOT standalone
                                         pController,
                                         globalObj,
                                         this);
                if (postIsNew)
                {
                    newPost->setPostAsNew();
                    connect(newPost, SIGNAL(postRead(bool)),
                            this, SLOT(decreaseUnreadPostsCount(bool)));

                    if (newPost->getHighlightType() != Post::NoHighlight)
                    {
                        ++this->newHighlightedPostsCount;
                    }

                }
                this->postsInTimeline.append(newPost);
                this->postsLayout->addWidget(newPost);

                // FIXME: this signal should go directly via a global shared object, instead
                connect(newPost, SIGNAL(commentingOnPost(QWidget*)),
                        this, SIGNAL(commentingOnPost(QWidget*)));
            }
            else
            {
                // If there's a "deleted" key, or it's filtered out, ignore this post
                qDebug() << "This post has been filtered out, or was deleted on"
                         << postDeletedTime << " / Not adding.";
                qDebug() << "Filter action:" << filtered
                         << "(0=filter out; 1=highlight, 999=no filtering)";

                // FIXME: show something in place of the deleted post

                delete activity;
            }
        }
        else  // singlePost.type() is not a QVariant::Map
        {
            qDebug() << "Expected a Map, got something else";
            qDebug() << postList;
        }

        qApp->processEvents(); // Avoid GUI freeze

    } // end foreach



    this->previousNewestPostId = newestPostId;
    this->unreadPostsCount = newPostCount;
    qDebug() << "New posts:" << newPostCount << "; Newest post ID:" << previousNewestPostId
             << "\nHighlighted posts:" << newHighlightedPostsCount
             << "\nTotal posts:" << totalItemsCount;


    this->updateCurrentPageNumber();
    this->resizePosts();

    emit timelineRendered(this->timelineType,
                          unreadPostsCount,
                          newHighlightedPostsCount,
                          totalItemsCount);

    // Show timeline again, since everything is added and drawn
    this->show();

    qDebug() << "setTimeLineContents() /END";
}





/*
 * Add the full list of likes to a post
 *
 */
void TimeLine::setLikesInPost(QVariantList likesList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setLikesInPost()";

    QString originatingPostCleanUrl = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanUrl;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->likesURL() << "==" << originatingPostCleanUrl;

        if (post->likesURL() == originatingPostCleanUrl)
        {
            qDebug() << "Found originating Post; setting likes on it...";
            post->setLikes(likesList);

            break;
        }
    }
}


/*
 * Add the full list of comments to a post
 *
 */
void TimeLine::setCommentsInPost(QVariantList commentsList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setCommentsInPost()";

    QString originatingPostCleanURL = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanURL;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->commentsURL() << "==" << originatingPostCleanURL;

        if (post->commentsURL() == originatingPostCleanURL)
        {
            qDebug() << "Found originating Post; setting comments on it...";
            post->setComments(commentsList);

            // break;
            /* Don't break, so comments get set in copies of the post too,
               like if JohnDoe posted something and JaneDoe shared it soon
               after, so both the original post and its shared copy are visible
               in the timeline. */
        }
    }
}





void TimeLine::goToFirstPage()
{
    qDebug() << "TimeLine::goToFirstPage()";

    this->timelineOffset = 0;
    this->requestTimelinePage();

    /// new method!
    //pController->getTimeline(this->timelineType, this->previousPageLink);
}



void TimeLine::goToPreviousPage()
{
    qDebug() << "TimeLine::goToPreviousPage()";

    this->timelineOffset -= this->postsPerPage;
    if (timelineOffset < 0)
    {
        timelineOffset = 0;
    }

    this->requestTimelinePage();
}



void TimeLine::goToNextPage()
{
    qDebug() << "TimeLine::goToNextPage()";

    this->timelineOffset += this->postsPerPage;

    this->requestTimelinePage();
}


void TimeLine::goToSpecificPage(int pageNumber)
{
    qDebug() << "TimeLine::goToSpecificPage(): " << pageNumber;

    this->timelineOffset = (pageNumber - 1) * this->postsPerPage;
    this->requestTimelinePage();
}


void TimeLine::showPageSelector()
{
    this->pageSelector->showForPage(this->getCurrentPage(),
                                    this->getTotalPages());
}



void TimeLine::scrollUp()
{
    emit scrollTo(QAbstractSlider::SliderSingleStepSub);
}

void TimeLine::scrollDown()
{
    emit scrollTo(QAbstractSlider::SliderSingleStepAdd);
}

void TimeLine::scrollPageUp()
{
    emit scrollTo(QAbstractSlider::SliderPageStepSub);
}

void TimeLine::scrollPageDown()
{
    emit scrollTo(QAbstractSlider::SliderPageStepAdd);
}

void TimeLine::scrollToTop()
{
    emit scrollTo(QAbstractSlider::SliderToMinimum);
}

void TimeLine::scrollToBottom()
{
    emit scrollTo(QAbstractSlider::SliderToMaximum);
}



/*
 * Decrease internal counter of unread posts (by 1), and inform
 * the parent window, so it can update its tab titles
 *
 */
void TimeLine::decreaseUnreadPostsCount(bool wasHighlighted)
{
    --unreadPostsCount;

    if (wasHighlighted)
    {
        --newHighlightedPostsCount;
    }

    emit unreadPostsCountChanged(this->timelineType,
                                 this->unreadPostsCount,
                                 this->newHighlightedPostsCount,
                                 this->totalItemsCount);
}

void TimeLine::updateAvatarFollowStates()
{
    foreach (Post *post, postsInTimeline)
    {
        post->syncAvatarFollowState();
    }
}
