//  $Id: dinifti.cc,v 1.31 2007/05/22 14:39:25 valerio Exp $

#define VERSION "2.27"

//****************************************************************************
//
// Modification History (most recent first)
// mm/dd/yy  Who  What
//
// 10/04/06  Add DiniftiOptions class and verbose mode
//           (contributed by Michael Hanke)
// 08/15/06  VPL  Fix bugs in output
//                Make software GPL
//                Format --help and --version to be in standard form
// 03/22/05  VPL  Allow processing of multiple series
//                Add NYU License agreement
// 01/24/05  VPL  
//
//****************************************************************************

#include <dirent.h>
#include <sstream>
#include <algorithm>
#include "dinifti.h"
#include "niftiout.h"



//****************************************************************************
//
// Purpose: Comaprison of image entries
//   
// Parameters: DICOMImage lhs - left-hand side item
//             DICOMImage rhs - left-hand side item
//   
// Returns: 0 if rhs >= lhs
//   
// Notes: Adapted from n2n2.c written by L&R Fleisher
//
//****************************************************************************

static int CompareImages(const DICOMImage &lhs, const DICOMImage &rhs)
{
	float lhsValue = lhs.SagPos() * lhs.SagNorm() + lhs.CorPos() * lhs.CorNorm() + lhs.TraPos() * lhs.TraNorm();
	
	float rhsValue = rhs.SagPos() * rhs.SagNorm() + rhs.CorPos() * rhs.CorNorm() + rhs.TraPos() * rhs.TraNorm();

	int retValue = 0;
	
	// If position the same, use image number to sort

	if (fabs(lhsValue - rhsValue) > 0.000001)
		retValue = lhsValue < rhsValue;
	else
		retValue = lhs.ImageNumber() < rhs.ImageNumber();
	
	return retValue;
}

//****************************************************************************
//
// Purpose: Compose file name from pieces and open for writing
//   
// Parameters: char *&fname           - where we store the name
//             const char  *name      - Root of output file
//             OUTPUT_TYPE form       - NIfTI output format
//             bool        compressed - write compressed
//             bool        header     - if true (default) then headr file, else
//                                      image file
//   
// Returns: FILE *fp - pointer to open FILE
//   
// Notes: 
//
//****************************************************************************

static void SetFileName(char *&fname, const char *name, OUTPUT_TYPE form, bool compressed, bool header = true)
{
	ostringstream outFileName(name, ostringstream::app);
	if (header)
		switch( form )
		{
		    case NIfTI_SINGLE:
				outFileName << ".nii";
				break;
		    case NIfTI_ASCII:
				outFileName << ".nia";
				break;
		    case ANALYZE:
		    case NIfTI_DUAL:
				outFileName << ".hdr";
				break;
		}
	else
		outFileName << ".img";

	if (compressed)
		outFileName << ".gz";

	fname = new char [outFileName.str().length() + 1];
	strcpy(fname, outFileName.str().c_str());
}

//****************************************************************************
//
// Purpose: Display usage and leave
//   
// Parameters: char **argv - standard arguments
//   
// Returns: NONE
//   
// Notes: 
//
//****************************************************************************

void ShowHelp(char **argv)
{
	cout << "'dinifti' converts DICOM image file to NIfTI format" << endl << endl
		 << "Usage: " << argv[0] << " [OPTION] <DICOM input> <NIfTI output>" << endl << endl
		 << "Options:" << endl
		 << " -g                   compressed output" << endl
		 << " -f <output format>   'a2' - ANALYZE 7.5 Dual file" << endl
		 << "                      'n2' - NIfTI-2 Dual file" << endl 
		 << "                      'n1' - NIfTI-1 Single file **Default**" << endl 
		 << " -d                   append series description to output file name(s)" << endl
		 << " -s #                 number of slices per volume" << endl
		 << " -v --verbose    enable verbose status output" << endl
		 << " -n --noact      do all processing, but do not write files" << endl << endl
		 << " -h --help       print this help and exit" << endl
		 << " -V --version    print version number and exit" << endl << endl
		 << "I/O Options" << endl
		 << "<DICOM input> can be single file, list of files or directory" << endl
		 << "<NIfTI output> can be single file or directory" << endl;
	exit(0);
}

//****************************************************************************
//
// Purpose: 
//   
// Parameters: name of input file(s) or input directory
//             last parameter name of output directory (create if needed)
//   
// Returns: 
//   
// Notes: 
//
//****************************************************************************

int main(int argc, char **argv)
{
	if (argc < 3)
	{
		if ( (argc > 1) && ((strcmp(argv[1], "--version") == 0) || (strcmp(argv[1], "-V") == 0)) )
		{
			cout << "dinifti " << VERSION << endl;
			exit(0);
		}

		ShowHelp(argv);
	}
	
	// Last parameter is output file, get that out of the way

	int outNameIndex = --argc;

	DiniftiOptions opts;
	
	int parseArg = 1;
	while (parseArg < argc)
	{
		if (argv[parseArg][0] != '-')
			break;
		
		if (! strcmp(argv[parseArg], "-f") )
		{
			++parseArg;
			if (! strcmp(argv[parseArg], "a2") )
				opts.niftiFormat = ANALYZE;
			else if (! strcmp(argv[parseArg], "n1") )
				opts.niftiFormat = NIfTI_SINGLE;
			else if (! strcmp(argv[parseArg], "n2") )
				opts.niftiFormat = NIfTI_DUAL;
			else
			{
				cerr << "\t**** Error: format " << argv[2] << " not supported." << endl << endl;
				ShowHelp(argv);
			}
		
		}
		else if (! strcmp(argv[parseArg], "-g") )
		{
			opts.compressed = true;
		}
		else if (! strcmp(argv[parseArg], "-d") )
		{
			opts.useSerDesc = true;
		}
		else if ( (strcmp(argv[parseArg], "-v") == 0) || (strcmp(argv[parseArg], "--verbose") == 0) )
		{
			opts.verbose = true;
		}
		else if ( (strcmp(argv[parseArg], "-n") == 0) || (strcmp(argv[parseArg], "--noact") == 0) )
		{
			opts.noact = true;
		}
		else if (! strcmp(argv[parseArg], "-s") )
		{
			++parseArg;
			opts.numSlices = atoi(argv[parseArg]);
		}
		else
			ShowHelp(argv);

		++parseArg;
	}

	if (parseArg == argc)
		ShowHelp(argv);
	
	vector<string> dicomFiles;
	
	// If only two file parameters, it might be a directory (try opening it)

	DIR *dirStr = NULL; 
	
	if ((argc == parseArg + 1) && ( dirStr = opendir(argv[parseArg])))
	{
		// If this is a directory, find all DICOM files there

		dirent *nextFile = NULL;
		while (nextFile = readdir(dirStr))
		{
			// Avoid hidden files
			if (nextFile->d_name[0] != '.')
				dicomFiles.push_back(string(argv[parseArg]) + string("/") + string(nextFile->d_name));
		}
	}
	else
	{
		while (--argc >= parseArg)
			dicomFiles.push_back(string(argv[argc]));
	}

	// Try to open DICOM file(s) and process one at a time and store in lists
	// then sort according to anatomical position

	// We create separate lists for each time point
	// If this is mosaic, each list will only contain one entry
	// The key is the acquisition number

	SERIESMAP seriesMap;

	StoreImageFiles(dicomFiles, seriesMap, argv[outNameIndex], opts);
	

	// Turn offf excessive nifti messages

	nifti_set_debug_level(0);
	
	// Process each series, one at a time

	for (SERIESMAP::iterator series = seriesMap.begin(); series != seriesMap.end(); series++)
		ProcessSeries(series->second, opts);

	exit(0);
}

//****************************************************************************
//
// Purpose: Create the series map from all the file's we've read
//   
// Parameters: vector<string> &dicomFiles - the list
//             SERIESMAP &seriesMap       - where to store the map
//             char *outName              - where it goes
//
//             in DiniftiOptions class:
//             bool useSerDesc            - append series description to name ?
//             int numSlices              - number of slices per volume
//   
// Returns: NONE
//   
// Notes: Each series is identified by their series ID, but in special cases
//        (anatomical images), the same ID is assigned to images that
//        effectively belong to separate series. In this case an equal number of
//        images is assigned to each group and we create new virtual series with
//        the series ID appended with the group number.
//
//        If numSlices not equal 0, use it to compute current acquisition number.
//
//****************************************************************************

void StoreImageFiles(vector<string> &dicomFiles, SERIESMAP &seriesMap, char *outName,
					 const DiniftiOptions& opts)
{
	// Figure out if the outName is a directory or the name of a file

	const bool outDirectory = (opendir(outName) != NULL);
	int relSeriesNumber = 0;

	if (opts.verbose)
	{
		cout << "Reading " << dicomFiles.size() << " DICOM files." << endl;
	}

	while (dicomFiles.size() > 0)
	{
		string dcmFile(dicomFiles.back());
		dicomFiles.pop_back();

		DICOMImage *dcmImg = new DICOMImage(dcmFile.c_str());
		if (*dcmImg)
		{
			SeriesInfo si = seriesMap[dcmImg->SeriesID()];
			
			// If no entry for this list exists yet, create it

			int imgNum = dcmImg->ImageNumber();

			// The acquisition number can come directly from the header or
			// it can be computed from the header (if numSlices > 1) or
			// it can be computed from the parameter given by the user
			
			int acqNum = 0;
			if ( opts.numSlices > 0 )
			{
				acqNum = ((imgNum - 1) / opts.numSlices) + 1;
			}
			else if ( !dcmImg->Mosaic() && (dcmImg->NumSlices() > 1) )
			{
				acqNum = ((imgNum - 1) / dcmImg->NumSlices()) + 1;
			}
			else
				acqNum = dcmImg->ACQNumber();

			// Try to determine if the acquisition number makes sense

			
			IMAGELIST *timePoint = si.imageMap[acqNum];
			if (timePoint == NULL)
			{
				si.imageMap[acqNum] = timePoint = new IMAGELIST;

				// If using series description, use that as the output name
				// else if outName is not a directory and more than one image,
				// create unique output name
				
				si.outName = outName;
				if (outDirectory)
				{
					if ( opts.useSerDesc )
					{
						si.outName += string("/") + dcmImg->SeriesDescription();
					}
					else
					{
						if (outName[strlen(outName)-1] != '/')
							si.outName += string("/");
						size_type begName = dcmImg->ImagePath().find_last_of('/');
						size_type endName = dcmImg->ImagePath().find_last_of('.');

						// If the extension is a number, it probably means something

						const int nameLen = dcmImg->ImagePath().length();
						for (int extC = endName+1; extC < nameLen; ++extC)
						{
							char c = dcmImg->ImagePath()[extC];
							if (! isdigit(c) ) break;
							if (extC == nameLen-1) endName = nameLen;
						}
						endName -= begName;
						si.outName += dcmImg->ImagePath().substr(++begName, --endName);
					}
				}
				else if ( opts.useSerDesc )
				{
					si.outName += string("+") + dcmImg->SeriesDescription();
				}
				else if ( seriesMap.size() > 1 )
				{
					ostringstream newFileName(si.outName, ostringstream::app);
					newFileName << "+" << ++relSeriesNumber;
					si.outName = newFileName.str();
				}
			}
			
			timePoint->push_back(*dcmImg);
			
			si.mosaic = dcmImg->Mosaic();
			si.numSlices = dcmImg->NumSlices();
			if (imgNum > si.maxImgNum)
				si.maxImgNum = imgNum;
			if ((imgNum < si.minImgNum) || (si.minImgNum == 0))
				si.minImgNum = imgNum;
			if (acqNum > si.maxAcqNum)
				si.maxAcqNum = acqNum;

			seriesMap[dcmImg->SeriesID()] = si;
		}
		else
			delete dcmImg;
	}

	SERIESMAP newSeriesMap;
	
	// Go through the series and see if any are a special case

	// use this list to store series outNames to detect possible duplicates
	list<string> series_ids;

	// do not increment the iterator here as this has to be handled conditionally at the end
	for (SERIESMAP::iterator series = seriesMap.begin(); series != seriesMap.end(); )
	{
		// For each series, check the number of groups (first image in first
		// time point is sufficient)

		SeriesInfo si = series->second;
		IMAGEMAP fullImageMap = si.imageMap;
		IMAGEMAP::iterator fimIter = fullImageMap.begin();
		IMAGELIST *firstTimePoint = fimIter->second;
		DICOMImage firstImg = firstTimePoint->front();
		const int numGroups = firstImg.NumGroups();
		
		// Try to guess if everything's allright with number of time points
		if (opts.verbose)
		{
			cout << "Found new image series. Using output destination '" << si.outName << "'." << endl;
		}

		// check for duplicate series outNames
		int dupes = count( series_ids.begin(), series_ids.end(), si.outName );

		// if duplicate series name
		if (dupes)
		{
			// store the old for a useful warning message
			string old = si.outName;

			// appen '+' until we have a unique name
			while (dupes)
			{
				si.outName.append("+");
				dupes = count( series_ids.begin(), series_ids.end(), si.outName );
			}
			// issue warning to notify the name change
			cerr << endl << "\t^G^G**** WARNING: Duplicate image series name '" <<
				old << "'" << endl
				<< "\t**** New image series name is '" << si.outName << "'." << endl;

			// important: reassign the modified series info (lost otherwise)
			series->second = si;
		}

		// store the current image series name
		series_ids.push_back(si.outName);

		if ( (opts.numSlices == 0) && (si.numSlices == 1) && (si.maxAcqNum == 1) && (si.maxImgNum > si.minImgNum) )
		{
			cerr << endl << "\t**** StoreImageFiles: Warning: number of slices per volume = 1" << endl
				 << "\t**** and number of time points = 1." << endl
				 << "\t**** If incorrect, run program with \"-s\" option." << endl;
		}
		
		if ( numGroups > 1 )
		{
			// We need to create "virtual" series and split the images among them

			string origSeriesID = series->first;
			const int numImgGroup = (si.maxImgNum - si.minImgNum + 1) / (numGroups * si.maxAcqNum);

			// Hold the names in a map for easy retrieval

			map<int, string> newSeriesNames;
			for (int group = 1; group <= numGroups; ++group)
			{
				ostringstream newName;
				newName << origSeriesID << "-" << group;
				newSeriesNames[group] = newName.str();
			}
			
			for (int acqNum = 1; acqNum <= si.maxAcqNum; ++acqNum)
			{
				// Grab the full list of images for this time point
				// And split them up among the groups
				
				IMAGELIST *timePoint = fullImageMap[acqNum];
				for (IMAGELIST::iterator img = timePoint->begin(); img != timePoint->end(); ++img)
				{
					// We know the acqisition number, figure out which group it
					// belongs to.

					int group = (img->ImageNumber() - si.minImgNum) / numImgGroup + 1;
					SeriesInfo newSeriesInfo = newSeriesMap[newSeriesNames[group]];
					IMAGELIST *newTimePoint = newSeriesInfo.imageMap[acqNum];
					if (newTimePoint == NULL)
					{
						newSeriesInfo.imageMap[acqNum] = newTimePoint = new IMAGELIST;
						ostringstream newName;
						newName << si.outName << "-" << group;
						newSeriesInfo.outName = newName.str();
						newSeriesInfo.mosaic = si.mosaic;
						newSeriesInfo.minImgNum = (group - 1) * numImgGroup + si.minImgNum;
						newSeriesInfo.maxImgNum = group * numImgGroup + si.minImgNum - 1;
						newSeriesInfo.maxAcqNum = si.maxAcqNum;
					}
					
					newTimePoint->push_back(*img);
					newSeriesMap[newSeriesNames[group]] = newSeriesInfo;
				}
			}

			// Now we can get rid of the old entries

			// FIRST increment the iterator THEN erase the entry
			// otherwise the iterator becomes invalid
			seriesMap.erase(series++);
		}
		else
			// if not incremented above do here
			++series;
	}

	// Move the added series to permanent storage

	seriesMap.insert(newSeriesMap.begin(), newSeriesMap.end());
	
	// If not mosaic sort and do some post-processing

	for (SERIESMAP::iterator series = seriesMap.begin(); series != seriesMap.end(); series++)
	{
		SeriesInfo si = series->second;
		if (!si.mosaic)
		{
			const int numSlices = (si.maxImgNum - si.minImgNum + 1) / si.maxAcqNum;
			
			for (int acqNum = 1; acqNum <= si.maxAcqNum; ++acqNum)
			{
				IMAGELIST *timePoint = si.imageMap[acqNum];
				timePoint->sort(CompareImages);

				// Get values from first and second image and compute directions
				// If only one item, don't bother with direction, just fix image number

				if (timePoint->size() > 1)
				{
					IMAGELIST::iterator image = timePoint->begin();
					float x1 = image->SagPos();
					float y1 = image->CorPos();
					float z1 = image->TraPos();

					// Get info from second image in same acquisition

					image++;
					float x2 = image->SagPos();
					float y2 = image->CorPos();
					float z2 = image->TraPos();

					bool sagDir = (x2 >= x1);
					bool corDir = (y2 >= y1);
					bool traDir = (z2 >= z1);
		
					for (image = timePoint->begin(); image != timePoint->end(); image++)
						image->AdjustSet(sagDir, corDir, traDir, numSlices, si.minImgNum-1);
				}
				else
					for (IMAGELIST::iterator image = timePoint->begin(); image != timePoint->end(); image++)
						image->AdjustSet(numSlices, si.minImgNum-1);
			}
		}
	}
	
}

//****************************************************************************
//
// Purpose: Process a series
//   
// Parameters: SeriesInfo &seriesInfo  - the series
//
//             in DiniftiOptions class:
//             OUTPUT_TYPE niftiFormat - how the output is written
//             bool compressed         - compressed output format
//   
// Returns: 
//   
// Notes: The name will be taken from the first file in the series
//
//****************************************************************************

void ProcessSeries(SeriesInfo &seriesInfo, const DiniftiOptions& opts)
{
	// Create NIfTI header with default values
	
	if (opts.verbose)
	{
		cout << "Processing image series ('" << seriesInfo.outName << "')." << endl;
	}

	nifti_image niftiHdr;
	if (! NIfTICreateHeader(niftiHdr, seriesInfo.imageMap))
		exit(1);
	string description("dinifti v");
	description += VERSION;
	switch ( opts.niftiFormat )
	{
		case ANALYZE:
			description += " ANALYZE 7.5 file format";
			break;
	    case NIfTI_SINGLE:
			description += " NIfTI-1 Single file format";
			break;
	    case NIfTI_DUAL:
			description += " NIfTI-1 Dual file format";
			break;
	}
	
	strcpy(niftiHdr.descrip, description.c_str());
	niftiHdr.nifti_type = (int)opts.niftiFormat;

	// Just one coil mode

	niftiHdr.nvox = niftiHdr.nvox / niftiHdr.nu;
	niftiHdr.nu = 1;

	// main processing loop over the repeat counters

	SetFileName(niftiHdr.fname, seriesInfo.outName.c_str(), opts.niftiFormat, opts.compressed);

	// In case of two files, set the name here before nifti muck about with it
	
	if ( (opts.niftiFormat == ANALYZE) || (opts.niftiFormat == NIfTI_DUAL) )
		SetFileName(niftiHdr.iname, seriesInfo.outName.c_str(), opts.niftiFormat, opts.compressed, false);

	if (!opts.noact)
	{
		znzFile outfp = NULL;
		outfp = nifti_image_write_hdr_img(&niftiHdr, 2, "wb");
		
		if (outfp == NULL)
		{
			exit(1);
		}
				
		if (! NIfTIWriteData(outfp, niftiHdr, seriesInfo.imageMap) )
		{
			cerr << "Writing output file failed, aborting." << endl << endl;
			znzclose(outfp);
			exit(1);
		}

		znzclose(outfp);
	}

	// Clean up time

	delete [] niftiHdr.fname;
	if (niftiHdr.iname != NULL) delete [] niftiHdr.iname;


}
