/* Copyright (C) 2000,2001,2002 Manuel Amador (Rudd-O)
   This file is part of Directory administrator.

   Directory administrator is free software; you can redistribute it
   and/or modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   Directory administrator is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <glib.h>
#include <gnome.h>
#include "appsupport.h"
#include "appfunctions.h"
#include "appglobals.h"
#include "groups.h"
#include "users.h"
#include "charset.h"
#include "mkntpwd.h"

#ifdef FREEBSD
#include <des.h>
#else
#include <crypt.h>
#endif /* FREEBSD */

#include <sys/time.h>
#include <unistd.h>
#include "md5.h"
#include "schema.h"

/* standard attributes to check on creation/modification */
gchar *attrs2chk[] = {
  "uid", "uidnumber", "gidnumber", "givenname",
  "initials", "sn", "cn", "homedirectory", "loginshell",
  "mail", "mailLocalAddress", "mailRoutingAddress", "mailHost",
  "gecos", "userPassword", "authPassword", "shadowLastChange",
  "shadowmin", "shadowmax", "shadowwarning", "shadowinactive",
  "shadowexpire", "title", "physicaldeliveryofficename", "ou", "l",
  "telephonenumber", "facsimiletelephonenumber", "o", "homephone",
  "mobile", SAMBA_DOMAIN_NAME, SAMBA_HOME_PATH, SAMBA_HOME_DRIVE,
  SAMBA_PROFILE_PATH, SAMBA_LOGIN_SCRIPT, SAMBA_LM_PASSWORD,
  SAMBA_NT_PASSWORD, SAMBA_SID, "objectclass", "host","employeeNumber",
  NULL
};

gchar *posixlist[] = {
  "uid", "uidnumber", "gidnumber", "homedirectory",
  "loginshell", "gecos", "userPassword", "authPassword",
  "shadowLastChange", "shadowmin", "shadowmax", "shadowwarning",
  "shadowinactive", "shadowexpire",
  NULL
};

gchar *sambalist[] = {
  SAMBA_HOME_PATH, SAMBA_HOME_DRIVE, SAMBA_PROFILE_PATH,
  SAMBA_LOGIN_SCRIPT, SAMBA_LM_PASSWORD, SAMBA_NT_PASSWORD, SAMBA_SID,
  NULL
};

gchar *maillist[] = {
  "mailLocalAddress", "mailRoutingAddress", "mailHost",
  NULL
};

gchar *accountlist[] = {
  "host",
  NULL
};

gchar *personlist[] = {
  "givenname", "initials", "sn", "cn", "mail", "title",
  "physicaldeliveryofficename", "ou", "l", "telephonenumber",
  "facsimiletelephonenumber", "o", "homephone", "mobile","employeenumber",
  NULL
};

gchar *handled_objectclasses[] = {
  "organizationalPerson","inetOrgPerson","account","posixAccount",
  "shadowAccount",SAMBA_SAM_ACCOUNT,"inetLocalMailRecipient","top",
  "extensibleObject",
  NULL
};

/* This is from 'pam_ldap.c' by Luke Howard, <lukeh@padl.com>
   Used without explicit permission. */
/* i64c - convert an integer to a radix 64 character */
static int
i64c (int i)
{
  const char *base64 =
    "./01234567890ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
  if (i < 0)
    i = 0;
  else if (i > 63)
    i = 63;

  return base64[i];
}

/* This is from 'pam_ldap.c' by Luke Howard, <lukeh@padl.com>
 Used with permission granted in the 'GNU Library General Public License'.
  THANKS, Luke! */
static char *
_get_salt (char salt[3])
{
  int i;
  int j;

  srand (time (NULL));

  for (j = 0; j < 2; j++)
  {
    i = rand () % 3;
    switch (i)
    {
      case 0:
        i = (rand () % (57 - 46)) + 46;
        break;
      case 1:
        i = (rand () % (90 - 65)) + 65;
        break;
      case 2:
        i = (rand () % (122 - 97)) + 97;
        break;
    }
    salt[j] = i;
  }
  salt[2] = '\0';

  return (salt);
}

static char *
_get_md5_salt (char saltbuf[16])
{
  md5_state_t state;
  md5_byte_t digest[16];
  struct timeval tv;
  int i;

  md5_init(&state);
  gettimeofday (&tv, NULL);
  md5_append(&state, (unsigned char *) &tv, sizeof (tv));
  i = getpid();
  md5_append(&state, (unsigned char *) &i, sizeof (i));
  i = clock();
  md5_append(&state, (unsigned char *) &i, sizeof (i));
  md5_append(&state, (unsigned char *) saltbuf, sizeof (saltbuf));
  md5_finish(&state, digest);

  strcpy (saltbuf, "$1$");
  for (i = 0; i < 8; i++)
    saltbuf[i + 3] = i64c (digest[i] & 0x3f);
  saltbuf[i + 3] = '\0';

  return saltbuf;
}

diradmin_user *
diradmin_user_new (gchar * dn, GList * allowed)
{
  diradmin_user *conn = g_new (diradmin_user, 1);
  conn->diradmin_user_data = NULL;
  conn->diradmin_user_allowedservers = NULL;
  conn->diradmin_user_mails = NULL;
  conn->diradmin_user_objectclasses = NULL;

  conn->diradmin_user_data =
    pairs_list_set_attribute (conn->diradmin_user_data, "dn", dn);

  diradmin_user_revokeallservers(conn);
  diradmin_user_set_allowedservers(conn, allowed);

  return conn;
}

diradmin_user *
diradmin_user_duplicate(diradmin_user * tobeduped)
{

  diradmin_user *conn = g_new(diradmin_user, 1);
  conn->diradmin_user_data = NULL;
  conn->diradmin_user_allowedservers = NULL;
  conn->diradmin_user_objectclasses = NULL;

  conn->diradmin_user_data =
    pairs_list_duplicate(tobeduped->diradmin_user_data);

  diradmin_user_set_allowedservers(conn,
                    tobeduped->diradmin_user_allowedservers);
  diradmin_user_set_mails(conn,
                    tobeduped->diradmin_user_mails);
  diradmin_user_set_objectclasses(conn,
                   tobeduped->diradmin_user_objectclasses);
  conn->allserversallowed = tobeduped->allserversallowed;
  return conn;

}

//these three functions receive pointers to pointers as arguments for the datalist
void
diradmin_user_destroy(diradmin_user * todestroy)
{
  if (todestroy)
  {
      pairs_list_destroy(todestroy->diradmin_user_data);
      diradmin_user_set_allowedservers(todestroy, NULL);
      diradmin_user_set_mails(todestroy, NULL);
      diradmin_user_set_objectclasses(todestroy, NULL);
      g_free(todestroy);
  }
}

gchar *
diradmin_user_get_attribute(diradmin_user * conn, gchar * attr)
{
  g_assert (conn);
  g_assert (attr);

  return pairs_list_get_attribute(conn->diradmin_user_data, attr);
}

void
diradmin_user_set_attribute(diradmin_user * conn, gchar * attr, gchar * to)
{
  int i;

  g_assert (conn);
  g_assert (attr);

  conn->diradmin_user_data =
    pairs_list_set_attribute(conn->diradmin_user_data, attr, to);

  diradmin_user_add_objectclass(conn, "top");
  for (i = 0; posixlist[i] != NULL; i++)
    if (g_strcasecmp (posixlist[i], attr) == 0)
    {
      diradmin_user_add_objectclass(conn, "posixAccount");
      diradmin_user_add_objectclass(conn, "shadowAccount");
      diradmin_user_add_objectclass(conn, "extensibleObject");
    }
  for (i = 0; sambalist[i] != NULL; i++)
    if (g_strcasecmp (sambalist[i], attr) == 0)
    {
      diradmin_user_add_objectclass(conn, SAMBA_SAM_ACCOUNT);
    }
  for (i = 0; maillist[i] != NULL; i++)
    if (g_strcasecmp (maillist[i], attr) == 0)
    {
      diradmin_user_add_objectclass(conn, "inetLocalMailRecipient");
    }
  for (i = 0; accountlist[i] != NULL; i++)
    if (g_strcasecmp (accountlist[i], attr) == 0)
    {
      diradmin_user_add_objectclass(conn, "account");
    }
  for (i = 0; personlist[i] != NULL; i++)
    if (g_strcasecmp (personlist[i], attr) == 0)
    {
      diradmin_user_add_objectclass(conn, "organizationalPerson");
      diradmin_user_add_objectclass(conn, "inetOrgPerson");
    }
}

/**
 * Strip unnecessary objectclasses
 *
 * This function analyses the attributes to be set, and
 * removes objectclasses that have no attributes associated
 * with them
 */
void
diradmin_user_tighten_objectclasses(diradmin_user * u)
{
  int isneeded, i;

  g_print("o Tightening objectclasses\n");

  isneeded = FALSE;
  for (i = 0; posixlist[i] != NULL; i++) {
    if (diradmin_user_get_attribute(u, posixlist[i])) {
      isneeded = TRUE;
    }
  }
  if (isneeded == FALSE) {
      g_print("  - Removing posix account classes");
      diradmin_user_remove_objectclass(u, "posixAccount");
      diradmin_user_remove_objectclass(u, "shadowAccount");
      diradmin_user_remove_objectclass(u, "extensibleObject");
  }

  isneeded = FALSE;
  for (i = 0; sambalist[i] != NULL; i++) {
    if (diradmin_user_get_attribute(u, sambalist[i])) {
      isneeded = TRUE;
    }
  }
  if (isneeded == FALSE) {
      g_print("  - Removing samba classes\n");
      diradmin_user_remove_objectclass(u, SAMBA_SAM_ACCOUNT);
  }

  isneeded = FALSE;
  for (i = 0; personlist[i] != NULL; i++) {
    if (diradmin_user_get_attribute(u, personlist[i])) {
      isneeded = TRUE;
    }
  }
  if (isneeded == FALSE) {
    g_print("  - Removing person classes\n");
    diradmin_user_remove_objectclass(u, "organizationalPerson");
    diradmin_user_remove_objectclass(u, "inetOrgPerson");
  }

  isneeded = FALSE;
  for (i = 0; maillist[i] != NULL; i++) {
    if (diradmin_user_get_attribute(u, maillist[i])) {
      isneeded = TRUE;
    }
  }
  if (g_list_length(diradmin_user_get_mails(u)) > 0) {
    isneeded = TRUE;
  }
  if (isneeded == FALSE) {
    g_print("  - Removing mail classes\n");
    diradmin_user_remove_objectclass(u, "inetLocalMailRecipient");
  }
}

void
diradmin_user_remove_attribute(diradmin_user * conn, gchar * attr)
{
  g_assert (conn);
  g_assert (attr);
  conn->diradmin_user_data =
    pairs_list_remove_attribute(conn->diradmin_user_data, attr);
}

void
diradmin_user_set_password(diradmin_user * conn, gchar * to)
{
  char seed[2], buf[96],buf2[96], saltbuf[16], md5crypt[64];
  g_print ("set password called with password: %s\n", to);
  if (preferences.passwordcrypttype == CRYPT)
  {
    snprintf (buf, sizeof buf, "{crypt}%s", crypt (to, _get_salt (seed)));
    snprintf (buf2, sizeof buf2, "CRYPT$%s", crypt (to, _get_salt (seed)));
  }
  else if (preferences.passwordcrypttype == MD5)
  {
    // Due to the way that md5 crypt (shadow) format works, we want
    // the prepend to be {crypt} instead of {md5}
    strncpy (md5crypt, to, sizeof (md5crypt) - 1);
    _get_md5_salt (saltbuf);
    g_print (crypt (md5crypt, saltbuf));
    snprintf (buf, sizeof buf, "{crypt}%s", crypt (md5crypt, saltbuf));
    snprintf (buf2, sizeof buf2, "MD5$%s", crypt (md5crypt, saltbuf));
  }
  else
  {
    snprintf (buf, sizeof buf, "%s", to);
    snprintf (buf2, sizeof buf2, "%s", to);
  }

  if (diradmin_user_has_objectclass(conn, SAMBA_SAM_ACCOUNT))
  {
    char hash[17], hexstr[33];

    mklmhash(to, hash);
    to_hexstr(hash, hexstr);
    g_print("Setting samba lm password/n");
    diradmin_user_set_attribute(conn, SAMBA_LM_PASSWORD, hexstr);

    mknthash(to, hash);
    to_hexstr(hash, hexstr);
    g_print("Setting samba nt password/n");
    diradmin_user_set_attribute(conn, SAMBA_NT_PASSWORD, hexstr);
  }
  diradmin_user_set_attribute (conn, "userPassword", buf);
  if (preferences.rfc2307bis == TRUE)
  {
    diradmin_user_set_attribute (conn, "authPassword", buf2);
  }

  //this line of code is taken from pam_ldap.c
  snprintf(buf, sizeof buf, "%ld", time (NULL) / (60 * 60 * 24));
  diradmin_user_set_attribute(conn, "shadowLastChange", buf);

  g_print ("set password set attribute: %s\n",
       diradmin_user_get_attribute(conn, "userPassword"));
}

diradmin_user *
diradmin_user_new_from_ldap(connection_profile * usethisone, char *userdn)
{
  diradmin_user *user = NULL;

  int ldap_errors;

  GList* ocs = NULL;

  LDAP *h;
  LDAPMessage *searchresults = NULL;
  LDAPMessage *entry = NULL;

  char **value_collection = NULL;

  gchar *attribute;
  BerElement *attributehandler;
  gchar *localed_attr;
  int i = 0;
  int ocnum = 0;

  g_print("\nFetching %s from directory\n", userdn);

  //check 4 connection
  h = connection_profile_get_ldap_handler(usethisone);
  g_assert (h);

  //look data up
  ldap_errors =
    ldap_search_s(h, userdn, LDAP_SCOPE_BASE, "(objectclass=*)", NULL, 0,
           &searchresults);

  if (ldap_errors)
  {
    g_print("LDAP error while creating a diradmin_user structure for ");
    g_print(userdn);
    g_print(": ");
    g_print(ldap_err2string (ldap_errors));
    g_print("\n");

    if (ldap_errors ==  LDAP_SERVER_DOWN)
    {
      connection_profile_invalidate(usethisone);
    }
    return NULL;
  }
  else
  {
    user = diradmin_user_new(userdn, NULL);

    //get only first entry
    entry = ldap_first_entry(h, searchresults);

    // loop thru attribute values
    attribute = ldap_first_attribute(h, entry, &attributehandler);
    g_assert (attribute);
    while (attribute)
    {
      value_collection = ldap_get_values(h, entry, attribute);
      g_assert(value_collection);
      if (g_strcasecmp(attribute, "mailLocalAddress") == 0)
      {
        for (i = 0; value_collection[i]; i++)
	{
          diradmin_user_add_mail(user, value_collection[i]);
        }
      }
      else if (g_strcasecmp(attribute, "host") == 0)
      {
        for (i = 0; value_collection[i]; i++)
	{
          if (g_strcasecmp(value_collection[i],"*")==0)
	  {
            diradmin_user_allowallservers(user);
          }
          else
	  {
            diradmin_user_add_allowedserver(user, value_collection[i]);
          }
        }
      }
      else if (g_strcasecmp(attribute, "objectclass") == 0)
      {
        for (i = 0; value_collection[i]; i++)
	{
          diradmin_user_set_objectclasses(user,NULL);
          for (ocnum=0;handled_objectclasses[ocnum];ocnum++)
	  {
            if (g_strcasecmp(value_collection[i],handled_objectclasses[ocnum])==0)
	    {
              ocs = g_list_append(ocs,g_strdup(value_collection[i]));
            }
          }
        }
      }
      else
      {
        g_assert (value_collection[0]);
        //detect if gecos
        if (g_strcasecmp (attribute, "gecos") == 0)
          localed_attr = convert_from_ascii(value_collection[0]);
        else
          localed_attr = convert_from_utf8(value_collection[0]);
        diradmin_user_set_attribute (user, attribute,
                                     localed_attr);
        g_free (localed_attr);
      }
      ldap_value_free (value_collection);
      attribute = ldap_next_attribute (h, entry, attributehandler);
    }
    ldap_msgfree (searchresults);
  }
  diradmin_user_set_objectclasses (user, ocs);

  return (user);
}

GList *
diradmin_user_get_allowedservers(diradmin_user * conn)
{
  return g_list_first(conn->diradmin_user_allowedservers);
}

void
diradmin_user_add_allowedserver(diradmin_user * conn, gchar * allowedserver)
{
  diradmin_user_remove_allowedserver(conn, allowedserver);
  conn->diradmin_user_allowedservers =
    g_list_append(conn->diradmin_user_allowedservers,
           g_strdup(allowedserver));
  diradmin_user_add_objectclass(conn, "account");
  conn->allserversallowed =FALSE;
}

void
diradmin_user_revokeallservers(diradmin_user*c) {
 diradmin_user_set_allowedservers(c,NULL);
 diradmin_user_remove_objectclass(c,"account");
 c->allserversallowed = FALSE;
}

void
diradmin_user_allowallservers(diradmin_user * conn)
{
  diradmin_user_revokeallservers(conn);
  diradmin_user_add_objectclass(conn,"account");
  conn->allserversallowed=TRUE;
}

gboolean
diradmin_user_has_allservers (diradmin_user * conn)
{
  return conn->allserversallowed;
}

void
diradmin_user_remove_allowedserver(diradmin_user * conn, gchar * allowedserver)
{
  GList *iterator = NULL;
  iterator = g_list_find_custom(conn->diradmin_user_allowedservers,
                 allowedserver, (GCompareFunc) g_strcasecmp);
  if (iterator)
  {
    conn->diradmin_user_allowedservers =
      g_list_remove_link (conn->diradmin_user_allowedservers, iterator);
    g_free (iterator->data);
    g_list_free_1 (iterator);
  }
  if (g_list_length (conn->diradmin_user_allowedservers) == 0)
    g_print ("\nRemoving object class Account\n");
  diradmin_user_remove_objectclass (conn, "account");
}

gboolean
diradmin_user_has_allowedserver(diradmin_user * conn, gchar * allowedserver)
{
  gchar *data;
  GList *iterator = NULL;
  iterator = g_list_first(conn->diradmin_user_allowedservers);
  while (iterator)
  {
    data = iterator->data;
    if (g_strcasecmp(allowedserver, iterator->data) == 0)
    {
      return (TRUE);
    }
      iterator = g_list_next(iterator);
    }
  return (FALSE);
}

void
diradmin_user_set_allowedservers(diradmin_user * conn, GList * allowedserverlist)
{
  //removes all allowedservers in the list.  that means frees its allocated storage.
  GList *newallowedservers = NULL;
  GList *allowedservers = NULL;

  //free the old allowedservers list
  allowedservers = g_list_first(diradmin_user_get_allowedservers (conn));
  while (allowedservers)
  {
    g_free (allowedservers->data);
    allowedservers = g_list_next(allowedservers);
  }
  g_list_free (g_list_first(diradmin_user_get_allowedservers (conn)));

  //make a copy of the passed allowedservers list into newallowedservers
  allowedservers = g_list_first(allowedserverlist);
  while (allowedservers)
  {
    newallowedservers =
      g_list_append (newallowedservers, g_strdup (allowedservers->data));
    allowedservers = g_list_next (allowedservers);
  }
  conn->diradmin_user_allowedservers = newallowedservers;
}

/* Get mail list */
GList *
diradmin_user_get_mails(diradmin_user * conn)
{
  //return a reference to the mails list.  shouldnt be manipulated outside.
  return g_list_first (conn->diradmin_user_mails);
}

/* Set mail list */
void diradmin_user_set_mails(diradmin_user *conn, GList *maillist)
{
  /* removes all mails in the list.  that means frees its allocated storage. */
  GList *newmails = NULL;
  GList *mails = NULL;

  /* free the old mails list */
  mails = g_list_first (diradmin_user_get_mails (conn));
  while (mails)
  {
    g_free (mails->data);
    mails = g_list_next(mails);
  }
  g_list_free (g_list_first(diradmin_user_get_mails (conn)));

  /* make a copy of the passed mails list into newmails */
  mails = g_list_first(maillist);
  while (mails)
  {
    newmails = g_list_append(newmails, g_strdup (mails->data));
    mails = g_list_next(mails);
  }
  conn->diradmin_user_mails = newmails;
}

/* Add mail to mail list */
void
diradmin_user_add_mail(diradmin_user *conn, gchar *mail)
{
  diradmin_user_remove_mail(conn, mail);
  conn->diradmin_user_mails =
      g_list_append(conn->diradmin_user_mails, g_strdup (mail));
  g_print ("\no Adding object class inetLocalMailRecipient\n");
  diradmin_user_add_objectclass (conn, "inetLocalMailRecipient");
}

/* Remove mail from mail list */
void
diradmin_user_remove_mail(diradmin_user * conn, gchar * mail)
{
  GList *iterator = NULL;
  iterator = g_list_find_custom(conn->diradmin_user_mails,
                 mail, (GCompareFunc) g_strcasecmp);
  if (iterator)
  {
    conn->diradmin_user_mails =
             g_list_remove_link(conn->diradmin_user_mails, iterator);
    g_free(iterator->data);
    g_list_free_1(iterator);
  }
  if (g_list_length(conn->diradmin_user_mails) == 0)
  {
    g_print("\no Removing object class Account\n");
    diradmin_user_remove_objectclass(conn, "inetLocalMailRecipient");
  }
}

/* Test if this particular mail is present */
gboolean
diradmin_user_has_mail(diradmin_user * conn, gchar *mail)
{
  gchar *data;
  GList *iterator = NULL;

  iterator = g_list_first(conn->diradmin_user_mails);
  while (iterator)
  {
    data = iterator->data;
    if (g_strcasecmp(mail, iterator->data) == 0)
    {
      return (TRUE);
    }
    iterator = g_list_next(iterator);
  }
  return (FALSE);
}

GList *
diradmin_user_get_objectclasses(diradmin_user * conn)
{
  return g_list_first(conn->diradmin_user_objectclasses);
}

void
diradmin_user_add_objectclass(diradmin_user * conn, gchar * allowedserver)
{
  g_assert(conn);
  g_assert(allowedserver);

  diradmin_user_remove_objectclass(conn, allowedserver);
  conn->diradmin_user_objectclasses =
    g_list_append(conn->diradmin_user_objectclasses,
           g_strdup(allowedserver));
}

void
diradmin_user_remove_objectclass(diradmin_user * conn, gchar * allowedserver)
{
  GList *iterator = NULL;
  iterator = g_list_find_custom(conn->diradmin_user_objectclasses,
                 allowedserver, (GCompareFunc) g_strcasecmp);
  if (iterator)
    {
      conn->diradmin_user_objectclasses =
    g_list_remove_link(conn->diradmin_user_objectclasses, iterator);
      g_free(iterator->data);
      g_list_free_1(iterator);
    }
}

gboolean
diradmin_user_has_objectclass(diradmin_user * conn, gchar * allowedserver)
{
  gchar *data;
  GList *iterator = NULL;

  iterator = g_list_first(conn->diradmin_user_objectclasses);
  while (iterator)
    {
      data = iterator->data;
      if (g_strcasecmp(allowedserver, iterator->data) == 0)
    {
      return (TRUE);
    }
      iterator = g_list_next(iterator);
    }
  return (FALSE);
}

void
diradmin_user_set_objectclasses(diradmin_user * conn, GList * allowedserverlist)
{
  //removes all allowedservers in the list.  that means frees its allocated storage.
  GList *newallowedservers = NULL;
  GList *allowedservers = NULL;

  //free the old allowedservers list
  allowedservers = g_list_first(diradmin_user_get_objectclasses(conn));
  while (allowedservers)
  {
    g_free(allowedservers->data);
    allowedservers = g_list_next(allowedservers);
  }
  g_list_free(g_list_first(conn->diradmin_user_objectclasses));

  //make a copy of the passed allowedservers list into newallowedservers
  allowedservers = g_list_first(allowedserverlist);
  while (allowedservers)
  {
    newallowedservers =
      g_list_append(newallowedservers, g_strdup(allowedservers->data));
    allowedservers = g_list_next(allowedservers);
  }

  conn->diradmin_user_objectclasses = newallowedservers;
}

void
diradmin_user_dump(diradmin_user * conn)
{
  GList *objectclasses = NULL;

  g_print("\nDump for user %s:", diradmin_user_get_attribute(conn,"dn"));

  objectclasses = g_list_first(diradmin_user_get_objectclasses(conn));
  while (objectclasses)
  {
    g_print("\n    Object class: %s" ,(char*)objectclasses->data);
    objectclasses = g_list_next(objectclasses);
  }
}

ldaptransaction *
diradmin_user_generate_ldapdiff (diradmin_user * oldone, diradmin_user * newone)
{
  ldaptransaction *t;
  gchar **vals = NULL;
  GList *l = NULL;
  gint acnt = 0;
  gchar *oldattr = NULL;
  gchar *newattr = NULL;
  gboolean samba_user = 0;
  gboolean holder;
  diradmin_user*m; gchar*n;
  gboolean ispassword;

  g_assert (oldone);
  g_assert (newone);

  diradmin_user_tighten_objectclasses(newone);
  samba_user = diradmin_user_has_objectclass(newone, SAMBA_SAM_ACCOUNT);

  t = ldaptransaction_new ();

  /* get old objectclasses */
  acnt = 0;
  l = diradmin_user_get_objectclasses (oldone);
  vals = g_new0 (gchar *, g_list_length(l) + 1);

  for (l = g_list_first(l); l; l = g_list_next (l))
  {
    if (diradmin_user_has_objectclass(newone, l->data) == FALSE)
    {
      vals[acnt] = g_strdup(l->data); acnt++;
    }
  }

  if (acnt)
  {
    ldaptransaction_delete(t, "objectclass", vals);
  }
  else
  {
    g_free(vals);
  }

  /* get new objectclasses */
  acnt = 0;
  l = diradmin_user_get_objectclasses(newone);
  vals = g_new0(gchar *, g_list_length(l) + 11);

  for (l = g_list_first(l); l; l = g_list_next (l))
  {
    m = oldone;
    n = l->data;
    holder = diradmin_user_has_objectclass(m,n);
    
    if (holder == FALSE)
    {
      vals[acnt] = g_strdup (l->data); acnt++;
    }
  }

  if (acnt)
  {
    ldaptransaction_add(t, "objectclass", vals);
  }
  else 
  {
    g_free(vals);
  }

  /* get old mailLocalAddresses */
  acnt = 0;
  l = diradmin_user_get_mails (oldone);
  vals = g_new0 (gchar *, g_list_length(l) + 1);

  for (l = g_list_first(l); l; l = g_list_next (l))
  {
    if (diradmin_user_has_mail(newone, l->data) == FALSE)
    {
      vals[acnt] = g_strdup(l->data); acnt++;
    }
  }

  if (acnt) 
  {
    ldaptransaction_delete(t, "mailLocalAddress", vals);
  }
  else 
  {
    g_free(vals);
  }

  /* get new mailLocalAddresses */
  acnt = 0;
  l = diradmin_user_get_mails(newone);
  vals = g_new0(gchar *, g_list_length(l) + 11);

  for (l = g_list_first(l); l; l = g_list_next (l))
  {
    m = oldone;
    n = l->data;
    holder = diradmin_user_has_mail(m,n);

    if (holder == FALSE)
    {
      vals[acnt] = g_strdup(l->data); acnt++;
    }
  }

  if (acnt) 
  {
    ldaptransaction_add(t, "mailLocalAddress", vals);
  }
  else
  {
    g_free(vals);
  }

  if (diradmin_user_has_allservers(newone))
  {
    if (!diradmin_user_has_allservers(oldone))
    {
      vals = g_new0(gchar *, 2);
      vals[0] = g_strdup("*");
      ldaptransaction_add(t, "host", vals);
    }
  }
  if (diradmin_user_has_allservers(oldone))
  {
    if (!diradmin_user_has_allservers(newone))
    {
      vals = g_new0(gchar *, 2);
      vals[0] = g_strdup("*");
      ldaptransaction_delete(t, "host", vals);
    }
  }

  /* get old allowedservers */
  acnt = 0;
  l = diradmin_user_get_allowedservers(oldone);
  vals = g_new0(gchar *, g_list_length(l) + 1);

  for (l = g_list_first(l); l; l = g_list_next(l))
  {
    if (diradmin_user_has_allowedserver(newone, l->data) == FALSE)
    {
      vals[acnt] = g_strdup(l->data); acnt++;
    }
  }

  if (acnt) 
  {
    ldaptransaction_delete(t, "host", vals);
  }
  else
  {
    g_free(vals);
  }

  /* get new allowedservers */
  acnt = 0;
  l = diradmin_user_get_allowedservers(newone);
  vals = g_new0(gchar *, g_list_length(l) + 1);

  for (l = g_list_first(l); l; l = g_list_next(l))
  {
    if (diradmin_user_has_allowedserver(oldone, l->data) == FALSE)
    {
      vals[acnt] = g_strdup(l->data); acnt++;
    }
  }

  if (acnt)
  {
    ldaptransaction_add(t, "host", vals);
  }
  else 
  {
    g_free(vals);
  }

  for (acnt = 0; attrs2chk[acnt]; acnt++)
  {
    oldattr = diradmin_user_get_attribute(oldone, attrs2chk[acnt]);
    newattr = diradmin_user_get_attribute(newone, attrs2chk[acnt]);
    ispassword = g_strcasecmp(attrs2chk[acnt], "userPassword") == 0
    	||  g_strcasecmp(attrs2chk[acnt], "authPassword") == 0
	||  (samba_user && g_strcasecmp(attrs2chk[acnt], SAMBA_NT_PASSWORD) == 0)
	|| (samba_user && g_strcasecmp(attrs2chk[acnt], SAMBA_LM_PASSWORD) == 0);

    /* if any attribute isn't empty */
    if (oldattr || newattr)
    {
      g_print("attribute %s:\n   old %s\n   new %s\n",
               attrs2chk[acnt], oldattr, newattr);

      /* if the attribute is userPassword, and LDAP_EXOP_MODIFY_PASSWD is
       * to be used to set the password
       */
      if (preferences.passwordcrypttype == EXTENDED
      		&& g_strcasecmp(attrs2chk[acnt], "userPassword") == 0)
      {
        ldaptransaction_set_password(t, convert_to_utf8(newattr));
      }

      /* if the attribute isn't a password or the date of last change, and
       * the old attribute existed but the new one doesn't
       */
      else if (!ispassword
      		&&  g_strcasecmp(attrs2chk[acnt], "shadowLastChange") != 0 
		&& oldattr != NULL
		&& newattr == NULL)
      {
        vals = g_new0(gchar *, 2);

        if (g_strcasecmp(attrs2chk[acnt], "gecos") == 0)
	{
          vals[0] = convert_to_ascii(oldattr);
        }
        else 
	{
          vals[0] = convert_to_utf8(oldattr);
        }
        ldaptransaction_delete(t, attrs2chk[acnt], vals);
      }

      /* else if the old attribute didn't exist but was added */
      else if (oldattr == NULL && newattr != NULL)
      {
        vals = g_new0(gchar *, 2);

        /* if it's the gecos, convert to ascii first */
        if (g_strcasecmp(attrs2chk[acnt], "gecos") == 0)
	{
          vals[0] = convert_to_ascii(newattr);
        }
        else {
          vals[0] = convert_to_utf8(newattr);
        }
        ldaptransaction_replace(t, attrs2chk[acnt], vals);
      }

      /* else if both existed but are different (excluding the case that
       * the attribute we're checking is the DN) 
       */
      else if (oldattr
      		&& newattr
		&& strcmp(oldattr, newattr) != 0
		&& strcmp(oldattr, "dn") != 0)
      {
        vals = g_new0(gchar *, 2);

        /* if it's the gecos, please convert to ascii first */
        if (g_strcasecmp(attrs2chk[acnt], "gecos") == 0)
	{
          vals[0] = convert_to_ascii(newattr);
        }
        else {
          vals[0] = convert_to_utf8(newattr);
        }
        ldaptransaction_replace(t, attrs2chk[acnt], vals);
      }
    }
  }
  return (t);
}

diradmin_user *
create_user_struct_from_dialogbox(connection_profile * conn,
                   GtkWidget * dialogbox, gchar * dn)
{
  diradmin_user *newuser;

  gint iterator;
  gchar *uid = NULL;
  gchar *uidnumber = NULL;
  gchar *gidnumber = NULL;
  gchar *givenname = NULL;
  gchar *sn = NULL;
  gchar *cn = NULL;
  gchar *homedirectory = NULL;
  gchar *loginshell = NULL;
  gchar *userpassword = NULL;
  gchar *gecos = NULL;
  gchar *mail = NULL;
  gchar *mailhost = NULL;
  gchar *mailroutingaddress = NULL;
  gchar *shadowmin = NULL;
  gchar *shadowmax = NULL;
  gchar *shadowwarning = NULL;
  gchar *sambaSID;
  gchar *sambaDomainName;
  gchar *sambaHomePath = NULL;
  gchar *sambaHomeDrive = NULL;
  gchar *sambaProfilePath = NULL;
  gchar *sambaLoginScript = NULL;
  gchar *sambaLMPassword = NULL;
  gchar *sambaNTPassword = NULL;
  gchar *shadowinactive = NULL;

  char buf[32];
  gchar *warning = NULL;
  time_t shadowexpiretime;

  GtkCList *allowedservers = NULL;
  gchar *allowedserver = NULL;
  GtkCList *mailLocalAddresses = NULL;
  gchar *mailLocalAddress = NULL;

  allowedservers = (GtkCList *) lookup_widget(dialogbox, "allowedservers");

  uid = gtk_entry_get_text (GTK_ENTRY (lookup_widget(dialogbox, "uid")));
  uidnumber =
    gtk_entry_get_text (GTK_ENTRY (lookup_widget(dialogbox, "uidnumber")));
  gidnumber =
    cn_to_gidnumber (conn,
             gtk_entry_get_text (GTK_ENTRY
                     (lookup_widget
                      (dialogbox, "gidnumber"))));
  sn = gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "sn")));
  cn = gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "cn")));
  givenname =
    gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "givenname")));
  homedirectory =
    gtk_entry_get_text (GTK_ENTRY
            (lookup_widget(dialogbox, "homedirectory")));
  loginshell =
    gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "loginshell")));

  gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "loginshell")));
  mail =
    gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "mail")));
  mailLocalAddresses = (GtkCList *) lookup_widget(dialogbox, "mailLocalAddress");
  mailhost =
    gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "mailhost")));
  mailroutingaddress =
    gtk_entry_get_text(GTK_ENTRY
            (lookup_widget(dialogbox, "mailroutingaddress")));

  userpassword =
    gtk_entry_get_text(GTK_ENTRY
            (lookup_widget(dialogbox, "userpassword")));

  sambaLMPassword =
    gtk_entry_get_text(GTK_ENTRY
            (lookup_widget(dialogbox, "userpassword")));
  sambaNTPassword =
    gtk_entry_get_text(GTK_ENTRY
            (lookup_widget(dialogbox, "userpassword")));
  gecos = gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "cn")));

  newuser = diradmin_user_new(dn, NULL);

  diradmin_user_set_attribute(newuser, "uid", uid);
  diradmin_user_set_attribute(newuser, "uidnumber", uidnumber);
  diradmin_user_set_attribute(newuser, "gidnumber", gidnumber);
  diradmin_user_set_attribute(newuser, "givenname", givenname);
  diradmin_user_set_attribute(newuser, "sn", sn);
  diradmin_user_set_attribute(newuser, "cn", cn);
  diradmin_user_set_attribute(newuser, "homedirectory", homedirectory);
  diradmin_user_set_attribute(newuser, "loginshell", loginshell);
  diradmin_user_set_attribute(newuser, "gecos", gecos);
  diradmin_user_set_attribute(newuser, "mail", mail);

  for (iterator = 0; iterator < mailLocalAddresses->rows; iterator++)
  {
    gtk_clist_get_text(mailLocalAddresses, iterator, 0, &mailLocalAddress);
    diradmin_user_add_mail(newuser, mailLocalAddress);
  }

  g_free (gidnumber);        //mallocced by the function

  //shadow password setting attributes
  shadowmin =
    gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "shadowmin")));
  shadowmax =
    gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "shadowmax")));
  shadowwarning =
    gtk_entry_get_text(GTK_ENTRY
            (lookup_widget(dialogbox, "shadowwarning")));
  shadowinactive =
    gtk_entry_get_text(GTK_ENTRY
            (lookup_widget(dialogbox, "shadowinactive")));
  diradmin_user_set_attribute(newuser, "shadowmin", shadowmin);
  diradmin_user_set_attribute(newuser, "shadowmax", shadowmax);
  diradmin_user_set_attribute(newuser, "shadowwarning", shadowwarning);
  diradmin_user_set_attribute(newuser, "shadowinactive", shadowinactive);

//organizationalperson
  diradmin_user_set_attribute(newuser, "initials",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox,
                             "initials"))));
  diradmin_user_set_attribute(newuser, "title",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox, "title"))));
  diradmin_user_set_attribute(newuser, "physicaldeliveryofficename",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox,
                             "physicaldeliveryofficename"))));
  diradmin_user_set_attribute(newuser, "ou",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox, "ou"))));
  diradmin_user_set_attribute(newuser, "l",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox, "l"))));
  diradmin_user_set_attribute(newuser, "employeenumber",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox, "employeenumber"))));
  diradmin_user_set_attribute(newuser, "telephonenumber",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox,
                             "telephonenumber"))));
  diradmin_user_set_attribute(newuser, "facsimiletelephonenumber",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox,
                             "facsimiletelephonenumber"))));
  diradmin_user_set_attribute(newuser, "o",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox, "o"))));
  diradmin_user_set_attribute(newuser, "homephone",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox,
                             "homephone"))));
  diradmin_user_set_attribute(newuser, "mobile",
                   gtk_entry_get_text(GTK_ENTRY
                           (lookup_widget
                            (dialogbox, "mobile"))));

  if (GTK_TOGGLE_BUTTON(lookup_widget(dialogbox, "shadowexpirecheck"))->active)
  {
    shadowexpiretime = gnome_date_edit_get_date((GnomeDateEdit *)
                       lookup_widget(dialogbox, "shadowexpire"));
    snprintf (buf, sizeof buf, "%ld", shadowexpiretime / (60 * 60 * 24 ) + 1);
    diradmin_user_set_attribute(newuser, "shadowexpire", buf);
  }

  if (!GTK_TOGGLE_BUTTON
      (lookup_widget(dialogbox, "logontoallservers"))->active)
  {
    diradmin_user_revokeallservers(newuser);
  }

  if (GTK_TOGGLE_BUTTON(lookup_widget(dialogbox, "logontoallservers"))->active)
  {
    diradmin_user_allowallservers(newuser);
  }
  else
  {
    for (iterator = 0; iterator < allowedservers->rows; iterator++)
    {
      gtk_clist_get_text(allowedservers, iterator, 0, &allowedserver);
      diradmin_user_add_allowedserver(newuser, allowedserver);
    }
  }

  if (GTK_TOGGLE_BUTTON
      (lookup_widget(dialogbox, "enablemailpolicycontrol"))->active) 
  {
    for (iterator = 0; mailLocalAddresses
    	&& iterator < mailLocalAddresses->rows; iterator++)
    {
      gtk_clist_get_text(mailLocalAddresses, iterator, 0, &mailLocalAddress);
      diradmin_user_add_mail(newuser, mailLocalAddress);
    }
    if (GTK_TOGGLE_BUTTON(lookup_widget(dialogbox, "mailroutingaddressset"))->active)
    {
      diradmin_user_set_attribute(newuser, "mailRoutingAddress",
                                   mailroutingaddress);
    }
    if (GTK_TOGGLE_BUTTON(lookup_widget(dialogbox, "mailhostset"))->active)
    {
      diradmin_user_set_attribute(newuser, "mailHost", mailhost);
    }
  }

  if (GTK_TOGGLE_BUTTON
      (lookup_widget(dialogbox, "enablesambaobjectclass"))->active) {

    sambaDomainName =
          gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "sambaDomainName")));

    sambaHomePath =
          gtk_entry_get_text(GTK_ENTRY (lookup_widget(dialogbox, "sambaHomePath")));

    sambaHomeDrive =
          gtk_entry_get_text(GTK_ENTRY
                (lookup_widget(dialogbox, "sambaHomeDrive")));
    sambaProfilePath =
          gtk_entry_get_text(GTK_ENTRY
                (lookup_widget(dialogbox, "sambaProfilePath")));
    sambaLoginScript =
          gtk_entry_get_text(GTK_ENTRY
                (lookup_widget(dialogbox, "sambaLogonScript")));

    sambaSID = app_get_sambasid_from_ldap(conn, sambaDomainName, 0, uidnumber, 0, &warning);
    if (!sambaSID)
    {
      gtk_widget_show(create_messagebox_with_message(g_strconcat(warning ? warning 
      	: "", "No valid Windows domains have been found. A user cannot have Windows domain support enabled when this is the case.", NULL)));
    }
    else
    {
      diradmin_user_set_attribute(newuser, SAMBA_SID, sambaSID);
      diradmin_user_set_attribute(newuser, SAMBA_DOMAIN_NAME, sambaDomainName);
      diradmin_user_set_attribute(newuser, SAMBA_HOME_PATH, sambaHomePath);
      diradmin_user_set_attribute(newuser, SAMBA_HOME_DRIVE, sambaHomeDrive);
      diradmin_user_set_attribute(newuser, SAMBA_PROFILE_PATH, sambaProfilePath);
      diradmin_user_set_attribute(newuser, SAMBA_LOGIN_SCRIPT, sambaLoginScript);
    }
  }
  else
  {
    diradmin_user_remove_attribute(newuser, SAMBA_LM_PASSWORD);
    diradmin_user_remove_attribute(newuser, SAMBA_NT_PASSWORD);
  }

  if (strlen (userpassword) > 0)
  {
    diradmin_user_set_password(newuser, userpassword);
  }

  return (newuser);
}

ldaptransaction *
diradmin_user_create_ldapdiff(diradmin_user * newone)
{
  ldaptransaction *t = NULL;
  gchar **values;
  GList *l = NULL;
  gint acnt = 0;

  g_assert(newone);
  t = ldaptransaction_new();
  l = diradmin_user_get_objectclasses(newone);
  values = g_new0(gchar*,g_list_length(l)+1);
  acnt = 0;

  for (l = g_list_first(l); l; l = g_list_next(l))
  {
    values[acnt] = g_strdup(l->data); acnt++;
  }
  ldaptransaction_add(t, "objectclass", values);

  if (diradmin_user_has_allservers(newone))
  {
    values = g_new0(gchar *, 2);
    values[0] = g_strdup("*");
    ldaptransaction_add(t, "host", values);
  }
  l = diradmin_user_get_allowedservers(newone);

  if (g_list_length(l))
  {
    values = g_new0(gchar*,g_list_length(l)+1);
    acnt = 0;
    for (l = g_list_first(l); l; l = g_list_next(l))
    {
      values[acnt] = g_strdup (l->data); acnt++;
    }
    ldaptransaction_add (t, "host", values);
  }
  l = diradmin_user_get_mails (newone);

  if (g_list_length(l))
  {
    values = g_new0(gchar*,g_list_length(l)+1);
    acnt = 0;
    for (l = g_list_first(l); l; l = g_list_next(l))
    {
      values[acnt] = g_strdup(l->data); acnt++;
    }
    ldaptransaction_add (t, "mailLocalAddress", values);
  }

  for (acnt = 0; attrs2chk[acnt]; acnt++)
    if (diradmin_user_get_attribute(newone, attrs2chk[acnt]) != NULL)
    {
      values = g_new0 (gchar *, 2);

      if (g_strcasecmp (attrs2chk[acnt], "gecos") == 0)
      {
           values[0] = convert_to_ascii(diradmin_user_get_attribute(newone, attrs2chk[acnt]));
      }
      else
      {
           values[0] = convert_to_utf8(diradmin_user_get_attribute (newone, attrs2chk[acnt]));
      }
    ldaptransaction_add (t, attrs2chk[acnt], values);
    }
  ldaptransaction_dump(t);

  return t;
}
