#include "audiofile.h"

#ifdef HAVE_LIBAUDIOFILE

#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <fcntl.h>
#include <qthread.h>
#include "mainwnd.h"
#include "pump.h"

#define BLOCK_SKIP 4096

AudiofileStreamSource::AudiofileStreamSource(QWidget *dest) : StreamSource(dest)
{
	fp=NULL;
	fd=-1;
	file_pos=0L;
}

AudiofileStreamSource::~AudiofileStreamSource()
{
	if(fp)
	{
		afCloseFile(fp);
	}
}

int AudiofileStreamSource::attach(QString file)
{
	if(access(file, 0) < 0)
	{
		printf("File not found\n");
		return -1;
	}
	
	filename=file;

	fd=open(file, O_RDONLY);
	if(fd < 0)
	{
		printf("Can't open file\n");
		return -1;
	}

	fp=afOpenFile((char *)(const char *)file, "r", NULL);
	if(fp == NULL)
	{
		printf("Can't open file as audio file\n");
		return -1;
	}
	
	int track_ids[128];
	if(!afGetTrackIDs(fp, track_ids))
	{
		printf("File has no tracks\n");
		afCloseFile(fp);
		close(fd);
		return -1;
	}

	track_id=track_ids[0];

	if(afGetChannels(fp, track_id) != 2)
	{
		printf("File is not stereo!\n");
		afCloseFile(fp);
		close(fd);
		return -1;
	}

	sample_rate=(int)afGetRate(fp, track_id);
	frame_pos=0L;
	file_pos=0L;
	max_frames=afGetFrameCount(fp, track_id);

	m_artist="";
	m_title="";

	char fnb[1024];
	strcpy(fnb, file);
	char *fn=strrchr(fnb, '/');
	if(!fn)
		fn=fnb;
	else
		++fn;
	char *dot=strrchr(fn, '.');
	if(dot)
		(*dot)='\0';
	char *sep=strstr(fn, " - ");
	if(sep)
	{
		(*sep)='\0';
		sep+=3;
	}

	if(sep)
	{
		m_artist=fn;
		m_title=sep;
	}
	else
	{
		m_artist="";
		m_title=fn;
	}

	return 0;
}

int AudiofileStreamSource::get_buffer(char *buf, int max)
{
	if(!fp || !playing)
		return 0;
	
	if(file_pos+max > max_frames)
		max=max_frames-file_pos;

	if(!max)
	{
		if(playing)
			QThread::postEvent(this, new QEvent(QEvent::User));
		playing=0;
		return 0;
	}

	if(play_samples)
	{
		if((unsigned long)max > play_samples)
			max=(int)play_samples;
	}

	short tb[max*sizeof(short)*2];

	afReadFrames(fp, track_id, tb, max);
	int i;

	for(i=0;i<max;i++)
	{
		*(((short *)buf)+i)=tb[i*2];
		*(((short *)buf)+max+i)=tb[i*2+1];
	}

	file_pos+=max;

	if(play_samples)
	{
		play_samples-=max;
		if(play_samples <= 0)
		{
			if(loop_play)
				play(last_start, last_samples, true);
			else
			{
				playing=0;
				QThread::postEvent(this, new QEvent(QEvent::User));
			}
		}
	}
	
	calcAgc(buf, max);
	return max;
}

int AudiofileStreamSource::play(unsigned long start_frame, unsigned long samples, bool loop)
{
	if(!fp)
		return -1;
	
	if(!loop_play || !loop)
		stop();

	if(start_frame >= max_frames)
		return -1;

	loop_play=loop;
	last_start=start_frame;

	afSeekFrame(fp, track_id, start_frame);

	file_pos=start_frame;
	frame_pos=start_frame;
	play_samples=samples;
	last_samples=play_samples;

	playing=1;
	app_window->get_pump()->work();
	return 0;
}

int AudiofileStreamSource::stop(void)
{
	if(!player || !fp || !playing)
		return 0;

	StreamSource::stop();
	playing=0;
	return 0;
}

StreamSource *AudiofileStreamSource::get_source(void)
{
	AudiofileStreamSource *s=new AudiofileStreamSource(app_window);
	s->attach(filename);
	return (StreamSource *)s;
}

void AudiofileStreamSource::exit_loop(void)
{
	pthread_mutex_lock(&control_lock);
	play_samples=0;
	loop_play=false;
	pthread_mutex_unlock(&control_lock);
}

#endif /* HAVE_LIBAUDIOFILE */
