#include "frame.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/joystick.h>
#include "config.h"
#include "jack.h"
#include "mixer.h"
#include <bitmapbutton.h>
#include <bitmapslider.h>
#include <qapplication.h>
#include "metatracker.h"
#include "cfgfile.h"
#include "mixerpanel.h"

extern MixerPanel *mix1, *mix2;

static int led_numbers[2][15]={{11, 14, 12, 0, 1, 3, 5, 10, 0, 0, 0, 0, 0, 0, 0},
						       {9, 8, 13, 15, 2, 4, 6, 7, 0, 0, 0, 0, 0, 0, 0}};

static int left_mode=1,right_mode=1;

static int led_states[16]={-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1};

#ifdef CONTROL_UNIT
#include <parapin.h>
#endif
#ifdef HAVE_LIBSDL
extern "C" {
#include <SDL.h>
#include <SDL_syswm.h>
}
#endif

#include <qsocketnotifier.h>
#include <qtimer.h>

#define LEFT_PLAY 0
#define LEFT_CUE 1
#define LEFT_BEND_DOWN 2
#define LEFT_BEND_UP 3
#define RIGHT_PLAY 4
#define RIGHT_CUE 5
#define RIGHT_BEND_DOWN 6
#define RIGHT_BEND_UP 7

#define ACCEL_START 30

Client *Frame::client()
{
	return j_client;
}

int prev_axis[20];

Frame::Frame(QWidget *parent, const char *name = 0, WFlags f = 0) : QWidget(parent, name, f)
{
	lv_on=true;
	rv_on=true;
	video_mode_on=false;
	left_factor=1.0f;
	right_factor=0.0f;
	left_source=0;
	right_source=0;
	left_buffer=0;
	right_buffer=0;
	frame1=frame2=0;
	metatracker=new Metatracker(this);
	blink_on=false;
	event_fd=-1;
#ifdef HAVE_LIBSDL
	display=0;
	vthread=0;

#ifdef USE_VIDEO
	int i;

	left_buffer=new unsigned char *[480];
	right_buffer=new unsigned char *[480];
	blank_buffer=new unsigned char *[480];

	for(i=0;i<480;++i)
	{
		left_buffer[i]=new unsigned char[640*3+4];
		right_buffer[i]=new unsigned char[640*3+4];
		blank_buffer[i]=new unsigned char[640*3+4];
		memset(blank_buffer[i], 0, 640*3+4);
	}

	if(video_on)
	{
		if(video_display[0])
		{
			memmove(video_display+8, video_display, strlen(video_display));
			memcpy(video_display, "DISPLAY=", 8);
			putenv(video_display);
		}
		if(SDL_Init(SDL_INIT_VIDEO) < 0)
		{
			printf("Failed to init SDL\n");
		}
		else
		{
			atexit(SDL_Quit);

			display=SDL_SetVideoMode(640, 480, 16, SDL_SWSURFACE);
			if(display)
			{
				SDL_WM_SetCaption("DJPlay video", NULL);
				vthread=new VThread();
				vthread->frame=this;
				vthread->start();
				wm_info=(void *)new SDL_SysWMinfo;
				SDL_VERSION(&((SDL_SysWMinfo*)wm_info)->version);
				int cc=SDL_GetWMInfo((SDL_SysWMinfo*)wm_info);
				(*((SDL_SysWMinfo*)wm_info)->info.x11.lock_func)();
				printf("SDL X display is %08lx, win is %08lx\n", ((SDL_SysWMinfo*)wm_info)->info.x11.display, ((SDL_SysWMinfo*)wm_info)->info.x11.window);
				(*((SDL_SysWMinfo*)wm_info)->info.x11.unlock_func)();
				Window parent_win;
				Window root_win;
				Window* child_windows;
				unsigned int num_child_windows;
				XQueryTree(((SDL_SysWMinfo*)wm_info)->info.x11.display, ((SDL_SysWMinfo*)wm_info)->info.x11.window,
						&root_win,
						&parent_win,
						&child_windows, &num_child_windows);
				XFree(child_windows);
				sdl_frame=parent_win;
				printf("SDL Frame window is %08lx\n", parent_win);
				if(video_display[0])
				{
					XUnmapWindow(((SDL_SysWMinfo*)wm_info)->info.x11.display, sdl_frame);
					XFlush(((SDL_SysWMinfo*)wm_info)->info.x11.display);
				}
			}
			else
			{
				printf("Error allocating display\n");
			}
		}
	}
#endif
#endif

	
	s_mixer=0;
	j_client=new Client();
	j_client->init("DJPlay");

	pfl_left=new Port(j_client, "pfl_left", -1, Port::Output, Port::Clear);
	pfl_right=new Port(j_client, "pfl_right", -1, Port::Output, Port::Clear);

	if(open_event_dev() >= 0)
	{
		fcntl(event_fd, F_SETFL, fcntl(event_fd, F_GETFL) | O_NONBLOCK);
		
		sn_read=new QSocketNotifier(event_fd, QSocketNotifier::Read, this);
		connect(sn_read, SIGNAL(activated(int)), this, SLOT(eventData(int)));
		sn_read->setEnabled(true);

		set_led(0, PFL_LED, 0);
		set_led(0, SYNC_LED, 0);
		set_led(0, LOOP_LED, 0);
		set_led(0, MASTERTEMPO_LED, 0);
		set_led(0, FX_LED, 0);
		set_led(0, CUELAMP_LED, 1);
		set_led(1, PFL_LED, 0);
		set_led(1, SYNC_LED, 0);
		set_led(1, LOOP_LED, 0);
		set_led(1, MASTERTEMPO_LED, 0);
		set_led(1, FX_LED, 0);
		set_led(1, CUELAMP_LED, 1);

		blink_timer=new QTimer(this);
		connect(blink_timer, SIGNAL(timeout()), this, SLOT(blink_timeout()));
		blink_timer->start(500);
	}
	else
	{
#ifdef CONTROL_UNIT
		if(config->getValue("JoystickDevice", "") != "")
		{
			joy_fd=open(config->getValue("JoystickDevice", ""), O_RDONLY);
			if(joy_fd < 0)
			{
				perror("Error opening joystick device");
			}
			else
			{
				fcntl(joy_fd, F_SETFL, fcntl(joy_fd, F_GETFL) | O_NONBLOCK);

				sn_read=new QSocketNotifier(joy_fd, QSocketNotifier::Read, this);
				connect(sn_read, SIGNAL(activated(int)), this, SLOT(joyData(int)));
				sn_read->setEnabled(true);
			}
		}
#endif
	}
	button_states[0]=0;
	button_states[1]=0;
	button_states[2]=0;
	button_states[3]=0;
	button_delta[0]=0;
	button_delta[1]=0;
	button_delta[2]=0;
	button_delta[3]=0;
	last_button[0]=-1;
	last_button[1]=-1;
	direction[0]=-1;
	direction[1]=-1;
	started[0]=0;
	started[1]=0;
	lowcount[0]=0;
	lowcount[1]=0;
	ignore[0]=2;
	ignore[1]=2;
	last_event=0;
	wheel_timer=new QTimer(this);
	connect(wheel_timer, SIGNAL(timeout()), this, SLOT(wheel_timeout()));
	key_timer=new QTimer(this);
#ifdef CONTROL_UNIT
	connect(key_timer, SIGNAL(timeout()), this, SLOT(key_timeout()));
	key_timer->start(10);
	prev_keys=0xff;
#endif
}

Frame::~Frame()
{
if(event_fd >= 0)
	close(event_fd);
event_fd=-1;
#ifdef HAVE_LIBSDL
	if(vthread)
		delete vthread;
#endif
	if(left_buffer)
	{
		int i;

		for(i=0;i<480;++i)
			delete left_buffer[i];
		delete left_buffer;
	}
	if(right_buffer)
	{
		int i;

		for(i=0;i<480;++i)
			delete right_buffer[i];
		delete right_buffer;
	}
	
	delete pfl_left;
	delete pfl_right;
	delete j_client;
	delete key_timer;
	delete metatracker;
}

void Frame::setFrames(MainWnd *f1, MainWnd *f2)
{
	frame1=f1;
	frame2=f2;
}

void Frame::joyData(int socket)
{
	struct js_event js;
	int pitch_value;
	int value;

	int button_base;
	int other_button;
	int step_value;
	int delta;
	MainWnd *curframe;

	while(read(joy_fd, (char *)&js, sizeof(js)) == sizeof(js))
	{
		js.type &= 0x7f;
#ifdef CONTROL_UNIT
		switch(js.type)
		{
		case JS_EVENT_BUTTON:
			if(js.number > 3)
				break;
			button_base=js.number >> 1;
			other_button=js.number ^ 1;
			if(button_states[js.number] != js.value)
			{
				button_states[js.number]=js.value;
				button_delta[js.number]=1;
			}
			else
				break;
			if(js.number == last_button[button_base])
				break;
			last_button[button_base]=js.number;
			if(js.value == button_states[other_button] && button_delta[other_button])
			{
				button_delta[js.number]=0;
				button_delta[other_button]=0;

				if(button_base)
					curframe=frame2;
				else
					curframe=frame1;

				wheel_timer->start(200, true);
				if(ignore[button_base] > 1)
				{
					--ignore[button_base];
					started[button_base]=0;
					direction[button_base]=js.number&1;
					last_event=js.time;
					break;
				}
				if(ignore[button_base])
					--ignore[button_base];
				if(js.time-last_event >= 200  && lowcount[button_base] < ACCEL_START/2 || !started[button_base])
				{
					started[button_base]=1;
					direction[button_base]=js.number&1;
				}
	
				step_value=500;
				if(js.time-last_event <= 30)
				{
					if(lowcount[button_base] < ACCEL_START*2)
					++lowcount[button_base];
					if(lowcount[button_base] > ACCEL_START)
						step_value*=(30-(js.time-last_event)/2+1);
				}
				else
				{
					if(lowcount[button_base] > 1)
						lowcount[button_base]-=ACCEL_START/10;
					if(lowcount[button_base] < 1)
						lowcount[button_base]=1;
				}

				if(direction[button_base])
				{
//					printf("Turn right %d last %d accel %d\n", js.number&1, js.time-last_event, lowcount[button_base]);
					curframe->step_forward(step_value);
				}
				else
				{
//					printf("Turn left %d last %d accel %d\n", js.number&1, js.time-last_event, lowcount[button_base]);
					curframe->step_back(step_value);
				}
				if(direction[button_base] != js.number&1 && js.time-last_event > 100 && lowcount[button_base] < ACCEL_START/2)
				{
					started[button_base]=0;
				}
				last_event=js.time;
			}
			break;
		case JS_EVENT_AXIS:
			if(js.value > 32560)
				js.value=32560;
			if(js.value < -32560)
				js.value=-32560;
			pitch_value=js.value/407;
			switch(js.number)
			{
			case 0:
				if(pitch_value == 0)
					set_pin(LP_PIN01);
				else
					clear_pin(LP_PIN01);
				frame1->set_base_pitch(pitch_value);
				break;
			case 1:
				if(pitch_value == 0)
					set_pin(LP_PIN14);
				else
					clear_pin(LP_PIN14);
				frame2->set_base_pitch(pitch_value);
				break;
			default:
				break;
			}
			break;
		default:
			break;
		}
#endif
	}
}

void Frame::eventData(int socket)
{
	struct input_event js;
	int pitch_value;
	int value;

	int button_base;
	int other_button;
	int step_value;
	int delta;
	MainWnd *curframe;

	while(read(event_fd, (char *)&js, sizeof(js)) == sizeof(js))
	{
		switch(js.type)
		{
			case EV_ABS:
				switch(js.code)
				{
					case 0: // Right bass
						value=(int)((double)js.value/1.275+0.4)-100;
						if(mix2)
							mix2->Low->setValue(value);
						break;
					case 1: // Right mid
						value=(int)((double)js.value/1.275+0.4)-100;
						if(mix2)
							mix2->Mid->setValue(value);
						break;
					case 2: // Crossfader
						value=((int)((double)js.value/1.275+0.4));
						if(mixer())
							mixer()->XFader->setValue(value);
						break;
					case 3: // Left bass
						value=(int)((double)js.value/1.275+0.4)-100;
						if(mix1)
							mix1->Low->setValue(value);
						break;
					case 4: // Left mid
						value=(int)((double)js.value/1.275+0.4)-100;
						if(mix1)
							mix1->Mid->setValue(value);
						break;
					case 5: // Left treble
						value=(int)((double)js.value/1.275+0.4)-100;
						if(mix1)
							mix1->High->setValue(value);
						break;
					case 6: // Right treble
						value=(int)((double)js.value/1.275+0.4)-100;
						if(mix2)
							mix2->High->setValue(value);
						break;
					case 7: // Left pitch
						pitch_value=-((int)((double)js.value/1.59375+0.4)-80);
						frame1->set_base_pitch(pitch_value);
						break;
					case 8: // Right pitch
						pitch_value=-((int)((double)js.value/1.59375+0.4)-80);
						frame2->set_base_pitch(pitch_value);
						break;
					case 9: // Left volume
						if(prev_axis[js.code]-(int)js.value < -120)
							prev_axis[js.code]+=256;
						else if(prev_axis[js.code]-(int)js.value > 120)
							prev_axis[js.code]-=265;
						if((int)js.value > prev_axis[js.code])
							delta=1;
						else
							delta=-1;
						if(mix1)
						{
							value=mix1->Gain->value();
							value-=delta;
							if(value < 0)
								value=0;
							if(value > 100)
								value=100;
							mix1->Gain->setValue(value);
						}
						break;
					case 10: // Right volume
						if(prev_axis[js.code]-(int)js.value < -120)
							prev_axis[js.code]+=256;
						else if(prev_axis[js.code]-(int)js.value > 120)
							prev_axis[js.code]-=265;
						if((int)js.value > prev_axis[js.code])
							delta=1;
						else
							delta=-1;
						if(mix1)
						{
							value=mix1->Gain->value();
							value-=delta;
							if(value < 0)
								value=0;
							if(value > 100)
								value=100;
							mix1->Gain->setValue(value);
						}
						break;
					case 11: // Left jog wheel
						if(prev_axis[js.code]-(int)js.value < -120)
							prev_axis[js.code]+=256;
						else if(prev_axis[js.code]-(int)js.value > 120)
							prev_axis[js.code]-=256;
						if((int)js.value > prev_axis[js.code])
							delta=1;
						else
							delta=-1;
						if(frame1 && frame1->get_source() && frame1->get_source()->is_playing() and frame1->is_seeking())
						{
							if(delta > 0)
								frame1->step_forward(1000);
							else
								frame1->step_back(1000);
						}
						break;
					case 12: // Right jog wheel
						if(prev_axis[js.code]-(int)js.value < -120)
							prev_axis[js.code]+=256;
						else if(prev_axis[js.code]-(int)js.value > 120)
							prev_axis[js.code]-=265;
						if((int)js.value > prev_axis[js.code])
							delta=1;
						else
							delta=-1;
						if(frame2 && frame2->get_source() && frame2->get_source()->is_playing() and frame2->is_seeking())
						{
							if(delta > 0)
								frame2->step_forward(1000);
							else
								frame2->step_back(1000);
						}
						break;
				}
				prev_axis[js.code]=js.value;
				break;
			case EV_KEY:
				js.code-=288;

				switch(js.code)
				{
					case 0:
						if(!js.value)
							break;
						switch(right_mode)
						{
							case 0:
								set_led(1, FX_LED, 0);
								break;
							case 1:
								set_led(1, CUELAMP_LED, 0);
								break;
							case 2:
								set_led(1, LOOP_LED, 0);
								break;
						}
						right_mode++;
						if(right_mode > 2)
							right_mode=0;
						switch(right_mode)
						{
							case 0:
								set_led(1, FX_LED, 1);
								break;
							case 1:
								set_led(1, CUELAMP_LED, 1);
								break;
							case 2:
								set_led(1, LOOP_LED, 1);
								break;
						}
						break;
					case 1:
						if(js.value)
							frame2->PlayButton_clicked();
						break;
					case 2:
						if(js.value)
						{
							if(frame2->get_source() && frame2->get_source()->is_playing())
								frame2->CueButton_clicked();
							else
							{
								frame2->SeekBack_pressed();
								frame2->SeekBack_released();
							}
						}
						break;
					case 4:
						if(js.value)
							frame2->SkipBack_clicked();
						break;
					case 5:
						if(js.value)
							frame2->SkipFwd_clicked();
						break;
					case 6:
						if(!js.value)
							break;
						switch(left_mode)
						{
							case 0:
								set_led(0, FX_LED, 0);
								break;
							case 1:
								set_led(0, CUELAMP_LED, 0);
								break;
							case 2:
								set_led(0, LOOP_LED, 0);
								break;
						}
						left_mode++;
						if(left_mode > 2)
							left_mode=0;
						switch(left_mode)
						{
							case 0:
								set_led(0, FX_LED, 1);
								break;
							case 1:
								set_led(0, CUELAMP_LED, 1);
								break;
							case 2:
								set_led(0, LOOP_LED, 1);
								break;
						}
						break;
					case 7:
						if(js.value)
							frame1->PlayButton_clicked();
						break;
					case 8:
						if(js.value)
						{
							if(frame1->get_source() && frame1->get_source()->is_playing())
								frame1->CueButton_clicked();
							else
							{
								frame1->SeekBack_pressed();
								frame1->SeekBack_released();
							}
						}
						break;
					case 10:
						if(js.value)
							frame1->SkipBack_clicked();
						break;
					case 11:
						if(js.value)
							frame1->SkipFwd_clicked();
						break;
					case 18:
						if(js.value)
							frame1->BendUp_pressed();
						else
							frame1->BendUp_released();
						break;
					case 19:
						if(js.value)
							frame1->BendDown_pressed();
						else
							frame1->BendDown_released();
						break;
					case 20:
						if(mix1 && js.value)
							mix1->Pfl->toggle();
						break;
					case 22:
						if(js.value)
							frame2->BendUp_pressed();
						else
							frame2->BendUp_released();
						break;
					case 23:
						if(js.value)
							frame2->BendDown_pressed();
						else
							frame2->BendDown_released();
						break;
					case 24:
						if(mix2 && js.value)
							mix2->Pfl->toggle();
						break;
				}
				break;
		}
	}
}

void Frame::wheel_timeout()
{
	if(ignore[0] == 1)
	{
		if(direction[1])
			frame1->step_forward(500);
		else
			frame1->step_back(500);
	}
	if(ignore[1] == 1)
	{
		if(direction[1])
			frame2->step_forward(500);
		else
			frame2->step_back(500);
	}
	started[0]=0;
	started[1]=0;
	ignore[0]=2;
	ignore[1]=2;
	lowcount[0]=lowcount[1]=0;
}

void Frame::key_timeout()
{
#ifdef CONTROL_UNIT
	int i;

	unsigned keys=pin_is_set(LP_DATA_PINS);
	if(keys != prev_keys)
	{
		unsigned delta=prev_keys ^ keys;
		unsigned pressed=delta&(~keys);

		for(i=0;i<8;i++)
		{
			unsigned mask=1<<i;
			if(delta&mask)
			{
				switch(i)
				{
					case LEFT_PLAY:
						if(!(pressed&mask))
							break;
						frame1->PlayButton_clicked();
						break;
					case LEFT_CUE:
						if(!(pressed&mask))
							break;
						frame1->CueButton_clicked();
						break;
					case RIGHT_PLAY:
						if(!(pressed&mask))
							break;
						frame2->PlayButton_clicked();
						break;
					case RIGHT_CUE:
						if(!(pressed&mask))
							break;
						frame2->CueButton_clicked();
						break;
					case LEFT_BEND_DOWN:
						if(!(pressed&mask))
							frame1->BendDown_released();
						else
							frame1->BendDown_pressed();
						break;
					case LEFT_BEND_UP:
						if(!(pressed&mask))
							frame1->BendUp_released();
						else
							frame1->BendUp_pressed();
						break;
					case RIGHT_BEND_DOWN:
						if(!(pressed&mask))
							frame2->BendDown_released();
						else
							frame2->BendDown_pressed();
						break;
					case RIGHT_BEND_UP:
						if(!(pressed&mask))
							frame2->BendUp_released();
						else
							frame2->BendUp_pressed();
						break;
				}
			}
		}
	}
	
	prev_keys=keys;
#endif
}

void Frame::closeEvent(QCloseEvent *e)
{
#ifdef HAVE_LIBSDL
	if(vthread)
		delete vthread;
	vthread=0;
#endif
	emit windowClosed();
}

MainWnd *Frame::left()
{
	return frame1;
}

MainWnd *Frame::right()
{
	return frame2;
}

MainWnd *Frame::player(int p)
{
	if(p < 0 || p > 1)
		return 0;
	if(!p)
		return left();
	return right();
}

Mixer *Frame::mixer()
{
	return s_mixer;
}

void Frame::setMixer(Mixer *m)
{
	s_mixer=m;
}

Playlist *Frame::playlist()
{
	return s_playlist;
}

void Frame::setPlaylist(Playlist *p)
{
	s_playlist=p;
}

void Frame::VideoXFader_valueChanged(int val)
{
	left_factor=(float)(100-val)/100.0f;
	right_factor=(float)(val)/100.0f;
	setVideoMode();
}

unsigned char **Frame::get_frame_buffer(int player)
{
	switch(player)
	{
	case 1:
		return left_buffer;
	case 2:
		return right_buffer;
	}
	return 0;
}

void Frame::enableLeftVideo(bool on)
{
	lv_on=on;
	setVideoMode();
}

void Frame::enableRightVideo(bool on)
{
	rv_on=on;
	setVideoMode();
}

void Frame::setVideoMode()
{
	if(!s_mixer)
		return;
	clear_lock.lock();
	StreamSource *s1=frame1->get_source(), *s2=frame2->get_source();
	bool lv=false, rv=false;
	if(s1 && s1->has_video() && s1->is_playing())
		lv=true;
	if(s2 && s2->has_video() && s2->is_playing())
		rv=true;
	clear_lock.unlock();

	if(s_mixer->VideoXFader->value() == 0)
		rv=false;
	if(s_mixer->VideoXFader->value() == 100)
		lv=false;

	bool on=false;
	if((lv && lv_on) || (rv && rv_on))
		on=true;
	
	if(video_mode_on != on)
	{
		if(on)
		{
#ifdef USE_VIDEO
			if(video_display[0])
			{
				app->lock();
				(*((SDL_SysWMinfo*)wm_info)->info.x11.lock_func)();
				XMapRaised(((SDL_SysWMinfo*)wm_info)->info.x11.display, sdl_frame);
				XFlush(((SDL_SysWMinfo*)wm_info)->info.x11.display);
				(*((SDL_SysWMinfo*)wm_info)->info.x11.unlock_func)();
				app->unlock();
			}
#endif
		}
		else
		{
#ifdef USE_VIDEO
			if(video_display[0])
			{
				app->lock();
	printf("Lower Video Window\n");
				(*((SDL_SysWMinfo*)wm_info)->info.x11.lock_func)();
				XUnmapWindow(((SDL_SysWMinfo*)wm_info)->info.x11.display, sdl_frame);
				XFlush(((SDL_SysWMinfo*)wm_info)->info.x11.display);
				(*((SDL_SysWMinfo*)wm_info)->info.x11.unlock_func)();
				app->unlock();
			}
#endif
		}
		video_mode_on=on;
	}
}

void Frame::new_track(int player_number)
{
}

int Frame::open_event_dev()
{
	int i;
	int fd=-1;
	char devstr[255];
	char devname[255];
	struct input_event ev;

	for(i=0;i<63;i++)
	{
		sprintf(devname, "/dev/input/event%d", i);
		fd=open(devname, O_RDWR | O_EXCL);
		if(fd >= 0)
		{
			if(ioctl(fd, EVIOCGNAME(sizeof(devstr)), devstr) < 0)
			{
				close(fd);
				continue;
			}
			if(strcmp(devstr, "Hercules Hercules DJ Console"))
			{
				close(fd);
				continue;
			}
			break;
		}
	}

	if(fd < 0)
		return -1;
	
	event_fd=fd;
	return fd;
}

void Frame::set_led(int player, int led, int on)
{
	if(led_states[led_numbers[player][led]] == on)
		return;

	if(event_fd < 0)
		return;

	struct input_event ev;

	memset(&ev, 0, sizeof(ev));
	ev.type=EV_LED;
	ev.code=led_numbers[player][led];
//	printf("led %d(%d %d) %d\n", led_numbers[player][led], player, led, on);
	led_states[led_numbers[player][led]]=on;

	if(on)
		ev.value=1;
	else
		ev.value=0;
	if(event_fd >= 0)
	{
		write(event_fd, &ev, sizeof(ev));
		usleep(50000);
	}
}

void Frame::blink_timeout()
{
	blink_on=!blink_on;

	if(!frame1 || !frame2)
		return;
	if(frame1->get_source() && frame1->get_source()->is_playing() && frame1->is_seeking())
		set_led(0, CUE_LED, blink_on);

	if(frame2->get_source() && frame2->get_source()->is_playing() && frame2->is_seeking())
		set_led(1, CUE_LED, blink_on);
}

