#include "recdata.h"
#include "recorder.h"
#include "RingBuffer.hpp"
#include "frame.h"
#include "jacklow.h"
#include <qapplication.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <time.h>
#include <qlabel.h>
#include "cfgfile.h"
#include <bitmapbutton.h>
#include <samplerate.h>

#include <stdio.h>

QWaitCondition rec_wait;
static SRC_STATE *src_state;
static SRC_DATA src_data;
static float inbuf[2048];
static float outbuf[2048];
static short intbuf[8192];

Recdata::Recdata(Recorder *i_rec)
{
	s_bc_monitor=false;
	rec=i_rec;
	fd=-1;
	do_record=false;
	s_stop=false;
	s_pipe=NULL;
	s_pipe_fp=NULL;
	s_rec_fp=NULL;
	s_recfile=0;
	int err;
	src_state=src_new(SRC_SINC_FASTEST, 2, &err);
	if(src_state)
	{
		src_data.data_in=inbuf;
		src_data.data_out=outbuf;
		src_data.end_of_input=0;
		src_data.src_ratio=44100.0 / (double)frame->client()->rate();
		src_data.input_frames=0;
	}
}

Recdata::~Recdata()
{
	if(src_state)
		src_delete(src_state);
	rec_lock.lock();
	s_stop=true;
	if(fd != -1)
		close(fd);
	rec_lock.unlock();
	rec_wait.wakeOne();
	while(!finished())
		usleep(10000);
}

void Recdata::run()
{
	do_record=false;
	int reopen_pipe=0;
	time_t reopen_at;

	while(1)
	{
		rec_wait.wait();
		rec_lock.lock();
		if(s_stop)
		{
			rec_lock.unlock();
			return;
		}
		
		RingBuffer *buffer=rec->getBuffer();
		RingBuffer *bc_buffer=rec->get_bc_buffer();

		short samps[4096];
		short bc_samps[4096];

		int max_l, max_r, avg_l, avg_r, val;
		double sum_l=0, sum_r=0;
		while(bc_buffer->read_space() >= sizeof(short)*2*1024)
		{
			max_l=max_r=val=0;
			sum_l=sum_r=0.0;
			buffer->read((char *)samps, sizeof(short)*2048);
			bc_buffer->read((char *)bc_samps, sizeof(short)*2048);
			int i;
			short *meter_samps=samps;
			if(s_bc_monitor)
				meter_samps=bc_samps;
			for(i=0;i<1024;i++)
			{
				val=meter_samps[i*2];
				if(val < 0)
					val=-val;
				sum_l+=sqrt(val);
				if(val > max_l)
					max_l=val;
				val=meter_samps[i*2+1];
				if(val < 0)
					val=-val;
				sum_r+=sqrt(val);
				if(val > max_r)
					max_r=val;
			}
			avg_l=(int)pow(sum_l/1024.0, 2.0);
			avg_r=(int)pow(sum_r/1024.0, 2.0);
			rec->setMeters((double)avg_l/32767.0, (double)avg_r/32767.0, (double)max_l/32767.0, (double)max_r/32768.0);
			if(do_record)
			{
				if(fd == -1 && s_recfile && *s_recfile)
					fd=open(s_recfile, O_WRONLY | O_APPEND | O_CREAT, 0644);
				if(fd != -1 && do_record)
				{
					write(fd, (char *)samps, sizeof(short)*2048);
				}
				if(reopen_pipe && time(0) >= reopen_at)
				{
					if(s_pipe && *s_pipe)
					{
						s_pipe_fp=popen(s_pipe, "w");
						if(s_pipe_fp)
						{
							app->lock();
							rec->LeftOnAir->show();
							rec->RightOnAir->show();
							app->unlock();
						}
					}
					reopen_pipe=0;
				}
				if(s_pipe_fp)
				{
					int i;
					src_data.input_frames*=2;
					for(i=0;i<2048;++i)
					{
						inbuf[src_data.input_frames]=(float)bc_samps[i]/32768.0;
						if(inbuf[src_data.input_frames] > 1.0)
							inbuf[src_data.input_frames]=1.0;
						++src_data.input_frames;
					}
					src_data.input_frames/=2;
					
					int cc;
					while(src_data.input_frames > 512)
					{
						src_data.output_frames=1024;
						src_data.end_of_input=0;
						src_data.data_in=inbuf;
						src_data.data_out=outbuf;
						src_process(src_state, &src_data);

						for(i=0;i<src_data.output_frames_gen*2;i++)
						{
							if(outbuf[i] > 1.0)
								outbuf[i]=1.0;
							intbuf[i]=(short)(outbuf[i]*32767.0);
						}
						cc=fwrite((char *)intbuf, 1, sizeof(short)*src_data.output_frames_gen*2, s_pipe_fp);
						if(!cc)
							break;
						src_data.input_frames-=src_data.input_frames_used;
						memmove(inbuf, &inbuf[src_data.input_frames_used*2], src_data.input_frames*2*sizeof(float));
					}

					if(!cc)
					{
						pclose(s_pipe_fp);
						s_pipe_fp=0;
						rec->LeftOnAir->hide();
						rec->RightOnAir->hide();
						if(out_relaunch)
						{
							reopen_pipe=1;
							reopen_at=time(0)+5;
						}
					}
				}
				if(s_rec_fp)
				{
					int cc=fwrite((char *)samps, 1, sizeof(short)*2048, s_rec_fp);
					if(!cc)
					{
						pclose(s_rec_fp);
						s_rec_fp=0;
					}
				}
			}
			else
			{
				if(fd != -1)
					close(fd);
				fd=-1;
			}
		}
		rec_lock.unlock();
	}
}

void Recdata::setRecord(bool on)
{
	do_record=on;
	if(s_pipe_fp)
	{
		pclose(s_pipe_fp);
		s_pipe_fp=NULL;
		rec->LeftOnAir->hide();
		rec->RightOnAir->hide();
	}
	if(s_rec_fp)
	{
		pclose(s_rec_fp);
		s_rec_fp=NULL;
	}

	if(on)
	{
		if(s_pipe && *s_pipe)
		{
			s_pipe_fp=popen(s_pipe, "w");
			if(s_pipe_fp)
			{
				app->lock();
				rec->LeftOnAir->show();
				rec->RightOnAir->show();
				app->unlock();
			}
		}
		QString rec_pipe=config->getValue("RecordPipeline", "");
		if(rec_pipe != "" && rec->RecordOn->isOn())
		{
			s_rec_fp=popen(rec_pipe, "w");
		}
	}
}

void Recdata::setOutput(const char *pipe)
{
	if(s_pipe)
		delete s_pipe;
	s_pipe=0;
	if(!pipe)
		return;
	s_pipe=new char[strlen(pipe)+1];
	strcpy(s_pipe, pipe);
}

void Recdata::setRecfile(const char *recfile)
{
	if(s_recfile)
		delete s_recfile;
	s_recfile=0;
	if(!recfile)
		return;
	s_recfile=new char[strlen(recfile)+1];
	strcpy(s_recfile, recfile);
}

void Recdata::setBcMonitor(bool on)
{
	s_bc_monitor=on;
}
