/*
 * $Id: ext.c,v 1.6 2003/03/23 15:25:14 stevecheng Exp $
 *
 * (C) 2001 Steve Cheng <stevecheng@users.sourceforge.net>
 *
 * This source compiles to db2x_xsltproc.
 *
 * See ../COPYING for the copyright status of this software.
 *
 * Derived from xsltproc.c from Daniel Veillard's libxslt.
 *
 */

#include "ext.h"

#include <stdlib.h>
#include <stdio.h>

#include <libxml/hash.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <libxslt/xsltconfig.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/extensions.h>
#include <libxslt/templates.h>
#include <libxslt/xsltutils.h>



static void
hashext_deallocator(void *payload, xmlChar *keyname)
{
	if(payload)
		xmlXPathFreeObject(payload);
}


/* 
 * Functions
 */
	
static void 
hashext_get(xmlXPathParserContextPtr ctxt, int nargs)
{
	xmlChar *key, *tablename;
	xmlXPathObjectPtr object;
	
	xmlHashTablePtr table = xsltGetExtData(
			xsltXPathGetTransformContext(ctxt),
			ctxt->context->functionURI);
	
	if (nargs != 2) {
		xmlXPathSetArityError(ctxt);
		return;
	}

	key = xmlXPathPopString(ctxt);
	if(xmlXPathCheckError(ctxt) || !key) {
		xmlXPathSetTypeError(ctxt);
		return;
	}

	tablename = xmlXPathPopString(ctxt);
	if(xmlXPathCheckError(ctxt) || !tablename) {
		xmlXPathSetTypeError(ctxt);
		xmlFree(key);
		return;
	}
	
	object = xmlHashLookup2(table, key, tablename); 

#ifdef DEBUG_PRINT
	fprintf(stderr, "hashext: get: %s, %s, %p\n", key, tablename, object);
#endif

	xmlFree(key);
	xmlFree(tablename);
	
	if(object) {
		valuePush(ctxt, xmlXPathObjectCopy(object));
	} else {
		xmlXPathReturnFalse(ctxt);
	}
}

static void 
hashext_containskey(xmlXPathParserContextPtr ctxt, int nargs)
{
	xmlChar *key, *tablename;
	xmlXPathObjectPtr object;
	int result;
	
	xmlHashTablePtr table = xsltGetExtData(
			xsltXPathGetTransformContext(ctxt),
			ctxt->context->functionURI);
	
	if (nargs != 2) {
		xmlXPathSetArityError(ctxt);
		return;
	}

	key = xmlXPathPopString(ctxt);
	if(xmlXPathCheckError(ctxt) || !key) {
		xmlXPathSetTypeError(ctxt);
		return;
	}

	tablename = xmlXPathPopString(ctxt);
	if(xmlXPathCheckError(ctxt) || !tablename) {
		xmlXPathSetTypeError(ctxt);
		xmlFree(key);
		return;
	}

	object = xmlHashLookup2(table, key, tablename);
	
	xmlFree(key);
	xmlFree(tablename);
	
	if(object) {
		xmlXPathReturnTrue(ctxt);
	} else {
		xmlXPathReturnFalse(ctxt);
	}
}

/*
 * Elements
 */

static void
hashext_put(	xsltTransformContextPtr ctxt,
		xmlNodePtr node,
		xmlNodePtr inst,
		xsltElemPreCompPtr comp)
{
	xmlHashTablePtr table = xsltGetExtData(ctxt, HASHEXT_NAMESPACE);
	xmlChar *tablename, *key, *value;
	xmlXPathObjectPtr obj_value;

	tablename = xsltEvalAttrValueTemplate(ctxt, inst,
			(const xmlChar *) "table",
			NULL);

	if(tablename == NULL) {
		xsltPrintErrorContext(ctxt, NULL, inst);
		xsltGenericError(xsltGenericErrorContext,
				"hashext:put: table name not given\n");
		return;
	}
	
	key = xsltEvalAttrValueTemplate(ctxt, inst,
			(const xmlChar *) "key",
			NULL);
	if(key == NULL) {
		xsltPrintErrorContext(ctxt, NULL, inst);
		xsltGenericError(xsltGenericErrorContext,
				"hashext:put: entry key not given\n");
		xmlFree(tablename);
		return;
	}

	value = xsltEvalAttrValueTemplate(ctxt, inst,
			(const xmlChar *) "value",
			NULL);
	if(value == NULL) {
		xsltPrintErrorContext(ctxt, NULL, inst);
		xsltGenericError(xsltGenericErrorContext,
				"hashext:put: entry value not given\n");
		xmlFree(tablename);
		xmlFree(key);
		return;
	}

	obj_value = xmlXPathNewString(value);
	
#ifdef DEBUG_PRINT
	fprintf(stderr, "hashext: put: %s, %s, %s\n", key, tablename, value);
#endif
	
	if(xmlHashUpdateEntry2(table, key, tablename, obj_value, hashext_deallocator) <= -1)
		abort();

	xmlFree(tablename);
	xmlFree(key);
	xmlFree(value);
}

static void
hashext_remove(	xsltTransformContextPtr ctxt,
		xmlNodePtr node,
		xmlNodePtr inst,
		xsltElemPreCompPtr comp)
{
	xmlHashTablePtr table = xsltGetExtData(ctxt, HASHEXT_NAMESPACE);
	xmlChar *tablename, *key;

	tablename = xsltEvalAttrValueTemplate(ctxt, inst,
			(const xmlChar *) "table",
			NULL);

	if(tablename == NULL) {
		xsltPrintErrorContext(ctxt, NULL, inst);
		xsltGenericError(xsltGenericErrorContext,
				"hashext:remove: table name not given\n");
		return;
	}
	
	key = xsltEvalAttrValueTemplate(ctxt, inst,
			(const xmlChar *) "key",
			NULL);
	if(key == NULL) {
		xsltPrintErrorContext(ctxt, NULL, inst);
		xsltGenericError(xsltGenericErrorContext,
				"hashext:remove: entry key not given\n");
		xmlFree(tablename);
		return;
	}
	
#ifdef DEBUG_PRINT
	fprintf(stderr, "hashext: remove: %s, %s\n", key, tablename);
#endif
	
	if(xmlHashRemoveEntry2(table, key, tablename, hashext_deallocator) <= -1) {
		xsltPrintErrorContext(ctxt, NULL, inst);
		xsltGenericError(xsltGenericErrorContext,
				"hashext:remove: entry not found\n");
	}

	xmlFree(tablename);
	xmlFree(key);
}

/* 
 * Init, exit functions 
 */


static void *
hashext_init(xsltTransformContextPtr ctxt,
             const xmlChar *URI)
{
	xmlHashTablePtr table;
	
	table = xmlHashCreate(256);
	if(!table) abort();

	return table;
}

static void
hashext_shutdown(xsltTransformContextPtr ctxt, const xmlChar *URI, void *data)
{
	xmlHashTablePtr table = data;
	xmlHashFree(table, hashext_deallocator);
}

void hashext_registermodule(void)
{
	xsltRegisterExtModule(HASHEXT_NAMESPACE, 
			hashext_init, hashext_shutdown);

	xsltRegisterExtModuleFunction((const xmlChar *)"get", 
			HASHEXT_NAMESPACE, hashext_get);
	xsltRegisterExtModuleFunction((const xmlChar *)"contains-key", 
			HASHEXT_NAMESPACE, hashext_containskey);
	xsltRegisterExtModuleElement((const xmlChar *)"put", 
			HASHEXT_NAMESPACE, NULL, hashext_put);
	xsltRegisterExtModuleElement((const xmlChar *)"remove", 
			HASHEXT_NAMESPACE, NULL, hashext_remove);

}
