<?xml version='1.0'?>
<!-- vim: sw=2 sta et
-->

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:hash="http://www.jclark.com/xt/java/java.util.Hashtable"
                xmlns:doc="http://nwalsh.com/xsl/documentation/1.0"
                exclude-result-prefixes="doc hash"
                version='1.0'
                xml:lang="en">

<!-- ********************************************************************
     $Id: texinode-impl-javaext.xsl,v 1.7 2003/05/24 13:52:46 stevecheng Exp $
     ********************************************************************

     &copy; 2000 Steve Cheng <stevecheng@users.sourceforge.net>

     This file is part of the docbook2X XSLT stylesheets for
     converting DocBook to Texinfo.

     Derived from files in Norman Walsh's XSL DocBook Stylesheet
     Distribution and Mark Burton's dbtotexi stylesheets.

     This code uses the XT interface to java.util.Hashtable
     to store Texinfo nodenames during processing, and hence is
     not portable verbatim.  It also uses the side effects of 
     Hashtable which may cause problems with some processors...

     ******************************************************************** -->

<!-- ==================================================================== -->
<!-- Variables -->

<!-- Texinfo nodename -> ID map -->
<xsl:variable name="texinodemap" select="hash:new()" />
<!-- ID -> Texinfo nodename map -->
<xsl:variable name="nodeteximap" select="hash:new()" />






<!-- ==================================================================== -->

<doc:template name="compute-texinfo-node-name" xmlns="">
<refpurpose>Compute (part of) the Texinfo nodename</refpurpose>
<refdescription>
<para>
This is a subroutine used by <function>get-texinfo-node-name</function>
to derive a nodename from the given node.  It checks the suggested
name for collisions with existing names.  If there is a collision, 
it prepends the parent's nodename to the suggested name.
</para>
<para>
If the suggested name is not given, it applies the texinfo-node-name
templates to find one for the given node,
</para>
<para>
This function returns the nodename <emphasis>with the filename and colon</emphasis>
prepended to it, simply to make the <function>get-texinfo-node-name</function> process 
more efficient.
</para>
<para>
This function is internal to the Java implementation because it needs to
test if a candidate name collides with an existing one.
</para>
</refdescription>
<refparameter>
<variablelist>
<varlistentry>
<term><parameter>sugname</parameter></term>
<listitem><para>
A string which is the suggested name.  If not given, regular templates
are applied.
</para></listitem>
</varlistentry>
</variablelist>
</refparameter>
</doc:template>

<xsl:template name="compute-texinfo-node-name">
  <xsl:param name="node" select="." />

  <xsl:param name="sugname">
    <xsl:apply-templates select="$node" mode="for-texinfo-node-name" />
  </xsl:param>

  <xsl:variable name="file">
    <xsl:call-template name="get-texinfo-file-name">
      <xsl:with-param name="node" select="$node" />
    </xsl:call-template>
  </xsl:variable>

  <xsl:variable name="qsugname"
                select="concat($file,':',$sugname)" />

  <xsl:choose>
    <xsl:when test="$sugname = ''">
      <xsl:call-template name="user-message">
        <xsl:with-param name="node" select="$node" />
        <xsl:with-param name="key">No readable node name; using generate-id</xsl:with-param>
      </xsl:call-template>

      <xsl:value-of select="concat($file,':',generate-id($node))" />
    </xsl:when>

    <xsl:when test="not(hash:contains-key($texinodemap,$qsugname))">
      <xsl:value-of select="$qsugname" />
    </xsl:when>

    <xsl:otherwise>
      <xsl:call-template name="user-message">
        <xsl:with-param name="node" select="$node" />
        <xsl:with-param name="key">Node name collision</xsl:with-param>
      </xsl:call-template>

      <xsl:variable name="parentnodename">
        <xsl:call-template name="get-texinfo-node-name">
          <xsl:with-param name="node" select="$node/.." />
        </xsl:call-template>
      </xsl:variable>

      <xsl:variable name="qpsugname"
                    select="concat($file,':',$parentnodename,' - ',$sugname)" />

      <xsl:choose>
        <xsl:when test="hash:contains-key($texinodemap,$qpsugname)">
          <xsl:call-template name="user-message">
            <xsl:with-param name="node" select="$node" />
            <xsl:with-param name="key">No readable node name; using generate-id</xsl:with-param>
          </xsl:call-template>

          <xsl:value-of select="concat($file,':',generate-id($node))" />
        </xsl:when>

        <xsl:otherwise>
          <xsl:value-of select="$qpsugname" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>






<doc:template name="get-texinfo-node-name" xmlns="">
<refpurpose>Find the Texinfo nodename</refpurpose>
<refdescription>
<para>
Returns the Texinfo nodename from the given node.  This nodename
is guaranteed to be unique across the target Texinfo file.
</para>
<para>
This function needs to be reimplemented for each type of extension the
stylesheet uses.
</para>
</refdescription>
<refparameter>
<variablelist>
<varlistentry>
<term><parameter>node</parameter></term>
<listitem><para>
The node to find information for.  Default is the context node.
</para></listitem>
</varlistentry>
<varlistentry>
<term><parameter>sugname</parameter></term>
<listitem><para>
If the template needs to create a new name, try to use the
suggested name instead of the default names.
</para></listitem>
</varlistentry>
</variablelist>
</refparameter>
</doc:template>

<xsl:template name="get-texinfo-node-name">
  <xsl:param name="node" select="." />
  <xsl:param name="sugname" />

  <xsl:variable name="id" select="generate-id($node)" />

  <xsl:choose>
    <xsl:when test="hash:contains-key($nodeteximap,$id)">
      <xsl:value-of select="substring-after(hash:get($nodeteximap,$id),':')" />
    </xsl:when>

    <xsl:otherwise>
      <xsl:variable name="newqname">
        <xsl:choose>
          <xsl:when test="$sugname">
            <xsl:call-template name="compute-texinfo-node-name">
              <xsl:with-param name="node" select="$node" />
              <xsl:with-param name="sugname" select="$sugname" />
            </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="compute-texinfo-node-name">
              <xsl:with-param name="node" select="$node" />
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>

      <!-- Evaluate side effects.
           Extension elements are clearly the best way to do this but
           there is unfortunately no simple standard to interface
           Java that way. -->
      <xsl:if test="concat(hash:put($nodeteximap,string($id),string($newqname)),
                           hash:put($texinodemap,string($newqname),string($id)))">
      </xsl:if>
      
      <xsl:value-of select="substring-after($newqname,':')" />
    </xsl:otherwise>

  </xsl:choose>
</xsl:template>
  
</xsl:stylesheet>

