#ifndef ContextReporter_h
#include "ContextReporter.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef UserPreferences_h
#include "UserPreferences.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef doctorj_math
#define doctorj_math
#include <math.h>
#endif

using namespace std;

using namespace doctorj;

ContextReporter::ContextReporter(ostream& os) : Reporter(os), lastFile_(NULL)
{
}

ContextReporter::~ContextReporter()
{
}

void ContextReporter::underline(char* startOfLine, char* startPos, char* endPos)
{
    indent(startOfLine, startPos, ' ');

    ostream& os = stream();

    if (startPos == endPos) {
        os << "^";
    }
    else {
        os << "<";
        for (char* p = startPos + 1; p < endPos; ++p) {
            os << "-";
        }
        os << ">";
    }
    os << endl;
}

void ContextReporter::markToEndOfLine(char* startOfLine, char* startPos)
{
    indent(startOfLine, startPos, ' ');

    ostream& os = stream();

    os << "<";
    for (char* p = startPos + 1; *p != '\n'; ++p) {
        os << "-";
    }
    os << endl;
}

void ContextReporter::markToStartPosition(char* startOfLine, char* startPos)
{
    // subtract one, to account for the '>' character
    indent(startOfLine, startPos - 1, '-');

    ostream& os = stream();

    int tw = UserPreferences::get()->tabWidth;

    // if the last character is a tab, add more dashes
    if (*startPos == '\t') {
        for (int i = 1; i < tw; ++i) {
            os << "-";
        }
    }
    os << ">" << endl;
}

void ContextReporter::writeError(const ReportedError& err)
{
    ostream& os = stream();

    if (lastFile_ != err.file) {
        os << "In \e[1m\e[7m" << err.file->name() << "\e[0m:" << endl;
        os << endl;
        lastFile_ = err.file;
    }

    // where does the error type go?

    if (err.startLine == err.lastLine) {
        writeLine(err.startLine, err.file);
        underline(err.startOfFirstLine, err.startPosition, err.lastPosition);
    }
    else {
        markToEndOfLine(err.startOfFirstLine, err.startPosition);
        for (int ln = err.startLine; ln <= err.lastLine; ++ln) {
            writeLine(ln, err.file);
        }
        markToStartPosition(err.startOfLastLine, err.lastPosition);
    }
    os << "*** " << err.type << ": " << err.message << endl;
    os << endl;
}

void ContextReporter::writeLine(int line, File* const file)
{
    // right align the line numbers.
    double l   = log10(double(line));
    int    len = 1 + int(floor(l));

    ostream& os = stream();
    
    int spaces = 6 - len;
    for (int i = 0; i < spaces; ++i) {
        os << " ";
    }
    os << line << ". " << file->fullLine(line) << endl;
}

void ContextReporter::indent(char* start, char* end, char ch) 
{
    ostream& os = stream();
    
    os << "        ";

    int tw = UserPreferences::get()->tabWidth;

    // move it over for the column, replacing tabs with spaces
    for (char* p = start; p < end; ++p) {
        if (*p == '\t') {
            for (int i = 0; i < tw; ++i) {
                os << ch;
            }
        }
        else {
            os << ch;
        }
    }
}
