#ifndef DocumentationErrors_h
#include "DocumentationErrors.h"
#endif

#ifndef Reporter_h
#include "Reporter.h"
#endif

#ifndef DocErrorRegistry_h
#include "DocErrorRegistry.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef StringEdit_h
#include "StringEdit.h"
#endif

#ifndef StringEdit_h
#include "StringEdit.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef JavadocCommentSpellCheck_h
#include "JavadocCommentSpellCheck.h"
#endif

#ifndef Java_h
#include "Java.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef JavadocTags_h
#include "JavadocTags.h"
#endif

#ifndef DocDebug_h
#include "DocDebug.h"
#endif

#ifndef DocUserPreferences_h
#include "DocUserPreferences.h"
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

using namespace doctorj;

DocumentationError::DocumentationError()
{
}

DocumentationError::DocumentationError(Reporter* const reporter, 
                                       AstItem* const item) : 
        Error(reporter, item)
{
}

DocumentationError::~DocumentationError()
{
}

ErrorRegistry* DocumentationError::getErrorRegistry() const
{
    return DocErrorRegistry::get();
}

int DocumentationError::identifier() const
{
    int lvl = level();
    int num = number();
    return lvl * 100 + num;
}

StringEdit* DocumentationError::makeInsertion(const string& defaultValue) const
{
    AstItem* what = item();
    char*    pos  = what->endPosition() + 1;

    if (defaultValue == "") {
        return NULL;
    }
    else {
        StringEdit* edit = new StringEditInsert(" " + defaultValue, pos);
        return edit;
    }
}

int DocumentationError::getIndentation(AstItem* const context) const
{
    int indentation = 0;
    char* p = context->position();
    File* sfile = context->sourceFile();
    char* start = sfile->head();
        
    while (p > start) {
        if (*(p - 1) == '\n') {
            break;
        }
        else {
            ++indentation;
            --p;
        }
    }
    return indentation;
}

REGISTER_DESCRIPTION(ErrorAuthorWithoutNameText);

ErrorAuthorWithoutNameText::ErrorAuthorWithoutNameText()
{
}

ErrorAuthorWithoutNameText::ErrorAuthorWithoutNameText(Reporter* const reporter, 
                                                       AstItem* const item) : 
        DocumentationError(reporter, item)
{
}

ErrorAuthorWithoutNameText::~ErrorAuthorWithoutNameText()
{
}

string ErrorAuthorWithoutNameText::title() const
{
    return "Author Without Name-Text";
}

void ErrorAuthorWithoutNameText::writeDescription(ostream& os) const
{
    os << "The author tag should have an associated name." << endl;
}

string ErrorAuthorWithoutNameText::message() const
{
    return JavadocTags::AUTHOR + " has no name-text";
}

int ErrorAuthorWithoutNameText::level() const
{
    return 3;
}

int ErrorAuthorWithoutNameText::number() const
{
    return 0;
}

StringEdit* ErrorAuthorWithoutNameText::getEdit()
{
    DocUserPreferences* dup = DocUserPreferences::get();
    string defAuthor = dup->defaultAuthorName;
    return makeInsertion(defAuthor);
}

REGISTER_DESCRIPTION(ErrorExceptionMismatchWithCode);

ErrorExceptionMismatchWithCode::ErrorExceptionMismatchWithCode()
{
}

ErrorExceptionMismatchWithCode::ErrorExceptionMismatchWithCode(Reporter* const reporter, 
                                                               AstItem* const item,
                                                               const string& tag)
        : DocumentationError(reporter, item), tag_(tag)
{
}

ErrorExceptionMismatchWithCode::~ErrorExceptionMismatchWithCode()
{
}

string ErrorExceptionMismatchWithCode::title() const
{
    return "Exception Mismatch with Code";
}

void ErrorExceptionMismatchWithCode::writeDescription(ostream& os) const
{
    os << "An exception was documented, but there is no such class-name in the throws" << endl;
    os << "list of the method." << endl;
}

string ErrorExceptionMismatchWithCode::message() const
{
    return tag_ + " documented for non-existing code exception";
}

int ErrorExceptionMismatchWithCode::level() const
{
    return 3;
}

int ErrorExceptionMismatchWithCode::number() const
{
    return 1;
}

StringEdit* ErrorExceptionMismatchWithCode::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();
    
    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}

REGISTER_DESCRIPTION(ErrorExceptionMisspelled);

ErrorExceptionMisspelled::ErrorExceptionMisspelled()
{
}

ErrorExceptionMisspelled::ErrorExceptionMisspelled(Reporter* const reporter, 
                                                   AstItem* const item,
                                                   const string& tag,
                                                   const string& docThrow,
                                                   const string& codeThrow) :
        DocumentationError(reporter, item),
     tag_(tag),
     docThrow_(docThrow),
     codeThrow_(codeThrow)
{
}

ErrorExceptionMisspelled::~ErrorExceptionMisspelled()
{
}

string ErrorExceptionMisspelled::title() const
{
    return "Exception Misspelled";
}

void ErrorExceptionMisspelled::writeDescription(ostream& os) const
{
    os << "An exception was documented and appears to be a misspelling of one in the" << endl;
    os << "throws list." << endl;
}

string ErrorExceptionMisspelled::message() const
{
    return tag_ + " '" + docThrow_ + "' appears a misspelling of '" + codeThrow_ + "'";
}

int ErrorExceptionMisspelled::level() const
{
    return 3;
}

int ErrorExceptionMisspelled::number() const
{
    return 2;
}

StringEdit* ErrorExceptionMisspelled::getEdit()
{
    AstItem*    what = item();
    char*       pos  = what->position();
    StringEdit* edit = new StringEditReplace(codeThrow_, pos, docThrow_.length());
    return edit;
}

REGISTER_DESCRIPTION(ErrorExceptionNotInCode);

ErrorExceptionNotInCode::ErrorExceptionNotInCode()
{
}

ErrorExceptionNotInCode::ErrorExceptionNotInCode(Reporter* const reporter, 
                                                 AstItem* const item,
                                                 const string& name) :
        DocumentationError(reporter, item), name_(name)
{
}

ErrorExceptionNotInCode::~ErrorExceptionNotInCode()
{
}

string ErrorExceptionNotInCode::title() const
{
    return "Exception not in Code";
}

void ErrorExceptionNotInCode::writeDescription(ostream& os) const
{
    os << "An exception was documented, but is not in the throws list of the code." << endl;
}

string ErrorExceptionNotInCode::message() const
{
    return JavadocTags::THROWS + " documented for method '" + name_ + "' without exceptions";
}

int ErrorExceptionNotInCode::level() const
{
    return 2;
}

int ErrorExceptionNotInCode::number() const
{
    return 2;
}

StringEdit* ErrorExceptionNotInCode::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();
    
    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}

REGISTER_DESCRIPTION(ErrorExceptionOrderNotAlphabetical);

ErrorExceptionOrderNotAlphabetical::ErrorExceptionOrderNotAlphabetical()
{
}

ErrorExceptionOrderNotAlphabetical::ErrorExceptionOrderNotAlphabetical(Reporter* const reporter, 
                                                                       AstTaggedDescribedComment* const current,
                                                                       AstItem* const item,
                                                                       const string& tag,
                                                                       const string& docThrow,
                                                                       AstTaggedDescribedComment* const previous) :
        DocumentationError(reporter, item), 
     current_(current),
     tag_(tag),
     docThrow_(docThrow),
     previous_(previous)
{
}

ErrorExceptionOrderNotAlphabetical::~ErrorExceptionOrderNotAlphabetical()
{
}

string ErrorExceptionOrderNotAlphabetical::title() const
{
    return "Exception Order not Alphabetical";
}

void ErrorExceptionOrderNotAlphabetical::writeDescription(ostream& os) const
{
    os << "Documented exceptions were not in alphabetical order." << endl;
}

string ErrorExceptionOrderNotAlphabetical::message() const
{
    return tag_ + " '" + docThrow_ + "' not in alphabetical order";
}

int ErrorExceptionOrderNotAlphabetical::level() const
{
    return 2;
}

int ErrorExceptionOrderNotAlphabetical::number() const
{
    return 3;
}

// StringEdit* ErrorExceptionOrderNotAlphabetical::getEdit()
// {
// //     AstItem* what = item();
// //     char*    from = what->position();
// //     char*    to   = what->endPosition();
// //     File*    file = what->sourceFile();

//     string current(current_->position(), 1 + current_->endPosition() - current_->position());
// //     cout << "ORD  current = '" << current << "'" << endl;
    
//     string proposed = previous_->text();
// //     cout << "ORD  proposed = '" << proposed << "'" << endl;

//     return NULL;
// }

REGISTER_DESCRIPTION(ErrorExceptionUndocumented);

ErrorExceptionUndocumented::ErrorExceptionUndocumented()
{
}

ErrorExceptionUndocumented::ErrorExceptionUndocumented(Reporter* const reporter, 
                                                       AstNoncode* const noncode,
                                                       AstName* const name,
                                                       const string& codeThrow) :
        DocumentationError(reporter, name), noncode_(noncode), codeThrow_(codeThrow)
{
}

ErrorExceptionUndocumented::~ErrorExceptionUndocumented()
{
}

string ErrorExceptionUndocumented::title() const
{
    return "Exception Undocumented";
}

void ErrorExceptionUndocumented::writeDescription(ostream& os) const
{
    os << "An exception is in the throws list of the code, but is not documented." << endl;
}

string ErrorExceptionUndocumented::message() const
{
    return "throw '" + codeThrow_ + "' not documented";
}

int ErrorExceptionUndocumented::level() const
{
    return 1;
}

int ErrorExceptionUndocumented::number() const
{
    return 0;
}

StringEdit* ErrorExceptionUndocumented::getEdit()
{
    AstJavadocComment* jc = JavaNoncode::getFirstJavadocComment(noncode_);
    DEBUG_DOCUMENTATION_ANALYZER(cout << "considering inserting into " << (void*)jc << endl);
    if (jc) {
        vector<AstTaggedComment*> cmts = jc->taggedComments();
        if (cmts.size() == 0) {
            // that's relatively easy.
            char* pos = jc->endPosition();
            while (*pos != '*' && *(pos + 1) != '/') {
                --pos;
            }
            return makeInsert(pos);
        }
        else {
            DEBUG_DOCUMENTATION_ANALYZER(cout << "there are " << cmts.size() << " tagged comments" << endl);
            // @throws (or @exception) should follow @return, and precede @see
            vector<AstTaggedComment*>::iterator it   = cmts.begin();
            vector<AstTaggedComment*>::iterator stop = cmts.end();
            JavadocTags jtags;
            int throwsIndex = jtags.indexForTag(JavadocTags::THROWS);
            char* pos = NULL;
            while (it != stop) {
                AstTaggedComment* tc = *it;
                int index = jtags.indexForTag(tc->tag());
                if (index == throwsIndex) {
                    // exceptions should be listed alphabetically
                    if (AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc)) {
                        string tgt = tdc->target();
                        DEBUG_DOCUMENTATION_ANALYZER(cout << "target is " << tgt << endl);
                        DEBUG_DOCUMENTATION_ANALYZER(cout << "codeThrow_ is " << codeThrow_ << endl);
                        if (codeThrow_ < tgt) {
                            // we've found our insertion location
                            pos = tc->position() - 1;
                            while (*pos != '*') {
                                --pos;
                            }
                            break;
                        }
                        else {
                            // we're not there yet
                            DEBUG_DOCUMENTATION_ANALYZER(cout << "not there yet" << endl);
                        }
                    }
                    
                    // cout << "UNDOC  returning NULL edit" << endl;
                    // return NULL;
                }
                else if (index >= throwsIndex) {
                    // we're beyond the throws tag index
                    DEBUG_DOCUMENTATION_ANALYZER(cout << "beyond the throws tag index" << endl);
                    pos = tc->position();
                    break;
                }
                
                ++it;
            }

            if (pos) {
            }
            else {
                pos = jc->endPosition();
                while (*pos != '*' && *(pos + 1) != '/') {
                    --pos;
                }
            }
            return makeInsert(pos);
        }
    }
    return NULL;
}

StringEditInsert* ErrorExceptionUndocumented::makeInsert(char* pos) const
{
    unsigned int nSpaces = 0;
    while (*pos != '\n') {
        // rewinding to start of line
        ++nSpaces;
        --pos;
    }
    --nSpaces;
    
    DEBUG_DOCUMENTATION_ANALYZER(cout << "UND  nSpaces = " << nSpaces << endl);

    ++pos;                      // back to the beginning (of the next line)

    DEBUG_DOCUMENTATION_ANALYZER(cout << "UND  pos = " << pos << endl);

    string repstr(nSpaces, ' ');
    repstr += "* @throws ";
    repstr += codeThrow_;
    repstr += ' ';
    DocUserPreferences* dup = DocUserPreferences::get();
    repstr += dup->defaultExceptionDescription;
    repstr += "\n";

    DEBUG_DOCUMENTATION_ANALYZER(cout << "UND  replacement string = '" << repstr << "'" << endl);

    return new StringEditInsertLine(repstr, pos);
}

REGISTER_DESCRIPTION(ErrorExceptionWithoutClassName);

ErrorExceptionWithoutClassName::ErrorExceptionWithoutClassName()
{
}

ErrorExceptionWithoutClassName::ErrorExceptionWithoutClassName(Reporter* const reporter, 
                                                               AstItem* const item, 
                                                               const string& tag) :
        DocumentationError(reporter, item), tag_(tag)
{
}

ErrorExceptionWithoutClassName::~ErrorExceptionWithoutClassName()
{
}

string ErrorExceptionWithoutClassName::title() const
{
    return "No Class Name for Exception";
}

void ErrorExceptionWithoutClassName::writeDescription(ostream& os) const
{
    os << "An exception was documented, but without an associated class name." << endl;
}

string ErrorExceptionWithoutClassName::message() const
{
    return tag_ + " has no class-name nor description";
}

int ErrorExceptionWithoutClassName::level() const
{
    return 3;
}

int ErrorExceptionWithoutClassName::number() const
{
    return 3;
}

StringEdit* ErrorExceptionWithoutClassName::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();
    
    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}


REGISTER_DESCRIPTION(ErrorExceptionWithoutDescription);

ErrorExceptionWithoutDescription::ErrorExceptionWithoutDescription()
{
}

ErrorExceptionWithoutDescription::ErrorExceptionWithoutDescription(Reporter* const reporter, 
                                                                   AstItem* const item, 
                                                                   const string& tag) :
        DocumentationError(reporter, item), tag_(tag)
{
}

ErrorExceptionWithoutDescription::~ErrorExceptionWithoutDescription()
{
}

string ErrorExceptionWithoutDescription::title() const
{
    return "No Description for Exception";
}

void ErrorExceptionWithoutDescription::writeDescription(ostream& os) const
{
    os << "An exception was documented, but without a description of the" << endl;
    os << "conditions under which the exception will be thrown." << endl;
}

string ErrorExceptionWithoutDescription::message() const
{
    return tag_ + " has no description";
}

int ErrorExceptionWithoutDescription::level() const
{
    return 3;
}

int ErrorExceptionWithoutDescription::number() const
{
    return 4;
}

StringEdit* ErrorExceptionWithoutDescription::getEdit()
{
    DocUserPreferences* dup = DocUserPreferences::get();
    if (dup->defaultExceptionDescription == "") {
        return NULL;
    }
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "EXWOD  ErrorExceptionWithoutDescription::getEdit()" << endl);
        AstItem* what = item();
        char* pos = what->endPosition() + 1;
        string desc = string(" ") + dup->defaultExceptionDescription;
        DEBUG_DOCUMENTATION_ANALYZER(cout << "EXWOD  description string = '" << what << "'" << endl);
        return new StringEditInsert(desc, pos);
    }
}


REGISTER_DESCRIPTION(ErrorFieldUndocumentedUncommented);

ErrorFieldUndocumentedUncommented::ErrorFieldUndocumentedUncommented()
{
}

ErrorFieldUndocumentedUncommented::ErrorFieldUndocumentedUncommented(Reporter* const reporter,
                                                                     AstItem* const item, 
                                                                     const string& itemType, 
                                                                     const string& errType,
                                                                     bool isPublic) :
        DocumentationError(reporter, item),
     itemType_(itemType),
     errType_(errType),
     isPublic_(isPublic)
{
}

ErrorFieldUndocumentedUncommented::~ErrorFieldUndocumentedUncommented()
{
}

string ErrorFieldUndocumentedUncommented::title() const
{
    return "Field Undocumented";
}

void ErrorFieldUndocumentedUncommented::writeDescription(ostream& os) const
{
    os << "Each field should be documented." << endl;
}

string ErrorFieldUndocumentedUncommented::message() const
{
    string msg(itemType_ + " '" + item()->text() + "' is not " + errType_);
    if (isPublic_) {
        msg = "public " + msg;
    }
    return msg;
}

int ErrorFieldUndocumentedUncommented::level() const
{
    return 1;
}

int ErrorFieldUndocumentedUncommented::number() const
{
    return 1;
}

StringEdit* ErrorFieldUndocumentedUncommented::getEdit()
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "ErrorFieldUndocumentedUncommented::getEdit" << endl);
    AstItem* ins = item();
    while (ins && !dynamic_cast<AstFieldDeclaration*>(ins)) {
        ins = ins->parent();
    }
    
    if (ins) {
        DocUserPreferences* dup = DocUserPreferences::get();
        string defDesc = dup->defaultFieldDescription;
        
        if (dup->defaultFieldDescription == "") {
            return NULL;
        }
        else {
            int indentation = getIndentation(ins);
            string doc = "/**\n" +
                string(indentation + 1, ' ')  +  "* "   + defDesc + "\n" +
                string(indentation + 1, ' ')  +  "*/\n" +
                string(indentation, ' ');
            char* pos = ins->position();
            return new StringEditInsertLine(doc, pos);
        }
    }
    else {
        return NULL;
    }
}


REGISTER_DESCRIPTION(ErrorInvalidJavadocTagClass);

ErrorInvalidJavadocTagClass::ErrorInvalidJavadocTagClass(Reporter* const reporter, 
                                                         AstItem* const item, 
                                                         const string& tag) :
        ErrorInvalidJavadocTagItem(reporter, item, tag)
{
}

ErrorInvalidJavadocTagClass::ErrorInvalidJavadocTagClass()
{
}

ErrorInvalidJavadocTagClass::~ErrorInvalidJavadocTagClass()
{
}

string ErrorInvalidJavadocTagClass::type() const
{
    return "class";
}

int ErrorInvalidJavadocTagClass::level() const
{
    return 3;
}

int ErrorInvalidJavadocTagClass::number() const
{
    return 6;
}

vector<string> ErrorInvalidJavadocTagClass::getValidTags() const
{
    return JavadocTags::getValidClassTags();
}


REGISTER_DESCRIPTION(ErrorInvalidJavadocTag);

ErrorInvalidJavadocTag::ErrorInvalidJavadocTag(Reporter* const reporter, 
                                               AstItem* const item, 
                                               const string& tag) :
        DocumentationError(reporter, item), tag_(tag)
{
}

ErrorInvalidJavadocTag::ErrorInvalidJavadocTag()
{
}

ErrorInvalidJavadocTag::~ErrorInvalidJavadocTag()
{
}

string ErrorInvalidJavadocTag::title() const
{
    return "Invalid Javadoc Tag";
}

string ErrorInvalidJavadocTag::tag() const
{
    return tag_;
}

void ErrorInvalidJavadocTag::writeDescription(ostream& os) const
{
    os << "The valid Javadoc tags are the following:" << endl;
    JavadocTags jt;
    vector<string> validTags = jt.allTags();
    EACHC(vector<string>, validTags, it) {
        string tag = *it;
        os << "    " << tag << endl;
    }
}

string ErrorInvalidJavadocTag::message() const
{
    return "'" + tag() + "' is not a valid Javadoc tag";
}

int ErrorInvalidJavadocTag::level() const
{
    return 3;
}

int ErrorInvalidJavadocTag::number() const
{
    return 5;
}


REGISTER_DESCRIPTION(ErrorInvalidJavadocTagCtor);

ErrorInvalidJavadocTagCtor::ErrorInvalidJavadocTagCtor(Reporter* const reporter, 
                                                       AstItem* const item,
                                                       const string& tag) :
        ErrorInvalidJavadocTagItem(reporter, item, tag)
{
}

ErrorInvalidJavadocTagCtor::ErrorInvalidJavadocTagCtor()
{
}

ErrorInvalidJavadocTagCtor::~ErrorInvalidJavadocTagCtor()
{
}

string ErrorInvalidJavadocTagCtor::type() const
{
    return "constructor";
}

int ErrorInvalidJavadocTagCtor::level() const
{
    return 3;
}

int ErrorInvalidJavadocTagCtor::number() const
{
    return 7;
}

vector<string> ErrorInvalidJavadocTagCtor::getValidTags() const
{
    return JavadocTags::getValidConstructorTags();
}


REGISTER_DESCRIPTION(ErrorInvalidJavadocTagField);

ErrorInvalidJavadocTagField::ErrorInvalidJavadocTagField(Reporter* const reporter, 
                                                         AstItem* const item, 
                                                         const string& tag) :
        ErrorInvalidJavadocTagItem(reporter, item, tag)
{
}

ErrorInvalidJavadocTagField::ErrorInvalidJavadocTagField()
{
}

ErrorInvalidJavadocTagField::~ErrorInvalidJavadocTagField()
{
}

string ErrorInvalidJavadocTagField::type() const
{
    return "field";
}

int ErrorInvalidJavadocTagField::level() const
{
    return 3;
}

int ErrorInvalidJavadocTagField::number() const
{
    return 8;
}

vector<string> ErrorInvalidJavadocTagField::getValidTags() const
{
    return JavadocTags::getValidFieldTags();
}


REGISTER_DESCRIPTION(ErrorInvalidJavadocTagInterface);

ErrorInvalidJavadocTagInterface::ErrorInvalidJavadocTagInterface(Reporter* const reporter, 
                                                                 AstItem* const item, 
                                                                 const string& tag) :
        ErrorInvalidJavadocTagItem(reporter, item, tag)
{
}

ErrorInvalidJavadocTagInterface::ErrorInvalidJavadocTagInterface()
{
}

ErrorInvalidJavadocTagInterface::~ErrorInvalidJavadocTagInterface()
{
}

string ErrorInvalidJavadocTagInterface::type() const
{
    return "interface";
}

string ErrorInvalidJavadocTagInterface::article() const
{
    return "an";
}

int ErrorInvalidJavadocTagInterface::level() const
{
    return 3;
}

int ErrorInvalidJavadocTagInterface::number() const
{
    return 9;
}

vector<string> ErrorInvalidJavadocTagInterface::getValidTags() const
{
    return JavadocTags::getValidInterfaceTags();
}


ErrorInvalidJavadocTagItem::ErrorInvalidJavadocTagItem(Reporter* const reporter, 
                                                       AstItem* const item, 
                                                       const string& tag) :
        ErrorInvalidJavadocTag(reporter, item, tag)
{
}

ErrorInvalidJavadocTagItem::ErrorInvalidJavadocTagItem()
{
}

ErrorInvalidJavadocTagItem::~ErrorInvalidJavadocTagItem()
{
}

string ErrorInvalidJavadocTagItem::title() const
{
    string what = type();
    what[0] = toupper(what[0]);
    return "Invalid Javadoc Tag for " + what;
}

string ErrorInvalidJavadocTagItem::article() const
{
    return "a";
}

void ErrorInvalidJavadocTagItem::writeDescription(ostream& os) const
{
    os << "The valid Javadoc tags for "
       << article() << " " << type()
       << " are the following:" << endl;
    vector<string> validTags = getValidTags();
    EACHC(vector<string>, validTags, it) {
        string tag = *it;
        os << "    " << tag << endl;
    }
}

string ErrorInvalidJavadocTagItem::message() const
{
    return "'" + tag() + "' is not a valid Javadoc tag for " + article() + " " + type();
}

int ErrorInvalidJavadocTagItem::level() const
{
    return 3;
}

int ErrorInvalidJavadocTagItem::number() const
{
    return 10;
}

StringEdit* ErrorInvalidJavadocTagItem::getEdit()
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "ErrorInvalidJavadocTagItem::getEdit()" << endl);

    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();
    
    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}


REGISTER_DESCRIPTION(ErrorInvalidJavadocTagMethod);

ErrorInvalidJavadocTagMethod::ErrorInvalidJavadocTagMethod(Reporter* const reporter, 
                                                           AstItem* const item, 
                                                           const string& tag) :
        ErrorInvalidJavadocTagItem(reporter, item, tag)
{
}

ErrorInvalidJavadocTagMethod::ErrorInvalidJavadocTagMethod()
{
}

ErrorInvalidJavadocTagMethod::~ErrorInvalidJavadocTagMethod()
{
}

string ErrorInvalidJavadocTagMethod::type() const
{
    return "method";
}

int ErrorInvalidJavadocTagMethod::level() const
{
    return 3;
}

int ErrorInvalidJavadocTagMethod::number() const
{
    return 11;
}

vector<string> ErrorInvalidJavadocTagMethod::getValidTags() const
{
    return JavadocTags::getValidMethodTags();
}


REGISTER_DESCRIPTION(ErrorMisorderedTags);

ErrorMisorderedTags::ErrorMisorderedTags()
{
}

ErrorMisorderedTags::ErrorMisorderedTags(Reporter* const reporter, 
                                         AstJavadocComment* const jc,
                                         const vector<AstTaggedComment*>& current,
                                         const vector<AstTaggedComment*>& ideal) :
        DocumentationError(reporter, jc), comment_(jc), current_(current), ideal_(ideal)
{
}
        
ErrorMisorderedTags::~ErrorMisorderedTags()
{
}

string ErrorMisorderedTags::title() const
{
    return "Misordered Tags";
}

void ErrorMisorderedTags::writeDescription(ostream& os) const
{
    os << "The tags are not in the proper order for the given type." << endl;
    // should this be broken down by type?
    JavadocTags jt;
    os << "The order of tags (for all types) is:" << endl;
    for (int i = jt.firstIndex(), last = jt.lastIndex(); i <= last; ++i) {
        vector<string> tags;
        jt.tagsAtIndex(i, &tags);
        bool isFirst = true;
        EACH(vector<string>, tags, it) {
            os << "    ";
            if (!isFirst) {
                os <<", ";
                isFirst = false;
            }
            os << *it << endl;
        }
    }
}

string ErrorMisorderedTags::message() const
{
    return "tags are not in the proper order";
}

int ErrorMisorderedTags::level() const
{
    return 2;
}

int ErrorMisorderedTags::number() const
{
    return 4;
}

StringEdit* ErrorMisorderedTags::getEdit()
{
    return NULL;

    // maybe some day, the following will work ...

    string istr;

    // so what do we have to do to get the current tags into the ideal order?

    StringEditMulti* edits = new StringEditMulti();

    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM  current" << endl);
    for (int i = 0; i < current_.size(); ++i) {
        AstTaggedComment* current = current_[i];
        DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM  current[" << i << "] = '" << current->text() << "'" << endl);
    }

    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM  ideal" << endl);
    for (int i = 0; i < ideal_.size(); ++i) {
        AstTaggedComment* ideal = ideal_[i];
        DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM  ideal[" << i << "] = '" << ideal->text() << "'" << endl);
    }

    for (int ci = 0; ci < current_.size(); ++ci) {
        AstTaggedComment* actual = current_[ci];
        AstTaggedComment* wanted = ideal_[ci];
        DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM  current_[" << ci << "]: " << current_[ci]->text() << endl);

        if (actual == wanted) {
            DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM      actual == wanted" << endl);
        }
        else {
            DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM      actual != wanted" << endl);
            for (int ii = 0; ii < ideal_.size(); ++ii) {

                DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM          ideal_[" << ii << "]: " << ideal_[ii]->text() << endl);

                AstTaggedComment* ideal = ideal_[ii];
                if (actual == ideal) {
                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM              actual matches ideal" << endl);
                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM                  current[" << ii << "] = '" << current_[ii]->text() << "'" << endl);
                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM                  ideal[" << ci << "]   = '" << ideal->text() << "'" << endl);

                    AstTaggedComment* current = current_[ii];

                    char* astart = actual->position();
                    char* aend = actual->endPosition() + 1;

                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM                  moving actual from " << ci << " to " << ii << endl);
                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM                  astart '" << string(astart, aend - astart) << "'" << endl);

                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM                  current '" << current->position() - 1 << "'" << endl);
                    
                    edits->add(new StringEditMoveAfter(astart, aend - astart, current->position()));

                    break;
                }
                else {
                    DEBUG_DOCUMENTATION_ANALYZER(cout << "MMMMM              actual does not match ideal" << endl);
                }
            }
        }
//         if (ci == 1) {
//             break;
//         }
    }

    for (int ci = 0; ci < current_.size(); ++ci) {
        AstTaggedComment* cur = current_[ci];
        AstTaggedComment* idl = ideal_[ci];
        char* curstart = cur->position();
        char* curend = cur->endPosition() + 1;
        char* idlpos = idl->position() - 1;
        DEBUG_DOCUMENTATION_ANALYZER(cout << "EMSTG  moving '" << string(curstart, curend - curstart) << "' to '" << string(idlpos, 20) << "'" << endl);
        edits->add(new StringEditMoveAfter(curstart, curend - curstart, idlpos));
    }

    AstTaggedComment* cur = current_[0];
    AstTaggedComment* idl = ideal_[0];
    char* curstart = cur->position();
    char* curend = cur->endPosition() + 1;
    char* idlpos = idl->position() - 1;
    DEBUG_DOCUMENTATION_ANALYZER(cout << "EMSTG  moving '" << curstart << "'" << endl);
    edits->add(new StringEditMoveAfter(curstart, curend - curstart, idlpos));

    return edits;
}


REGISTER_DESCRIPTION(ErrorMisspelledJavadocTag);

ErrorMisspelledJavadocTag::ErrorMisspelledJavadocTag()
{
}

ErrorMisspelledJavadocTag::ErrorMisspelledJavadocTag(Reporter* const reporter, 
                                                     AstItem* const item,
                                                     const string& tag,
                                                     const vector<string>& matches) :
        DocumentationError(reporter, item), tag_(tag), matches_(matches)
{
}

ErrorMisspelledJavadocTag::~ErrorMisspelledJavadocTag()
{
}

string ErrorMisspelledJavadocTag::title() const
{
    return "Misspelled Javadoc Tag";
}

void ErrorMisspelledJavadocTag::writeDescription(ostream& os) const
{
    os << "The valid Javadoc tags are the following:" << endl;
    // should this be broken down by type?
    JavadocTags jt;
    for (int i = jt.firstIndex(), last = jt.lastIndex(); i <= last; ++i) {
        vector<string> tags;
        jt.tagsAtIndex(i, &tags);
        EACH(vector<string>, tags, it) {
            os << "    " << *it << endl;
        }
    }
}

string ErrorMisspelledJavadocTag::message() const
{
    string msg("'" + tag_ + "' appears to match ");

    int nMatches = matches_.size();

    if (nMatches > 1) {
        for (int i = 0; i < nMatches - 1; ++i) {
            msg += matches_[i];
            if (nMatches != 2) {
                msg += ", ";
            }
        }
        
        msg += " and ";
    }
    msg += matches_[nMatches - 1];
    
    return msg;
}

int ErrorMisspelledJavadocTag::level() const
{
    return 3;
}

int ErrorMisspelledJavadocTag::number() const
{
    return 12;
}

StringEdit* ErrorMisspelledJavadocTag::getEdit()
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "ErrorMisspelledJavadocTag::getEdit()" << endl);

    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();

    int nMatches = matches_.size();
    if (nMatches == 1) {
        string match = matches_[0];
        StringEdit* edit = new StringEditReplace(match, from, to - from + 1);
        return edit;
    }
    else {
        return NULL;
    }
}


REGISTER_DESCRIPTION(ErrorMisspelledWord);

ErrorMisspelledWord::ErrorMisspelledWord()
{
}

ErrorMisspelledWord::ErrorMisspelledWord(Reporter* const reporter, 
                                         AstItem* const item,
                                         const string& word,
                                         const vector<string>& matches) :
        DocumentationError(reporter, item), word_(word), matches_(matches)
{
}

ErrorMisspelledWord::~ErrorMisspelledWord()
{
}

string ErrorMisspelledWord::title() const
{
    return "Misspelled Word in Comment";
}

void ErrorMisspelledWord::writeDescription(ostream& os) const
{
    os << "Word does not match any in the dictionaries." << endl;
}

string ErrorMisspelledWord::message() const
{
    string msg("'" + word_ + "' not found. ");
    int nMatches = matches_.size();
    if (nMatches <= 0) {
        msg += "No close matches.";
    }
    else if (nMatches == 1) {
        msg += "Closest match: " + matches_[0] + ".";
    }
    else {
        msg += "Closest matches: " + StringUtilities::join(", ", matches_.begin(), matches_.end() - 1) + " and " + *(matches_.end() - 1) + ".";
    }
    return msg;
}

int ErrorMisspelledWord::level() const
{
    return 2;
}

int ErrorMisspelledWord::number() const
{
    return 9;
}

static bool isValid(const string& word) 
{
    return word.length() > 0;
}

StringEdit* ErrorMisspelledWord::getEdit()
{
    // The usage of "escond" hides the couts from being mixed up with my usual
    // debugging output
    AstItem* word = item();
    char* start = word->position();
    char* end   = word->endPosition();

    string chosenWord = "";

    int nMatches = matches_.size();
    /* escond */ cout << "closest matching words:" << endl;
    for (int i = 0; i < matches_.size(); ++i) {
        /* escond */ cout << "    " << i << ": " << matches_[i] << endl;
    }
    // cout << "--" << endl;
    /* escond */ cout << "    a: <accept and add word>" << endl;
    /* escond */ cout << "    w: <accept, but do not add>" << endl;
    /* escond */ cout << "    x: <enter another word>" << endl;
    //cout << endl;
    /* escond */ cout << "    choice?> " << flush;
    string answer;
    getline(cin, answer);

//     DEBUG_DOCUMENTATION_ANALYZER(cout << "answer = '" << answer << "'" << endl);

    if (answer == "" || answer == "w") {
        // nothing
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "no answer" << endl);
    }
    else if (answer == "a") {
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "accepting word" << endl);

        // add it to the list of OK words
        JavadocCommentSpellCheck* jcsc = JavadocCommentSpellCheck::get(reporter());
        jcsc->addWord(word_);
    }
    else if (answer == "x") {
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "adding word" << endl);

        /* escond */ cout << "    word> " << flush;
        getline(cin, chosenWord);

//         DEBUG_DOCUMENTATION_ANALYZER(cout << "word = '" << chosenWord << "'" << endl);
        
        if (isValid(chosenWord)) {
            // let's add this to the list of known words
            JavadocCommentSpellCheck* jcsc = JavadocCommentSpellCheck::get(reporter());
            jcsc->addWord(chosenWord);
        }
        else {
//             DEBUG_DOCUMENTATION_ANALYZER(cout << "word is not valid!" << endl);
        }
    }
    else if (isdigit(answer[0])) {
        int i;
        StringUtilities::fromString(answer, i);
        if (i >= 0 && i < matches_.size()) {
            chosenWord = matches_[i];
        }
        else {
            // invalid -- should we ignore this, or make them try again?
        }
    }
    else {
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "invalid answer '" << answer << "'" << endl);
    }

    if (chosenWord == "") {
        return NULL;
    }
    else {
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "-------------------------" << endl);
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "chosenWord      = " << chosenWord << endl);
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "start           = " << (void*)start << endl);
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "end             = " << (void*)end << endl);
//         DEBUG_DOCUMENTATION_ANALYZER(cout << "end - start + 1 = " << (int)(end - start + 1) << endl);
        StringEditReplace* edit = new StringEditReplace(chosenWord, start, end - start + 1);
        return edit;
    }
}


REGISTER_DESCRIPTION(ErrorMultipleVersionTags);

ErrorMultipleVersionTags::ErrorMultipleVersionTags()
{
}

ErrorMultipleVersionTags::ErrorMultipleVersionTags(Reporter* const reporter, 
                                                   AstItem* const item) 
        :  DocumentationError(reporter, item)
{
}

ErrorMultipleVersionTags::~ErrorMultipleVersionTags()
{
}

string ErrorMultipleVersionTags::title() const
{
    return "Multiple Version Tags";
}

void ErrorMultipleVersionTags::writeDescription(ostream& os) const
{
    os << "The version tag should be used only once for an item." << endl;
}

string ErrorMultipleVersionTags::message() const
{
    return JavadocTags::VERSION_ + " used multiple times";
}

int ErrorMultipleVersionTags::level() const
{
    return 2;
}

int ErrorMultipleVersionTags::number() const
{
    return 5;
}

StringEdit* ErrorMultipleVersionTags::getEdit()
{
    AstItem* what = item();
    File*    file = what->sourceFile();
    char*    from = what->position();
    char*    to   = what->endPosition();
    
    return new StringEditDeleteLines(from, to, file->head(), file->end());
}


REGISTER_DESCRIPTION(ErrorNoComment);

ErrorNoComment::ErrorNoComment()
{
}

ErrorNoComment::ErrorNoComment(Reporter* const reporter, 
                               AstItem* const item,
                               const string& type) :
        ErrorUndocumented(reporter, item, type)
{
}

ErrorNoComment::~ErrorNoComment()
{
}

string ErrorNoComment::title() const
{
    return "Uncommented Code";
}

int ErrorNoComment::number() const
{
    return 2;
}


REGISTER_DESCRIPTION(ErrorNoCommentForPublic);

ErrorNoCommentForPublic::ErrorNoCommentForPublic()
{
}

ErrorNoCommentForPublic::ErrorNoCommentForPublic(Reporter* const reporter, 
                                                 AstItem* const item,
                                                 const string& type) :
        ErrorUndocumented(reporter, item, type)
{
}

ErrorNoCommentForPublic::~ErrorNoCommentForPublic()
{
}

string ErrorNoCommentForPublic::title() const
{
    return "Uncommented Public Code";
}

void ErrorNoCommentForPublic::writeDescription(ostream& os) const
{
    os << "The given article of public code should be commented" << endl;
}

string ErrorNoCommentForPublic::message() const
{
    string msg = "public " + ErrorUndocumented::message();
    return msg;
}

int ErrorNoCommentForPublic::number() const
{
    return 3;
}


REGISTER_DESCRIPTION(ErrorNoDeprecatedText);

ErrorNoDeprecatedText::ErrorNoDeprecatedText()
{
}

ErrorNoDeprecatedText::ErrorNoDeprecatedText(Reporter* const reporter, 
                                             AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorNoDeprecatedText::~ErrorNoDeprecatedText()
{
}

string ErrorNoDeprecatedText::title() const
{
    return "No Text for Deprecated Tag";
}

void ErrorNoDeprecatedText::writeDescription(ostream& os) const
{
    os << "The @deprecated tag has no associated text to describe why the" << endl;
    os << "item is deprecated, and what the programmer should use instead." << endl;
}

string ErrorNoDeprecatedText::message() const
{
    return JavadocTags::DEPRECATED + " has no text";
}

int ErrorNoDeprecatedText::level() const
{
    return 3;
}

int ErrorNoDeprecatedText::number() const
{
    return 13;
}


REGISTER_DESCRIPTION(ErrorNoJavadocComment);

ErrorNoJavadocComment::ErrorNoJavadocComment()
{
}

ErrorNoJavadocComment::ErrorNoJavadocComment(Reporter* const reporter, 
                                             AstItem* const item,
                                             const string& type) :
        ErrorUndocumented(reporter, item, type)
{
}

ErrorNoJavadocComment::~ErrorNoJavadocComment()
{
}

string ErrorNoJavadocComment::title() const
{
    return "Undocumented Code";
}

void ErrorNoJavadocComment::writeDescription(ostream& os) const
{
    os << "The given article of code should have a Javadoc-formatted comment." << endl;
}

int ErrorNoJavadocComment::number() const
{
    return 4;
}


REGISTER_DESCRIPTION(ErrorNoJavadocCommentForPublic);

ErrorNoJavadocCommentForPublic::ErrorNoJavadocCommentForPublic()
{
}

ErrorNoJavadocCommentForPublic::ErrorNoJavadocCommentForPublic(Reporter* const reporter, 
                                                               AstItem* const item,
                                                               const string& type) :
        ErrorUndocumented(reporter, item, type)
{
}

ErrorNoJavadocCommentForPublic::~ErrorNoJavadocCommentForPublic()
{
}

string ErrorNoJavadocCommentForPublic::title() const
{
    return "Undocumented Public Code";
}

void ErrorNoJavadocCommentForPublic::writeDescription(ostream& os) const
{
    os << "The given article of public code should have a Javadoc-formatted comment." << endl;
}

string ErrorNoJavadocCommentForPublic::message() const
{
    string msg = "public " + ErrorUndocumented::message();
    return msg;
}

int ErrorNoJavadocCommentForPublic::number() const
{
    return 5;
}

REGISTER_DESCRIPTION(ErrorNoParameterDescription);

ErrorNoParameterDescription::ErrorNoParameterDescription()
{
}

ErrorNoParameterDescription::ErrorNoParameterDescription(Reporter* const reporter, 
                                                         AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorNoParameterDescription::~ErrorNoParameterDescription()
{
}

string ErrorNoParameterDescription::title() const
{
    return "No Parameter Description";
}

void ErrorNoParameterDescription::writeDescription(ostream& os) const
{
    os << "A parameter tag and name were documented, but no description was provided." << endl;
}

string ErrorNoParameterDescription::message() const
{
    return JavadocTags::PARAM + " has no description";
}

int ErrorNoParameterDescription::level() const
{
    return 3;
}

int ErrorNoParameterDescription::number() const
{
    return 14;
}

StringEdit* ErrorNoParameterDescription::getEdit()
{
    DocUserPreferences* dup = DocUserPreferences::get();
    string def = dup->defaultParameterDescription;
    return makeInsertion(def);
}


REGISTER_DESCRIPTION(ErrorNoParameterName);

ErrorNoParameterName::ErrorNoParameterName()
{
}

ErrorNoParameterName::ErrorNoParameterName(Reporter* const reporter, 
                                           AstItem* const item,
                                           const string& paramName) :
        DocumentationError(reporter, item), paramName_(paramName)
{
}

ErrorNoParameterName::~ErrorNoParameterName()
{
}

string ErrorNoParameterName::title() const
{
    return "No Parameter Name";
}

void ErrorNoParameterName::writeDescription(ostream& os) const
{
    os << "The parameter tag was used, but without an associated parameter name." << endl;
}

string ErrorNoParameterName::message() const
{
    return JavadocTags::PARAM + " has no param-name or description";
}

int ErrorNoParameterName::level() const
{
    return 3;
}

int ErrorNoParameterName::number() const
{
    return 15;
}

StringEdit* ErrorNoParameterName::getEdit()
{
    DocUserPreferences* dup = DocUserPreferences::get();
    string estr = string(" ") + paramName_ + " " + dup->defaultParameterDescription;
    AstItem* what = item();
    char* pos = what->endPosition() + 1;
    StringEdit* edit = new StringEditInsertLine(estr, pos);
    return edit;
}


REGISTER_DESCRIPTION(ErrorNoPeriodEndingSummarySentence);

ErrorNoPeriodEndingSummarySentence::ErrorNoPeriodEndingSummarySentence()
{
}

ErrorNoPeriodEndingSummarySentence::ErrorNoPeriodEndingSummarySentence(Reporter* const reporter,
                                                                       AstItem* const sentence) :
        DocumentationError(reporter, sentence)
{
}

ErrorNoPeriodEndingSummarySentence::~ErrorNoPeriodEndingSummarySentence()
{
}

string ErrorNoPeriodEndingSummarySentence::title() const
{
    return "No Period at End of Summary Sentence";
}

void ErrorNoPeriodEndingSummarySentence::writeDescription(ostream& os) const
{
    os << "There should be a period at the end of the the summary sentence," << endl;
    os << "which is also known as the title. The lack of a period may indicate" << endl;
    os << "that the sentence is improperly formatted." << endl;
}

string ErrorNoPeriodEndingSummarySentence::message() const
{
    return "summary sentence does not end with period";
}

int ErrorNoPeriodEndingSummarySentence::level() const
{
    return 2;
}

int ErrorNoPeriodEndingSummarySentence::number() const
{
    return 6;
}

StringEdit* ErrorNoPeriodEndingSummarySentence::getEdit()
{
    AstItem* what = item();
    char*    end  = what->endPosition();

    DEBUG_DOCUMENTATION_ANALYZER(cout << "ENPSS  description string = '" << what << "'" << endl);
    StringEdit* edit = new StringEditInsert(".", end + 1);
    return edit;
}


REGISTER_DESCRIPTION(ErrorNoSeeReference);

ErrorNoSeeReference::ErrorNoSeeReference()
{
}

ErrorNoSeeReference::ErrorNoSeeReference(Reporter* const reporter, 
                                         AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorNoSeeReference::~ErrorNoSeeReference()
{
}

string ErrorNoSeeReference::title() const
{
    return "No Reference for See Tag";
}

void ErrorNoSeeReference::writeDescription(ostream& os) const
{
    os << "The @see tag has no associated reference to describe the item" << endl;
    os << "that should be seen." << endl;
}

string ErrorNoSeeReference::message() const
{
    return JavadocTags::SEE + " has no reference";
}

int ErrorNoSeeReference::level() const
{
    return 3;
}

int ErrorNoSeeReference::number() const
{
    return 16;
}


REGISTER_DESCRIPTION(ErrorNoSinceText);

ErrorNoSinceText::ErrorNoSinceText()
{
}

ErrorNoSinceText::ErrorNoSinceText(Reporter* const reporter, 
                                   AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorNoSinceText::~ErrorNoSinceText()
{
}

string ErrorNoSinceText::title() const
{
    return "No Text for Since Tag";
}

void ErrorNoSinceText::writeDescription(ostream& os) const
{
    os << "The @since tag has no associated text to describe when the item" << endl;
    os << "was written." << endl;
}

string ErrorNoSinceText::message() const
{
    return JavadocTags::SINCE + " has no text";
}

int ErrorNoSinceText::level() const
{
    return 3;
}

int ErrorNoSinceText::number() const
{
    return 20;
}


REGISTER_DESCRIPTION(ErrorNoSummarySentence);

ErrorNoSummarySentence::ErrorNoSummarySentence()
{
}

ErrorNoSummarySentence::ErrorNoSummarySentence(Reporter* const reporter, 
                                               AstJavadocComment* const jdc) 
        : DocumentationError(reporter, jdc)
{
}

ErrorNoSummarySentence::~ErrorNoSummarySentence()
{
}

string ErrorNoSummarySentence::title() const
{
    return "No Summary Sentence";
}

void ErrorNoSummarySentence::writeDescription(ostream& os) const
{
    os << "The given article of code should have a descriptive comment" << endl;
    os << "There should be a summary sentence, which is also known as the title." << endl;
    os << "This sentence is used in the overview of the component. The lack of a" << endl;
    os << "summary sentence will make the documentation less useful." << endl;
}

string ErrorNoSummarySentence::message() const
{
    return "no summary sentence";
}

int ErrorNoSummarySentence::level() const
{
    return 2;
}

int ErrorNoSummarySentence::number() const
{
    return 7;
}


REGISTER_DESCRIPTION(ErrorParameterDataTypeUsed);

ErrorParameterDataTypeUsed::ErrorParameterDataTypeUsed()
{
}

ErrorParameterDataTypeUsed::ErrorParameterDataTypeUsed(Reporter* const reporter, 
                                                       AstItem* const item,
                                                       const string& docName,
                                                       const string& dataType,
                                                       const string& codeName) :
        DocumentationError(reporter, item),
     docName_(docName),
     dataType_(dataType),
     codeName_(codeName)
{
}

ErrorParameterDataTypeUsed::~ErrorParameterDataTypeUsed()
{
}

string ErrorParameterDataTypeUsed::title() const
{
    return "Parameter Data Type Used";
}

void ErrorParameterDataTypeUsed::writeDescription(ostream& os) const
{
    os << "Instead of the parameter name being used in the documentation, the data type" << endl;
    os << "was used instead." << endl;
}

string ErrorParameterDataTypeUsed::message() const
{
    return JavadocTags::PARAM + " '" + docName_ + "' references parameter data type '" + 
        dataType_ + "', not parameter name '" + codeName_ + "'";
}

int ErrorParameterDataTypeUsed::level() const
{
    return 3;
}

int ErrorParameterDataTypeUsed::number() const
{
    return 21;
}

StringEdit* ErrorParameterDataTypeUsed::getEdit()
{
    AstItem* datatype = item();
    char* start = datatype->position();
    char* end   = datatype->endPosition();
    
    StringEditReplace* edit = new StringEditReplace(codeName_, start, end - start + 1);
    return edit;
}


REGISTER_DESCRIPTION(ErrorParameterDocumentationMismatchWithCode);

ErrorParameterDocumentationMismatchWithCode::ErrorParameterDocumentationMismatchWithCode()
{
}

ErrorParameterDocumentationMismatchWithCode::~ErrorParameterDocumentationMismatchWithCode()
{
}

string ErrorParameterDocumentationMismatchWithCode::title() const
{
    return "Parameter Documentation Mismatch With Code";
}

void ErrorParameterDocumentationMismatchWithCode::writeDescription(ostream& os) const
{
    os << "The documented parameter name does not match the one in the code." << endl;
}

string ErrorParameterDocumentationMismatchWithCode::message() const
{
    return JavadocTags::PARAM + " name does not match the argument in the code";
}

int ErrorParameterDocumentationMismatchWithCode::level() const
{
    return 3;
}

int ErrorParameterDocumentationMismatchWithCode::number() const
{
    return 22;
}


REGISTER_DESCRIPTION(ErrorParameterMisspelled);

ErrorParameterMisspelled::ErrorParameterMisspelled()
{
}

ErrorParameterMisspelled::ErrorParameterMisspelled(Reporter* const reporter, 
                                                   AstItem* const item,
                                                   const string& docName,
                                                   const string& codeName) :
        DocumentationError(reporter, item),
     docName_(docName),
     codeName_(codeName)
{
}

ErrorParameterMisspelled::~ErrorParameterMisspelled()
{
}

string ErrorParameterMisspelled::title() const
{
    return "Parameter Misspelled";
}

void ErrorParameterMisspelled::writeDescription(ostream& os) const
{
    os << "The documented parameter name appears to be a misspelling of the parameter" << endl;
    os << "in the code." << endl;
}

string ErrorParameterMisspelled::message() const
{
    return JavadocTags::PARAM + " '" + docName_ + "' appears a misspelling of '" + codeName_ + "'";
}

int ErrorParameterMisspelled::level() const
{
    return 3;
}

int ErrorParameterMisspelled::number() const
{
    return 23;
}

StringEdit* ErrorParameterMisspelled::getEdit()
{
    AstItem* param = item();
    char* start = param->position();
    char* end   = param->endPosition();

    StringEditReplace* edit = new StringEditReplace(codeName_, start, end - start + 1);
    return edit;
}


REGISTER_DESCRIPTION(ErrorParameterNotInCode);

ErrorParameterNotInCode::ErrorParameterNotInCode()
{
}

ErrorParameterNotInCode::ErrorParameterNotInCode(Reporter* const reporter, 
                                                 AstItem* const item, 
                                                 const string& name) :
        DocumentationError(reporter, item), name_(name)
{
}

ErrorParameterNotInCode::~ErrorParameterNotInCode()
{
}

string ErrorParameterNotInCode::title() const
{
    return "Parameter Not In Code";
}

void ErrorParameterNotInCode::writeDescription(ostream& os) const
{
    os << "No such parameter was found for the given documented one." << endl;
}

string ErrorParameterNotInCode::message() const
{
    return JavadocTags::PARAM + " '" + name_ + "' documented for missing argument";
}

int ErrorParameterNotInCode::level() const
{
    return 3;
}

int ErrorParameterNotInCode::number() const
{
    return 24;
}

StringEdit* ErrorParameterNotInCode::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();

    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}


REGISTER_DESCRIPTION(ErrorParameterOutOfOrderAndMispelled);

ErrorParameterOutOfOrderAndMispelled::ErrorParameterOutOfOrderAndMispelled()
{
}

ErrorParameterOutOfOrderAndMispelled::ErrorParameterOutOfOrderAndMispelled(Reporter* const reporter, 
                                                                           AstItem* const item, 
                                                                           const string& docName, 
                                                                           const string& codeName) :
        DocumentationError(reporter, item),
     docName_(docName),
     codeName_(codeName)
{
}

ErrorParameterOutOfOrderAndMispelled::~ErrorParameterOutOfOrderAndMispelled()
{
}

string ErrorParameterOutOfOrderAndMispelled::title() const
{
    return "Parameter Out Of Order And Mispelled";
}

void ErrorParameterOutOfOrderAndMispelled::writeDescription(ostream& os) const
{
    os << "The documented parameter appears to be a misspelling of a parameter in the code" << endl;
    os << "that is not in the same order as the documentation." << endl;
}

string ErrorParameterOutOfOrderAndMispelled::message() const 
{
    return JavadocTags::PARAM + " '" + docName_ +
        "' appears a misspelling of '" + codeName_
        + "', which is not in the same order as the documentation";
}

int ErrorParameterOutOfOrderAndMispelled::level() const
{
    return 3;
}

int ErrorParameterOutOfOrderAndMispelled::number() const
{
    return 26;
}


REGISTER_DESCRIPTION(ErrorParameterOutOfOrder);

ErrorParameterOutOfOrder::ErrorParameterOutOfOrder()
{
}

ErrorParameterOutOfOrder::ErrorParameterOutOfOrder(Reporter* const reporter, 
                                                   AstItem* const item, 
                                                   const string& name) :
        DocumentationError(reporter, item), name_(name)
{
}

ErrorParameterOutOfOrder::~ErrorParameterOutOfOrder()
{
}

string ErrorParameterOutOfOrder::title() const
{
    return "Parameter Out Of Order";
}

void ErrorParameterOutOfOrder::writeDescription(ostream& os) const
{
    os << "The documented parameter is not in the same order as the parameters" << endl;
    os << "in the code." << endl;
}

string ErrorParameterOutOfOrder::message() const 
{
    return JavadocTags::PARAM + " '" + name_ + "' is not in the same order as the code";
}

int ErrorParameterOutOfOrder::level() const
{
    return 3;
}

int ErrorParameterOutOfOrder::number() const
{
    return 25;
}


REGISTER_DESCRIPTION(ErrorParameterRepeatedInDocumentation);

ErrorParameterRepeatedInDocumentation::ErrorParameterRepeatedInDocumentation()
{
}

ErrorParameterRepeatedInDocumentation::ErrorParameterRepeatedInDocumentation(Reporter* const reporter, 
                                                                             AstItem* const item,
                                                                             const string& name) 
        :  DocumentationError(reporter, item), name_(name)
{
}

ErrorParameterRepeatedInDocumentation::~ErrorParameterRepeatedInDocumentation()
{
}

string ErrorParameterRepeatedInDocumentation::title() const
{
    return "Parameter Repeated In Documentation";
}

void ErrorParameterRepeatedInDocumentation::writeDescription(ostream& os) const
{
    os << "The parameter was documented multiple times in the documentation." << endl;
}

string ErrorParameterRepeatedInDocumentation::message() const
{
    return JavadocTags::PARAM + " '" + name_ + "' is already documented";
}

int ErrorParameterRepeatedInDocumentation::level() const
{
    return 3;
}

int ErrorParameterRepeatedInDocumentation::number() const
{
    return 27;
}

StringEdit* ErrorParameterRepeatedInDocumentation::getEdit()
{
    AstItem* what = item();
    File*    file = what->sourceFile();
    char*    from = what->position();
    char*    to   = what->endPosition();
    
    return new StringEditDeleteLines(from, to, file->head(), file->end());
}


REGISTER_DESCRIPTION(ErrorParametersDocumentedButNotInCode);

ErrorParametersDocumentedButNotInCode::ErrorParametersDocumentedButNotInCode()
{
}

ErrorParametersDocumentedButNotInCode::ErrorParametersDocumentedButNotInCode(Reporter* const reporter, 
                                                                             AstItem* const item, 
                                                                             const string& type) :
        DocumentationError(reporter, item), type_(type)
{
}

ErrorParametersDocumentedButNotInCode::~ErrorParametersDocumentedButNotInCode()
{
}

string ErrorParametersDocumentedButNotInCode::title() const
{
    return "Parameters Documented But Not In Code";
}

void ErrorParametersDocumentedButNotInCode::writeDescription(ostream& os) const
{
    os << "One or more parameters were documented, yet the method takes no parameters." << endl;
}

string ErrorParametersDocumentedButNotInCode::message() const
{
    return JavadocTags::PARAM + " documented for " + type_ + " without arguments";
}

int ErrorParametersDocumentedButNotInCode::level() const
{
    return 3;
}

int ErrorParametersDocumentedButNotInCode::number() const
{
    return 28;
}

StringEdit* ErrorParametersDocumentedButNotInCode::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();

    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}


REGISTER_DESCRIPTION(ErrorParameterUndocumented);

ErrorParameterUndocumented::ErrorParameterUndocumented()
{
}

ErrorParameterUndocumented::ErrorParameterUndocumented(Reporter* const reporter, 
                                                       AstItem* const item, 
                                                       const string& argName) :
        DocumentationError(reporter, item), argName_(argName)
{
}

ErrorParameterUndocumented::~ErrorParameterUndocumented()
{
}

string ErrorParameterUndocumented::title() const
{
    return "Parameter Undocumented";
}

void ErrorParameterUndocumented::writeDescription(ostream& os) const
{
    os << "The parameter is not documented." << endl;
}

string ErrorParameterUndocumented::message() const
{
    return "parameter '" + argName_ + "' not documented";
}

int ErrorParameterUndocumented::level() const
{
    return 1;
}

int ErrorParameterUndocumented::number() const
{
    return 6;
}


REGISTER_DESCRIPTION(ErrorReturnDocumentedForMethodReturningVoid);

ErrorReturnDocumentedForMethodReturningVoid::ErrorReturnDocumentedForMethodReturningVoid()
{
}

ErrorReturnDocumentedForMethodReturningVoid::ErrorReturnDocumentedForMethodReturningVoid(Reporter* const reporter, 
                                                                                         AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorReturnDocumentedForMethodReturningVoid::~ErrorReturnDocumentedForMethodReturningVoid()
{
}

string ErrorReturnDocumentedForMethodReturningVoid::title() const
{
    return "Return Documented For Method Returning Void";
}

void ErrorReturnDocumentedForMethodReturningVoid::writeDescription(ostream& os) const
{
    os << "A return tag is inappropriate for usage with a method that returns void." << endl;
}

string ErrorReturnDocumentedForMethodReturningVoid::message() const
{
    return JavadocTags::RETURN + " documented on void method";
}

int ErrorReturnDocumentedForMethodReturningVoid::level() const
{
    return 3;
}

int ErrorReturnDocumentedForMethodReturningVoid::number() const
{
    return 29;
}

StringEdit* ErrorReturnDocumentedForMethodReturningVoid::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();

    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}


REGISTER_DESCRIPTION(ErrorReturnTypeUsedAsTarget);

ErrorReturnTypeUsedAsTarget::ErrorReturnTypeUsedAsTarget()
{
}

ErrorReturnTypeUsedAsTarget::ErrorReturnTypeUsedAsTarget(Reporter* const reporter, 
                                                         AstItem* const item, 
                                                         const string& retType) :
        DocumentationError(reporter, item), retType_(retType)
{
}

ErrorReturnTypeUsedAsTarget::~ErrorReturnTypeUsedAsTarget()
{
}

string ErrorReturnTypeUsedAsTarget::title() const
{
    return "Return Type Used As Target";
}

void ErrorReturnTypeUsedAsTarget::writeDescription(ostream& os) const
{
    os << "The return tag does not take a target, and the first word of the description" << endl;
    os << "for the return was the return type of the associated method." << endl;
}

string ErrorReturnTypeUsedAsTarget::message() const
{
    return "Return type '" + retType_ + "' used as documentation target";
}

int ErrorReturnTypeUsedAsTarget::level() const
{
    return 3;
}

int ErrorReturnTypeUsedAsTarget::number() const
{
    return 30;
}

StringEdit* ErrorReturnTypeUsedAsTarget::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    int      len  = retType_.length();
    char*    to   = from + len;
    
//     if (*to == '\n') {
//         //--to;
//     }
//     else {
        while (*to != '\n' && isspace(*to)) {
            ++to;
        }
//     }
    
    return new StringEditDelete(from, to);
}


REGISTER_DESCRIPTION(ErrorReturnUndocumented);

ErrorReturnUndocumented::ErrorReturnUndocumented()
{
}

ErrorReturnUndocumented::ErrorReturnUndocumented(Reporter* const reporter, 
                                                 AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorReturnUndocumented::~ErrorReturnUndocumented()
{
}

string ErrorReturnUndocumented::title() const
{
    return "Return Undocumented";
}

void ErrorReturnUndocumented::writeDescription(ostream& os) const
{
    os << "A method whose return type is not void should have an associated description" << endl;
    os << "of the value it returns." << endl;
}

string ErrorReturnUndocumented::message() const
{
    return "return not documented";
}

int ErrorReturnUndocumented::level() const
{
    return 1;
}

int ErrorReturnUndocumented::number() const
{
    return 7;
}


REGISTER_DESCRIPTION(ErrorReturnUsedWithConstructor);

ErrorReturnUsedWithConstructor::ErrorReturnUsedWithConstructor(Reporter* const reporter, 
                                                               AstItem* const item) : 
        DocumentationError(reporter, item)
{
}

ErrorReturnUsedWithConstructor::ErrorReturnUsedWithConstructor()
{
}

ErrorReturnUsedWithConstructor::~ErrorReturnUsedWithConstructor()
{
}

string ErrorReturnUsedWithConstructor::title() const
{
    return "Return Used With Constructor";
}

void ErrorReturnUsedWithConstructor::writeDescription(ostream& os) const
{
    os << "A return tag is inappropriate for usage with a constructor." << endl;
}

string ErrorReturnUsedWithConstructor::message() const
{
    return JavadocTags::RETURN + " documented for constructor";
}

int ErrorReturnUsedWithConstructor::level() const
{
    return 3;
}

int ErrorReturnUsedWithConstructor::number() const
{
    return 31;
}

StringEdit* ErrorReturnUsedWithConstructor::getEdit()
{
    AstItem* what = item();
    char*    from = what->position();
    char*    to   = what->endPosition();
    File*    file = what->sourceFile();

    StringEdit* edit = new StringEditDeleteLines(from, to, file->head(), file->end());
    return edit;
}


REGISTER_DESCRIPTION(ErrorReturnWithoutDescription);

ErrorReturnWithoutDescription::ErrorReturnWithoutDescription()
{
}

ErrorReturnWithoutDescription::ErrorReturnWithoutDescription(Reporter* const reporter,
                                                             AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorReturnWithoutDescription::~ErrorReturnWithoutDescription()
{
}

string ErrorReturnWithoutDescription::title() const
{
    return "Return Without Description";
}

void ErrorReturnWithoutDescription::writeDescription(ostream& os) const
{
    os << "A return tag should have an associated description." << endl;
}

string ErrorReturnWithoutDescription::message() const
{
    return JavadocTags::RETURN + " has no description";
}

int ErrorReturnWithoutDescription::level() const
{
    return 3;
}

int ErrorReturnWithoutDescription::number() const
{
    return 32;
}


REGISTER_DESCRIPTION(ErrorSerialDataWithoutDescription);

ErrorSerialDataWithoutDescription::ErrorSerialDataWithoutDescription()
{
}

ErrorSerialDataWithoutDescription::ErrorSerialDataWithoutDescription(Reporter* const reporter, 
                                                           AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorSerialDataWithoutDescription::~ErrorSerialDataWithoutDescription()
{
}

string ErrorSerialDataWithoutDescription::title() const
{
    return "No Serial Data Description";
}

void ErrorSerialDataWithoutDescription::writeDescription(ostream& os) const
{
    os << "The @serialData tag should have a data description." << endl;
}

string ErrorSerialDataWithoutDescription::message() const
{
    return JavadocTags::SERIALDATA + " has no data-description";
}

int ErrorSerialDataWithoutDescription::level() const
{
    return 3;
}

int ErrorSerialDataWithoutDescription::number() const
{
    return 17;
}


REGISTER_DESCRIPTION(ErrorSerialFieldWithoutDataDescription);

ErrorSerialFieldWithoutDataDescription::ErrorSerialFieldWithoutDataDescription()
{
}

ErrorSerialFieldWithoutDataDescription::ErrorSerialFieldWithoutDataDescription(Reporter* const reporter, 
                                                                               AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorSerialFieldWithoutDataDescription::~ErrorSerialFieldWithoutDataDescription()
{
}

string ErrorSerialFieldWithoutDataDescription::title() const
{
    return "Serial Field Without Field Description";
}

void ErrorSerialFieldWithoutDataDescription::writeDescription(ostream& os) const
{
    os << "The @serialField tag should be have an associated field description." << endl;
}

string ErrorSerialFieldWithoutDataDescription::message() const
{
    return JavadocTags::SERIALFIELD + " has no field-description";
}

int ErrorSerialFieldWithoutDataDescription::level() const
{
    return 1;
}

int ErrorSerialFieldWithoutDataDescription::number() const
{
    return 8;
}


REGISTER_DESCRIPTION(ErrorSerialFieldWithoutNameTypeOrDescription);

ErrorSerialFieldWithoutNameTypeOrDescription::ErrorSerialFieldWithoutNameTypeOrDescription()
{
}

ErrorSerialFieldWithoutNameTypeOrDescription::ErrorSerialFieldWithoutNameTypeOrDescription(Reporter* const reporter, 
                                                                                 AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorSerialFieldWithoutNameTypeOrDescription::~ErrorSerialFieldWithoutNameTypeOrDescription()
{
}

string ErrorSerialFieldWithoutNameTypeOrDescription::title() const
{
    return "No Serial Field Name, Type, Or Description";
}

void ErrorSerialFieldWithoutNameTypeOrDescription::writeDescription(ostream& os) const
{
    os << "The @serialField tag should have a field name, type, and description." << endl;
}

string ErrorSerialFieldWithoutNameTypeOrDescription::message() const
{
    return JavadocTags::SERIALFIELD + " has no field-name, field-type, nor field-description";
}

int ErrorSerialFieldWithoutNameTypeOrDescription::level() const
{
    return 3;
}

int ErrorSerialFieldWithoutNameTypeOrDescription::number() const
{
    return 18;
}


REGISTER_DESCRIPTION(ErrorSerialFieldWithoutTypeOrDescription);

ErrorSerialFieldWithoutTypeOrDescription::ErrorSerialFieldWithoutTypeOrDescription()
{
}

ErrorSerialFieldWithoutTypeOrDescription::ErrorSerialFieldWithoutTypeOrDescription(Reporter* const reporter, 
                                                                         AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorSerialFieldWithoutTypeOrDescription::~ErrorSerialFieldWithoutTypeOrDescription()
{
}

string ErrorSerialFieldWithoutTypeOrDescription::title() const
{
    return "No Serial Field Type Or Description";
}

void ErrorSerialFieldWithoutTypeOrDescription::writeDescription(ostream& os) const
{
    os << "The @serialField tag should have a field type and description." << endl;
}

string ErrorSerialFieldWithoutTypeOrDescription::message() const
{
    return JavadocTags::SERIALFIELD + " has no field-type nor field-description";
}

int ErrorSerialFieldWithoutTypeOrDescription::level() const
{
    return 3;
}

int ErrorSerialFieldWithoutTypeOrDescription::number() const
{
    return 19;
}


REGISTER_DESCRIPTION(ErrorSerialWithoutDescription);

ErrorSerialWithoutDescription::ErrorSerialWithoutDescription()
{
}

ErrorSerialWithoutDescription::ErrorSerialWithoutDescription(Reporter* const reporter, 
                                                         AstItem* const item) : 
        DocumentationError(reporter, item)
{
}

ErrorSerialWithoutDescription::~ErrorSerialWithoutDescription()
{
}

string ErrorSerialWithoutDescription::title() const
{
    return "Serial Undescribed";
}

void ErrorSerialWithoutDescription::writeDescription(ostream& os) const
{
    os << "Each @serial should be described." << endl;
}

string ErrorSerialWithoutDescription::message() const
{
    return JavadocTags::SERIAL + " has no field-description";
}

int ErrorSerialWithoutDescription::level() const
{
    return 1;
}

int ErrorSerialWithoutDescription::number() const
{
    return 9;
}


REGISTER_DESCRIPTION(ErrorShortSummarySentence);

ErrorShortSummarySentence::ErrorShortSummarySentence()
{
}

ErrorShortSummarySentence::ErrorShortSummarySentence(Reporter* const reporter, 
                                                     AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorShortSummarySentence::~ErrorShortSummarySentence()
{
}

string ErrorShortSummarySentence::title() const
{
    return "Short Summary Sentence";
}

void ErrorShortSummarySentence::writeDescription(ostream& os) const
{
    os << "The summary sentence should be of a minimal length. A summary sentence" << endl;
    os << "that is too short may not be sufficiently descriptive." << endl;
}

string ErrorShortSummarySentence::message() const
{
    return "short summary sentence";
}

int ErrorShortSummarySentence::level() const
{
    return 2;
}

int ErrorShortSummarySentence::number() const
{
    return 8;
}


static void addUnlessEmpty(string* const doc, 
                           const string& tag, 
                           const string& value,
                           int indentation)
{
    if (value != "") {
        (*doc) += string(indentation + 1, ' ');
        (*doc) += "* ";
        (*doc) += tag;
        (*doc) += " ";
        (*doc) += value;
        (*doc) += "\n";
    }
}

ErrorUndocumented::ErrorUndocumented()
{
}

ErrorUndocumented::ErrorUndocumented(Reporter* const reporter, 
                                     AstItem* const item,
                                     const string& type) :
        DocumentationError(reporter, item), type_(type)
{
}

ErrorUndocumented::~ErrorUndocumented()
{
}

void ErrorUndocumented::writeDescription(ostream& os) const
{
    os << "The given item should be documented." << endl;
}

string ErrorUndocumented::message() const
{
    return type_ + " '" + item()->text() + "' is not documented";
}

int ErrorUndocumented::level() const
{
    return 1;
}

string ErrorUndocumented::makeDocumentation(int indentation, 
                                            const string& defaultDescription) const
{
    string doc = "/**\n"  +
        string(indentation + 1, ' ')  +  "* "    + defaultDescription + "\n" +
        string(indentation + 1, ' ')  +  "*\n";

    return doc;
}

StringEdit* ErrorUndocumented::documentTypeDeclaration(AstItem* const typedec,
                                                       const string& defaultDescription) const
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "generating documentation for type declaration (class/interface)" << endl);

    if (defaultDescription == "") {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "no default description for " << typedec << endl);
        return NULL;
    }
    else {
        int indentation = getIndentation(typedec);

        string doc = makeDocumentation(indentation, defaultDescription);

        DocUserPreferences* dup = DocUserPreferences::get();

        addUnlessEmpty(&doc,
                       JavadocTags::AUTHOR, 
                       dup->defaultAuthorName,
                       indentation);

        addUnlessEmpty(&doc, 
                       JavadocTags::VERSION_, 
                       dup->defaultVersion,
                       indentation);

        doc += string(indentation + 1, ' ') + "*/\n";
        doc += string(indentation, ' ');

        char* pos = typedec->position();

        return new StringEditInsertLine(doc, pos);
    }
}

void ErrorUndocumented::addDocumentation(string* const doc,
                                         int indentation, 
                                         AstFormalParameterList* const params,
                                         AstThrowsNameList* const throwslist) const
{
    DocUserPreferences* dup = DocUserPreferences::get();
    
    string indspc = string(indentation + 1, ' ');

    if (params) {
        int pcount = params->getFormalParameterCount();
        DEBUG_DOCUMENTATION_ANALYZER(cout << "parameter count = " << pcount << endl);
            
        int nparams = params->getFormalParameterCount();
        for (int i = 0; i < nparams; ++i) {
            AstFormalParameter* fp = params->getFormalParameter(i);
            string nm = fp->name();
            DEBUG_DOCUMENTATION_ANALYZER(cout << "processing param #" << i << ": " << nm << endl);
            (*doc) += indspc + "* @param " + nm + " " + dup->defaultParameterDescription + "\n";
        }
    }
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "no parameters" << endl);
    }

    DEBUG_DOCUMENTATION_ANALYZER(cout << "throws list = " << (void*)throwslist << endl);

    if (throwslist) {
        AstNameList* namelist = throwslist->getNameList();
            
        int nthrows = namelist->getNameCount();
        DEBUG_DOCUMENTATION_ANALYZER(cout << "throws count = " << nthrows << endl);
        
        for (int i = 0; i < nthrows; ++i) {
            AstName* name = namelist->getName(i);
            string nm = name->fullName();
            DEBUG_DOCUMENTATION_ANALYZER(cout << "processing exception #" << i << ": " << nm << endl);
            (*doc) += indspc + "* @throws " + nm + " " + dup->defaultExceptionDescription + "\n";
        }
    }
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "no throws" << endl);
    }
}

StringEdit* ErrorUndocumented::documentMethod(AstItem* const it, AstItem* const context) const
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "generating documentation for method" << endl);

    DocUserPreferences* dup = DocUserPreferences::get();

    if (dup->defaultMethodDescription == "") {
        return NULL;
    }
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "item = " << it->text() << endl);

        AstItem* hdritem = context->parent();
        AstMethodHeader* header = dynamic_cast<AstMethodHeader*>(hdritem);

        DEBUG_DOCUMENTATION_ANALYZER(cout << "context = " << header->text() << endl);
        
        int indentation = getIndentation(header);

        DEBUG_DOCUMENTATION_ANALYZER(cout << "indentation = " << indentation << endl);

        string doc = makeDocumentation(indentation, dup->defaultMethodDescription);

        AstMethodDeclarator* method = dynamic_cast<AstMethodDeclarator*>(context);

        DEBUG_DOCUMENTATION_ANALYZER(cout << "method = " << (void*)method << endl);

        AstFormalParameterList* params = method->getFormalParameterList();

        AstThrowsNameList* throwslist = header->getThrowsNameList();

        addDocumentation(&doc, indentation, params, throwslist);

        doc += string(indentation + 1, ' ') + "*/\n";
        doc += string(indentation, ' ');

        char* pos = header->position();

        return new StringEditInsertLine(doc, pos);
    }
}

StringEdit* ErrorUndocumented::documentField(AstItem* const it, AstItem* const context) const
{
    // For some crazy reason, this isn't necessary. The crazy reason is that
    // undocumented fields have their own Error.
    return NULL;
}

StringEdit* ErrorUndocumented::documentConstructor(AstItem* const it, AstItem* const context) const
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "generating documentation for constructor" << endl);

    DocUserPreferences* dup = DocUserPreferences::get();
    
    if (dup->defaultConstructorDescription == "") {
        return NULL;
    }
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "item = " << it->text() << endl);
        DEBUG_DOCUMENTATION_ANALYZER(cout << "item.type = " << it->type() << endl);
        DEBUG_DOCUMENTATION_ANALYZER(cout << "context = " << context->text() << endl);
        DEBUG_DOCUMENTATION_ANALYZER(cout << "context.type = " << context->type() << endl);

        AstConstructorDeclarator* ctordtor = dynamic_cast<AstConstructorDeclarator*>(context);

        DEBUG_DOCUMENTATION_ANALYZER(cout << "ctordtor = " << (void*)ctordtor << endl);

        AstConstructorDeclaration* ctordtn = dynamic_cast<AstConstructorDeclaration*>(context->parent());

        DEBUG_DOCUMENTATION_ANALYZER(cout << "ctordtn = " << (void*)ctordtn << endl);

        int indentation = getIndentation(ctordtn);

        DEBUG_DOCUMENTATION_ANALYZER(cout << "indentation = " << indentation << endl);

        string doc = makeDocumentation(indentation, dup->defaultConstructorDescription);

        AstFormalParameterList* params = ctordtor->getFormalParameterList();

        AstThrowsNameList* throwslist = ctordtn->getThrowsNameList();

        addDocumentation(&doc, indentation, params, throwslist);

        doc += string(indentation + 1, ' ') + "*/\n";
        doc += string(indentation, ' ');

        char* pos = ctordtn->position();

        return new StringEditInsertLine(doc, pos);
    }
}

StringEdit* ErrorUndocumented::getEdit()
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "ErrorUndocumented::getEdit()" << endl);

    DocUserPreferences* dup = DocUserPreferences::get();
    AstItem* it = item();
    AstItem* context = it->parent();

    DEBUG_DOCUMENTATION_ANALYZER(cout << "context = " << context->type() << endl);

    if (type_ == "method") {
        return documentMethod(it, context);
    }        
    else if (type_ == "field") {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "generating documentation for field" << endl);
        return documentField(it, context);
    }        
    else if (type_ == "constructor") {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "generating documentation for constructor" << endl);
        return documentConstructor(it, context);
    }        
    else if (type_ == "class") {
        DocUserPreferences* dup = DocUserPreferences::get();
        return documentTypeDeclaration(context, dup->defaultClassDescription);
    }
    else if (type_ == "interface") {
        DocUserPreferences* dup = DocUserPreferences::get();
        return documentTypeDeclaration(context, dup->defaultInterfaceDescription);
    }
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "undocumented type_ = " << type_ << endl);
    }

    return NULL;
}

REGISTER_DESCRIPTION(ErrorVersionWithoutText);

ErrorVersionWithoutText::ErrorVersionWithoutText()
{
}

ErrorVersionWithoutText::ErrorVersionWithoutText(Reporter* const reporter, 
                                                 AstItem* const item) :
        DocumentationError(reporter, item)
{
}

ErrorVersionWithoutText::~ErrorVersionWithoutText()
{
}

string ErrorVersionWithoutText::title() const
{
    return "Version Without Text";
}

void ErrorVersionWithoutText::writeDescription(ostream& os) const
{
    os << "The version tag should have an associated version." << endl;
}

string ErrorVersionWithoutText::message() const
{
    return JavadocTags::VERSION_ + " has no version-text";
}

int ErrorVersionWithoutText::level() const
{
    return 3;
}

int ErrorVersionWithoutText::number() const
{
    return 35;
}

StringEdit* ErrorVersionWithoutText::getEdit()
{
    DocUserPreferences* dup = DocUserPreferences::get();
    string defVersion = dup->defaultVersion;
    return makeInsertion(defVersion);
}
