#ifndef NameChecker_h
#include "NameChecker.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef Reporter_h
#include "Reporter.h"
#endif

#ifndef Error_h
#include "Error.h"
#endif

#ifndef ErrorNameStandard_h
#include "ErrorNameStandard.h"
#endif

#ifndef RegExp_h
#include "RegExp.h"
#endif

#ifndef UserPreferences_h
#include "UserPreferences.h"
#endif

using namespace doctorj;

NameChecker::NameChecker(Reporter* const rep,
                         AstVariableDeclaratorList* const vdl,
                         bool isStatic,
                         bool isFinal,
                         AstLeaf* const access) :
        reporter_(rep),
     variableList_(vdl),
     isStatic_(isStatic),
     isFinal_(isFinal),
     access_(access)
{
}

NameChecker::~NameChecker()
{
}


AstVariableDeclaratorList* NameChecker::getVariableDeclaratorList() const
{
    return variableList_;
}

Reporter* NameChecker::getReporter() const
{
    return reporter_;
}

bool NameChecker::isStatic() const
{
    return isStatic_;
}

bool NameChecker::isFinal() const
{
    return isFinal_;
}

AstLeaf* NameChecker::getAccess() const
{
    return access_;
}

template <class ErrorType>
void NameChecker::runCheck(const string& pattern) const
{
    Reporter*               rep   = getReporter();
    AstVariableDeclaratorList* vars  = getVariableDeclaratorList();
    int                     nVars = vars->getVariableDeclaratorCount();
    RegExp                  re(pattern);
    
    for (int i = 0; i < nVars; ++i) {
        AstVariableDeclarator* vd = vars->getVariableDeclarator(i);
        string              nm = vd->name();
        if (!re.match(nm)) {
            ErrorType err(rep, vd, vd->name(), pattern);
            err.process();
        }
    }
}

enum AccessType { PUBLIC_ACCESS, PROTECTED_ACCESS, PACKAGE_ACCESS, PRIVATE_ACCESS };

static AccessType getAccessType(AstLeaf* const acc)
{
    if (dynamic_cast<AstPublic*>(acc)) {
        return PUBLIC_ACCESS;
    }
    else if (dynamic_cast<AstProtected*>(acc)) {
        return PROTECTED_ACCESS;
    }
    else if (dynamic_cast<AstPrivate*>(acc)) {
        return PRIVATE_ACCESS;
    }
    else {
        return PACKAGE_ACCESS;
    }
}

void NameChecker::check() const
{
    // cout << "NameChecker::check()" << endl;
    // check variable name format

    bool             stat        = isStatic();
    bool             fnl         = isFinal();
    AccessType       accType     = getAccessType(getAccess());
    UserPreferences* prefs       = UserPreferences::get();
    const int        NUM_FORMATS = 4;
    string           formats[NUM_FORMATS];
    UserPreferences::get()->getFormats(isStatic(), isFinal(), formats, NUM_FORMATS);

    if (isStatic()) {
        if (isFinal()) {
            switch (accType) {
                case PUBLIC_ACCESS:
                    runCheck<ErrorStaticFinalPublicFieldName>(prefs->formatStaticFinalPublic);
                    break;
                case PROTECTED_ACCESS:
                    runCheck<ErrorStaticFinalProtectedFieldName>(prefs->formatStaticFinalProtected);
                    break;
                case PACKAGE_ACCESS:
                    runCheck<ErrorStaticFinalPackageFieldName>(prefs->formatStaticFinalPackage);
                    break;
                case PRIVATE_ACCESS:
                    runCheck<ErrorStaticFinalPrivateFieldName>(prefs->formatStaticFinalPrivate);
                    break;
                default:
                    cerr << "unknown access type: " << accType << endl;
            }
        }
        else {
            switch (accType) {
                case PUBLIC_ACCESS:
                    runCheck<ErrorStaticNonfinalPublicFieldName>(prefs->formatStaticNonfinalPublic);
                    break;
                case PROTECTED_ACCESS:
                    runCheck<ErrorStaticNonfinalProtectedFieldName>(prefs->formatStaticNonfinalProtected);
                    break;
                case PACKAGE_ACCESS:
                    runCheck<ErrorStaticNonfinalPackageFieldName>(prefs->formatStaticNonfinalPackage);
                    break;
                case PRIVATE_ACCESS:
                    runCheck<ErrorStaticNonfinalPrivateFieldName>(prefs->formatStaticNonfinalPrivate);
                    break;
                default:
                    cerr << "unknown access type: " << accType << endl;
            }
        }
    }
    else {
        // what about finals?
        if (isFinal()) {
            switch (accType) {
                case PUBLIC_ACCESS:
                    runCheck<ErrorInstanceFinalPublicFieldName>(prefs->formatInstanceFinalPublic);
                    break;
                case PROTECTED_ACCESS:
                    runCheck<ErrorInstanceFinalProtectedFieldName>(prefs->formatInstanceFinalProtected);
                    break;
                case PACKAGE_ACCESS:
                    runCheck<ErrorInstanceFinalPackageFieldName>(prefs->formatInstanceFinalPackage);
                    break;
                case PRIVATE_ACCESS:
                    runCheck<ErrorInstanceFinalPrivateFieldName>(prefs->formatInstanceFinalPrivate);
                    break;
                default:
                    cerr << "unknown access type: " << accType << endl;
            }
        }
        else {
            switch (accType) {
                case PUBLIC_ACCESS:
                    runCheck<ErrorInstanceNonfinalPublicFieldName>(prefs->formatInstanceNonfinalPublic);
                    break;
                case PROTECTED_ACCESS:
                    runCheck<ErrorInstanceNonfinalProtectedFieldName>(prefs->formatInstanceNonfinalProtected);
                    break;
                case PACKAGE_ACCESS:
                    runCheck<ErrorInstanceNonfinalPackageFieldName>(prefs->formatInstanceNonfinalPackage);
                    break;
                case PRIVATE_ACCESS:
                    runCheck<ErrorInstanceNonfinalPrivateFieldName>(prefs->formatInstanceNonfinalPrivate);
                    break;
                default:
                    cerr << "unknown access type: " << accType << endl;
            }
        }
    }
}
