#ifndef SequenceChecker_h
#include "SequenceChecker.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef UserPreferences_h
#include "UserPreferences.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef SequenceOrder_h
#include "SequenceOrder.h"
#endif

#ifndef Java_h
#include "Java.h"
#endif

#ifndef SynDebug_h
#include "SynDebug.h"
#endif

#ifndef doctorj_algorithm
#define doctorj_algorithm
#include <algorithm>
#endif

#include <ctype.h>

using namespace std;
using namespace doctorj;

SequenceChecker* SequenceChecker::instance_ = NULL;

SequenceChecker* SequenceChecker::get()
{
    if (!instance_) {
        instance_ = new SequenceChecker();
    }
    return instance_;
}

SequenceChecker::SequenceChecker()
{
    vector<string>& orders = UserPreferences::get()->typeOrder;

    // Some legacy support here. It used to be that "types.order" was a very
    // long string. Now it's an array (OK, a vector), but we'll still support
    // the old format. For now.
    if (orders.size() == 1) {
        orders = StringUtilities::split(orders[0], " ");
    }

    SYNLOGF("# orders = %d", orders.size());
    
    for (int i = 0; i < orders.size(); ++i) {
        SYNLOGF("    order[%d] = '%s'", i, orders[i].c_str());

        if (orders[i].length() > 0) {
            
            vector<string> types = StringUtilities::split(orders[i], ".");

            string acc;             // private, protected, package, public
            string fnf;             // final or nonfinal
            string si;              // static or instance
            string type;            // class, abtractclass or interface, field, method, or abstractmethod
            string name;            // name
        
            for (int ti = 0; ti < types.size(); ++ti) {
                const string t = types[ti];
                SYNLOGF("        type[%d] = '%s'", ti, t.c_str());
                if (t == "private" || t == "protected" || t == "package" || t == "public") {
                    acc = t;
                }
                else if (t == "final" || t == "nonfinal") {
                    fnf = t;
                }
                else if (t == "static" || t == "instance") {
                    si = t;
                }
                else if (t == "abstractclass" ||
                         t == "class"         || 
                         t == "interface"     ||
                         t == "field"         || 
                         t == "method"        || 
                         t == "ctor"          || 
                         t == "abstractmethod") {
                    type = t;
                }
                else {
                    name = t;
                }
            }

            // The order in which the preferences are added determines the
            // preferred order.
            SequenceOrder* so = new SequenceOrder(acc, fnf, si, type, name);
            preferences_.push_back(so);
        }
    }
}

SequenceChecker::~SequenceChecker()
{
    vector<SequenceOrder*>::iterator it   = preferences_.begin();
    vector<SequenceOrder*>::iterator stop = preferences_.end();
    for (; it != stop; ++it) {
        SequenceOrder* so = *it;
        delete so;
    }
}

int SequenceChecker::order(AstItem* const item) const
{
    SYNLOG(item->text());
    
    AstModifierList*  mods      = JavaItem::getModifierList(item);
    string            access    = JavaModifierList::accessOf(mods);
    SYNLOG("access: " + access);
    string            finality  = JavaModifierList::finalityOf(mods);
    SYNLOG("finality: " + finality);
    string            ownership = JavaModifierList::ownershipOf(mods);
    SYNLOG("ownership: " + ownership);
    string            type      = JavaItem::typeOf(item);
    SYNLOG("type: " + type);
    vector<string>    names     = JavaItem::namesOf(item);

//       cout << "SequenceChecker::order(): names:\n\t";
//       copy(names.begin(), names.end(), ostream_iterator<string>(cout, " "));
//       cout << endl;

    string name = names.size() == 1 ? names[0] : "";

    int bestIndex = -1;
    int bestMatch = -1;
    for (int i = 0; i < preferences_.size(); ++i) {
        int m = preferences_[i]->matchValue(access, finality, ownership, type, name);
        SYNLOGF("checking: preferences_[%d]:", i);
        if (m > bestMatch) {
            SYNLOGF("resetting best index to %d for match %d", i, m);
            bestIndex = i;
            bestMatch = m;
        }
        else {
            SYNLOGF("score %d is not more than best match %d", m, bestMatch);
        }
    }
    
    SYNLOG("SequenceChecker::order():");
    SYNLOG("        item:");
    SYNLOGF("        text: %s", item->text().c_str());
    SYNLOGF("      access: %s", access.c_str());
    SYNLOGF("    finality: %s", finality.c_str());
    SYNLOGF("   ownership: %s", ownership.c_str());
    SYNLOGF("        type: %s", type.c_str());
    SYNLOGF("      #names: %d", names.size());
    SYNLOGF("    names[0]: %s", names[0].c_str());
    SYNLOGF("        type: %s", item->type().c_str());
    SYNLOGF("   bestIndex: %d", bestIndex);
    SYNLOGF("   bestMatch: %d", bestMatch);
    
    return bestIndex;
}

bool SequenceChecker::shouldFollow(AstItem* const current, AstItem* const previous)
{
    SYNLOG("SequenceChecker::shouldFollow(): comparing");
    SYNLOG("    previous:");
    SYNLOGF("        ptr:  %p", previous);
    SYNLOGF("        text: %s", (previous ? previous->text().c_str() : "<<<>>>"));
    SYNLOGF("        type: %s", (previous ? previous->type().c_str() : "<<<>>>"));
    SYNLOG("    current : ");
    SYNLOGF("        ptr:  %p", current);
    SYNLOGF("        text: %s", (current ? current->text().c_str() : "<<<>>>"));
    SYNLOGF("        type: %s", (current ? current->type().c_str() : "<<<>>>"));

    if (previous == NULL) {
        return true;
    }
    else {
        int corder = order(current);
        int porder = order(previous);
        
        SYNLOGF("SequenceChecker::shouldFollow(): current order  = %d", corder);
        SYNLOGF("SequenceChecker::shouldFollow(): previous order = %d", porder);
        
        return porder <= corder;
    }
}
