#ifndef DebugOptions_h
#define DebugOptions_h

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

using namespace std;


/**
 * Macro to be used in implementation (.cpp) files. Only one per file, thus the
 * curious name. If you need more than one debug option per file, use a static
 * DebugOptionRegistrant variable.
 */
#define REGISTER_DEBUG_OPTION(Char, Desc) \
    static doctorj::DebugOptionRegistrant only_one_debug_option_per_file(Char, Desc)

namespace doctorj
{
    /**
     * Registers with the debug options set.
     */
    class DebugOptionRegistrant
    {
    public:
        /**
         * Registers with the debug options set singleton.
         */
        DebugOptionRegistrant(char ch, const string& desc);

        /**
         * Does NOT de-register with the debug options set singleton.
         */
        virtual ~DebugOptionRegistrant();

    };


    /**
     * A debugging option.
     */
    class DebugOption
    {
    public:
        /**
         * Creates a debug option. The number is not assigned.
         *
         * @param ch The character associated with the option.
         * @param desc The description of the option.
         */
        DebugOption(char ch, const string& desc);

        /**
         * Destructor.
         */
        virtual ~DebugOption();
        
        /**
         * Returns the number associated with this option.
         */
        int getNumber() const;

        /**
         * Sets the number associated with this option.
         */
        void setNumber(int n);

        /**
         * Returns the character associated with this option.
         */
        char getChar() const;

        /**
         * Returns the description.
         */
        string getDescription() const;

    private:
        int num_;

        char ch_;

        string description_;

    };


    /**
     * A set of debug options.
     */
    class DebugOptions
    {
    public:
        /**
         * Singleton accesor.
         */
        static DebugOptions* get();

        /**
         * Destructor. Does not delete options.
         */
        virtual ~DebugOptions();

        /**
         * Adds the option. Memory ownership is NOT taken by this object.
         */
        void add(DebugOption* const opt);

        /**
         * Returns an array of all the one-character codes for options.
         */
        string allChars() const;

        /**
         * Writes all options to the given output stream.
         */
        void writeAll(ostream& os);

        /**
         * Returns whether the given option is enabled.
         */
        bool isEnabled(char ch) const;

        /**
         * Returns whether any option is enabled.
         */
        bool isEnabled() const;

        /**
         * Sets the debug options, from a string.
         */
        void set(const string& str);

    protected:
        DebugOptions();

        static DebugOptions* instance_;

        vector<DebugOption*> options_;

        int debug_;

        int num_;

    };

}

#endif //! DebugOptions_h
