#ifndef File_h
#define File_h

#ifndef StringEdit_h
#include "StringEdit.h"
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

using namespace std;

namespace doctorj
{
    class StringEditDelete;
    
    class FileException
    {
    };

    /**
     * An instantiation of the file class contains all the text for the file
     * that has been read, and can display text at a given point using filePos
     * objects. It also does end-of-line conversions to the Unix style ('\n').
     */
    class File : public StringEditor
    {
    public:
        /**
         * Reads the given file. Throws a <code>fileException</code> on error.
         */
        File(const string& fname) throw(FileException);

        /**
         * Frees the string representing the text of the file.
         */
        virtual ~File();

        string name() const;

        /**
         * Reads the file and converts end-of-lines to Unix format ('\n').
         */
        void readFile();
        
        /**
         * Fetches text from the file read, and returns the amount remaining.
         * This is for usage by the scanner/Lex.
         */
        int getInput(char* buf, size_t maxSize);

        /**
         * Returns the beginning of the text.
         */
        char* head() const;

        /**
         * Returns  the number  of the  line for the  given position.  Note that
         * lines are 1 indexed, so the first line of the file is line #1.
         */
        int lineOf(char* position) const;

        /**
         * Returns the column for the given position. Positions are 0 indexed.
         * Note that a tab gets counted as 1 column, no matter what the tab
         * width is set to.
         */
        int columnOf(char* position) const;

        /**
         * Returns the full line where the given position is found. This does
         * not include the end of line (carriage return), if any.
         */
        string fullLine(char* position) const;

        /**
         * Returns the full line for the given (1-indexed) line number. This does
         * not include the end of line (carriage return), if any.
         */
        string fullLine(int line) const;

        /**
         * Returns the number of characters in the file.
         */
        int length() const;

        /**
         * Returns the end of the file.
         */
        char* end() const;

        /**
         * Returns the position at the start of the given line.
         */
        char* startPosition(unsigned int line) const;

        /**
         * Adds the given edit. Executes the edit immediately.
         */
        void addEdit(StringEdit* const edit);

//         /**
//          * Removes any lines containing the start through end positions.
//          */
//         void removeLines(char* const start, char* const end);
        
        /**
         * Writes this file to the given stream, executing any pending edits.
         */
        void write(ostream& os);
        
        /**
         * Writes this file back out, executing any pending edits.
         */
        void write();

        /**
         * Returns whether the contents of the file changed.
         */
        bool changed() const;

        void showString(const string& str, char* const start, char* const end);

        void showCurrent(char* const start, char* const end);
        
        void showPreview(StringEdit* const edit, char* const start, char* const end);
        
        void showPreview(StringEditDelete* const del, char* const start, char* const end);

        void showLine(const string& str, int startLine);

    private:
        /**
         * Line feed ('\n').
         */
        static char LF;

        /**
         * Carriage return ('\r').
         */
        static char CR;
        
        /**
         * The text of the original file.
         */
        char* text_;
        
        /**
         * Where in the string each line starts. Index == line.
         */
        vector<char*> positions_;
        
        /**
         * The last place the object returned input. See the <code>getInput</code>
         * method.
         */
        char* inputPosition_;

        /**
         * The end of the string.
         */
        char* end_;

        /**
         * The name of the file.
         */
        string name_;

        /**
         * Whether the file contents were changed.
         */
        bool changed_;

    };


    class FileName
    {
    public:
        FileName(const string& str);

        virtual ~FileName();

        string extension() const;

        string basename() const;

        static string extension(const string& fname);

        static string basename(const string& fname);

    private:
        string str_;

    };


    class OutFile
    {
    public:
        OutFile(const string& name, const string& contents, const string& eoln);
        
        virtual ~OutFile();

        void write();

    private:
        string name_;

        string contents_;

        string eoln_;
    };

}

#endif //! File_h
