#ifndef StringEdit_h
#include "StringEdit.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef UtilDebug_h
#include "UtilDebug.h"
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

#include <stdio.h>

using namespace doctorj;
using namespace std;

int EditableString::INVALID_INDEX = -1;

EditableString::EditableString(const string& str, char* const head) : str_(str), head_(head)
{
    npositions_ = str_.length();
    for (int i = 0; i < npositions_; ++i) {
        positions_.push_back(i);
    }

    DEBUG_UTIL(dbout << "EDSTR  ::EditableString(str, head)" << endl);
    DEBUG_UTIL(dbout << "EDSTR  head = " << (void*)head_ << endl);
}

EditableString::EditableString(const EditableString& rhs)
{
    str_ = rhs.str_;
    npositions_ = rhs.npositions_;
    for (int i = 0; i < npositions_; ++i) {
        positions_.push_back(rhs.positions_[i]);
    }
    head_ = rhs.head_;
    DEBUG_UTIL(dbout << "EDSTR  ::EditableString(rhs)" << endl);
    DEBUG_UTIL(dbout << "EDSTR  head = " << (void*)head_ << endl);
}

EditableString* EditableString::clone()
{
    return new EditableString(*this);
}

EditableString::~EditableString()
{
}

void EditableString::insert(int pos, const string& s)
{
    DEBUG_UTIL(dbout << "EDSTR  ::insert(" << pos << ", " << s << ")" << endl);
    DEBUG_UTIL(dump());
    DEBUG_UTIL(dbout << "EDSTR  positions_[" << pos << "] = " << positions_[pos] << endl);
    str_.replace(positions_[pos], 0, s);
    DEBUG_UTIL(dbout << "EDSTR  shifting positions" << endl);
    shiftEachPosition(pos, npositions_ - 1, s.length());
}

void EditableString::remove(int pos, int len)
{
    DEBUG_UTIL(dbout << "EDSTR  ::remove(" << pos << ", " << len << ")" << endl);

    str_.replace(positions_[pos], len, "");

    for (int i = pos; i < pos + len; ++i) {
        positions_[i] = INVALID_INDEX;
    }

    shiftEachPosition(pos + len, npositions_ - 1, -len);
}

void EditableString::replace(int apos, int alen, const string& bstr)
{
    DEBUG_UTIL(dbout << "EDSTR  ::replace(" << apos << ", " << alen << ", " << bstr << ")" << endl);
    DEBUG_UTIL(dump());

    int blen = bstr.length();

    DEBUG_UTIL(dbout << "EDSTR  positions_[" << apos << "] = " << positions_[apos] << endl);
    DEBUG_UTIL(dbout << "EDSTR  alen = " << alen << endl);
    DEBUG_UTIL(dbout << "EDSTR  bstr = " << bstr << endl);
    str_.replace(positions_[apos], alen, bstr);

    for (int i = 0; i < alen; ++i) {
        // puts "@positions[#{apos + i}] #{@positions[apos + i]} += i #{i} (blen #{blen})"
        positions_[apos + i] = INVALID_INDEX;

//         if (positions_[apos + i] != INVALID_INDEX) {
//             if (i < blen) {
//                 positions_[apos + i] += i;
//             }
//             else {
//                 positions_[apos + i] = INVALID_INDEX;
//             }
//         }
    }

    shiftEachPosition(apos + alen, npositions_ - 1, blen - alen); 
    DEBUG_UTIL(dbout << "EDSTR  ----- editable string after replace" << endl);
    DEBUG_UTIL(dump());
    DEBUG_UTIL(dbout << "EDSTR  ::replace done." << endl);
}

void EditableString::move(int fromindex, int fromlen, int toindex, int toposition)
{
    DEBUG_UTIL(dbout << "EDSTR  ::move(" << fromindex << ", " << fromlen << ", " << toindex << ", " << toposition << ")" << endl);

    if (toposition == -999) {
        toposition = positions_[toindex];
    }

    DEBUG_UTIL(dbout << "EDSTR  toposition = " << toposition << endl);
    DEBUG_UTIL(dump());

    string mvstr = str_.substr(positions_[fromindex], fromlen);

    DEBUG_UTIL(dbout << "EDSTR  move string: '" << mvstr << "'" << endl);

    if (fromindex > toindex) {
        DEBUG_UTIL(dbout << "EDSTR  to follows from" << endl);
        str_.replace(positions_[fromindex], fromlen, "");
        str_.replace(toposition,            0,       mvstr);

        int orig = toposition;
        
        shiftEachPosition(positions_[toindex], positions_[fromindex - 1], fromlen);
        
        for (int i = 0; i < fromlen; ++i) {
            positions_[fromindex + i] = orig + i;
        }
    }
    else {
        DEBUG_UTIL(dbout << "EDSTR  from follows to" << endl);
        str_.replace(toposition,            0,       mvstr);
        str_.replace(positions_[fromindex], fromlen, "");

        for (int i = 0; i < fromlen; ++i) {
            DEBUG_UTIL(dbout << "EDSTR  positions_[" << fromindex + i << "] = " << toposition - fromlen + i << endl);
            positions_[fromindex + i] = toposition - fromlen + i;
        }
        
        shiftEachPosition(fromindex + fromlen, toposition - 1, -fromlen);
    }
}

void EditableString::insert(char* const pos, const string& s)
{
    DEBUG_UTIL(dbout << "EDSTR  ::insert(" << (void*)pos << ", " << s << ")" << endl);
    DEBUG_UTIL(dbout << "EDSTR  head = " << (void*)head_ << endl);
    DEBUG_UTIL(dbout << "EDSTR  pos - head = " << pos - head_ << endl);
    insert(pos - head_, s);
}

void EditableString::remove(char* const pos, int len)
{
    remove(pos - head_, len);
}

void EditableString::replace(char* const apos, int alen, const string& bstr)
{
    replace(apos - head_, alen, bstr);
}

void EditableString::moveInsert(char* const apos, int alen, char* const bpos)
{
    DEBUG_UTIL(dbout << "EditableString::moveInsert(" << (void*)apos << ", " << alen << ", " << (void*)bpos << ")" << endl);
    move(apos - head_, alen, bpos - head_, positions_[bpos - head_]);
}

void EditableString::moveAfter(char* const apos, int alen, char* const bpos)
{
    DEBUG_UTIL(dbout << "EditableString::moveAfter(" << (void*)apos << ", " << alen << ", " << (void*)bpos << ")" << endl);
    DEBUG_UTIL(dbout << "EDSTR  apos - head_ = " << apos - head_ << endl);
    DEBUG_UTIL(dbout << "EDSTR  alen = " << alen << endl);
    DEBUG_UTIL(dbout << "EDSTR  bpos - head_ = " << bpos - head_ << endl);
    int bposition = bpos > head_ ? positions_[bpos - head_ - 1] + 1 : positions_[0];
    move(apos - head_, alen, bpos - head_, bposition);
}

void EditableString::shiftEachPosition(int from, int to, int amount)
{
    DEBUG_UTIL(dbout << "EDSTR  shiftEachPosition(" << from << ", " << to << ", " << amount << ")" << endl);
    for (int i = from; i <= to; ++i) {
        shiftPosition(i, amount);
    }
}

void EditableString::shiftPosition(int index, int amount)
{
    if (positions_[index] != INVALID_INDEX) {
        positions_[index] += amount;
    }
}

void EditableString::dump() const
{
    DEBUG_UTIL(dbout << "    indices: ");
    for (int i = 0; i < str_.length(); i += 10) {
        DEBUG_UTIL(dbout << ((i / 10) % 10) << "         ");
    }
    DEBUG_UTIL(dbout << endl);
    DEBUG_UTIL(dbout << "             ");
    for (int i = 0; i < str_.length(); ++i) {
        DEBUG_UTIL(dbout << i % 10);
    }
    DEBUG_UTIL(dbout << endl);
    DEBUG_UTIL(dbout << "    string: '");
    for (int i = 0; i < str_.length(); ++i) {
        if (str_[i] == '\n') {
            DEBUG_UTIL(dbout << '|');
        }
        else {
            DEBUG_UTIL(dbout << str_[i]);
        }
    }
    DEBUG_UTIL(dbout << "'" << endl);
    DEBUG_UTIL(dbout << "    positions:" << endl);
    DEBUG_UTIL(dbout << "    npositions_ = " << npositions_ << endl);
    const int COLS = 15;
    for (int a = 0; a <= npositions_ / COLS; ++a) {
        DEBUG_UTIL(printf("        "));
        for (int i = 0; i < COLS; ++i) {
            int index = (a * COLS) + i;
            if (index < npositions_) {
                DEBUG_UTIL(printf("[%3d] %4d ", index, positions_[index]));
            }
        }
        DEBUG_UTIL(printf("\n"));
    }
}

string EditableString::str() const
{
    return str_;
}

char* EditableString::head() const
{
    return head_;
}

int EditableString::index(char* const pos) const
{
    int index = pos - head_;
    
    if (index >= npositions_) {
        cerr << "ERROR in EditableString::index(" << (void*)pos << "): index " << index << " beyond " << npositions_ << endl;
    }

    return positions_[pos - head_];
}


StringEdit::StringEdit()
{
}

StringEdit::~StringEdit()
{
}

void StringEdit::showUnedited(File* const file)
{
    char* start = position();
    char* end = endPosition();
    file->showCurrent(start, end);
}

void StringEdit::showEdited(File* const file)
{
    char* start = position();
    char* end = endPosition();
    file->showPreview(this, start, end);
}


static char* getFrom(char* const from, char* const lowerBound)
{
    return from > lowerBound ? from : lowerBound;
}

static char* getTo(char* const to, char* const upperBound)
{
    return to < upperBound ? to : upperBound;
}

StringEditDelete::StringEditDelete(char* const from, 
                                   char* const to, 
                                   char* const lowerBound, 
                                   char* const upperBound)
        : StringEditSingle(getFrom(from, lowerBound))
{
    len_ = getTo(to, upperBound) - position();
}

StringEditDelete::StringEditDelete(char* const from, 
                                   char* const to)
        : StringEditSingle(from)
{
    len_ = getTo(to, from + strlen(from)) - from;
}

StringEditDelete::StringEditDelete(char* const from, 
                                   int len)
        : StringEditSingle(from)
{
    len_ = getTo(from + len, from + strlen(from)) - from;
}

StringEditDelete::~StringEditDelete()
{
}

void StringEditDelete::execute(EditableString* const estr) const
{
    DEBUG_ANY(cout << "StringEditInsert::execute" << endl);
    estr->remove(position(), len_);
}

int StringEditDelete::length() const
{
    return len_;
}

void StringEditDelete::showEdited(File* const file)
{
    char* start = position();
    char* end = endPosition();
    file->showPreview(this, start, start);
}


static char* getLineFrom(char* const from, char* const lowerBound)
{
    char* f = from > lowerBound ? from : lowerBound;
    
    while (*f != '\n' && f > lowerBound) {
        // cout << "seeking start" << endl;
        --f;
    }
//     if (*f == '\n') {
//         ++f;
//     }
    
    return f;
}

static char* getLineTo(char* const to, char* const upperBound)
{
    char* t = to < upperBound ? to : upperBound;
    
    while (*t != '\n' && t/*  + 1 */ < upperBound) {
        //         cout << "seeking end" << endl;
        ++t;
    }
//     ++t;

    return t;
}

StringEditDeleteLines::StringEditDeleteLines(char* const from, 
                                             char* const to, 
                                             char* const lowerBound, 
                                             char* const upperBound)
        : StringEditSingle(getLineFrom(from, lowerBound))
{

//     cout << "DLN  from = " << (void*)from << endl;
//     cout << "DLN  to = " << (void*)to << endl;
//     cout << "DLN  lb = " << (void*)lowerBound << endl;
//     cout << "DLN  ub = " << (void*)upperBound << endl;
//     cout << "DLN  f = " << (void*)f << endl;
//     cout << "DLN  f = " << f << endl;
//     cout << "DLN  t = " << (void*)t << endl;
//     cout << "DLN  t = " << t << endl;

    len_ = getLineTo(to, upperBound) - position();

//     cout << "SEDLN  from = " << (void*)from << endl;
//     cout << "SEDLN  len_ = " << len_ << endl;
}

StringEditDeleteLines::~StringEditDeleteLines()
{
}

void StringEditDeleteLines::showEdited(File* const file)
{
    cout << "    <<<deleted>>>" << endl;
}

void StringEditDeleteLines::execute(EditableString* const estr) const
{
    DEBUG_UTIL(cout << "STRDL  ::execute" << endl);
    estr->remove(position(), len_);
}

int StringEditDeleteLines::length() const
{
    return len_;
}


StringEditInsert::StringEditInsert(const string& str, char* pos) : 
        StringEditSingle(pos), str_(str)
{
//     cout << "INS  pos_ = " << (void*)pos_ << endl;
//     cout << "INS  pos_ = " << pos_ << endl;
//     cout << "INS  len_ = " << 0 << endl;
}

StringEditInsert::~StringEditInsert()
{
}

string StringEditInsert::str() const
{
    return str_;
}

int StringEditInsert::length() const
{
    return 0;
}

void StringEditInsert::showUnedited(File* const file)
{
    char* cstart = position();
    int cidxstart = file->index(cstart);
    int cidxend = cidxstart + length() - 1;

//     cout << "cstart  = " << (void*)cstart    << endl;
//     cout << "cend    = " << (void*)cend      << endl;
//     cout << "currstr = " << current          << endl;
//     cout << "cidxst  = " << cidxstart        << endl;
//     cout << "cidxend = " << cidxend          << endl;

//     string fileName = file->name();
//     cout << "-----" << endl;
//     cout << fileName << endl;
//     cout << endl;

    if (cidxstart >= cidxend) {
        //$$$ this should be better. Maybe a way to check if there is already
        //text on that line?

        showUneditedLine(file, cstart);
    }
    else {
        StringEdit::showUnedited(file);
    }
}

void StringEditInsert::showUneditedLine(File* const file, char* const cstart)
{
    int lnum = file->lineOf(cstart);
    //             cout << "----- line number " << lnum << endl;
    //             cout << "----- showing full line " << endl;
    string fline = file->fullLine(lnum);
    //             cout << "----- full line " << fline << endl;
    file->showLine(fline, lnum);
}

void StringEditInsert::execute(EditableString* const estr) const
{
    DEBUG_UTIL(cout << "SEINS  execute" << endl);
    DEBUG_UTIL(cout << "SEINS  execute: position() = " << (void*)position() << endl);
    estr->insert(position(), str_);
    DEBUG_UTIL(cout << "SEINS  execute done." << endl);
}

void StringEditInsert::showEdited(File* const file)
{
    char* start = position();
    char* end = start + str_.length();

    DEBUG_UTIL(cout << "SEINS  position = " << (void*)start << endl);
    DEBUG_UTIL(cout << "SEINS  end position = " << (void*)end << endl);    

    file->showPreview(this, start, end);
}


StringEditInsertLine::StringEditInsertLine(const string& str, char* pos) : 
        StringEditInsert(str, pos)
{
//     cout << "INS  pos_ = " << (void*)pos_ << endl;
//     cout << "INS  pos_ = " << pos_ << endl;
//     cout << "INS  len_ = " << 0 << endl;
}

StringEditInsertLine::~StringEditInsertLine()
{
}

void StringEditInsertLine::showUneditedLine(File* const file, char* const cstart)
{
    cout << "    <<<none>>>" << endl;
}


StringEditMoveAfter::StringEditMoveAfter(char* from, int len, char* to) : 
        StringEditMove(from, len, to)
{
    // DEBUG_UTIL(cout << "SEMOV  StringEditMoveAfter(" << from << ", " << len << ", " << to << ")" << endl);
    DEBUG_ANY(cout << "SEMOV  StringEditMoveAfter(" << from << ", " << len << ", " << to << ")" << endl);
}

StringEditMoveAfter::~StringEditMoveAfter()
{
}

void StringEditMoveAfter::execute(EditableString* const estr) const
{
    estr->moveAfter(from_, length_, to_);
}



StringEditMoveBefore::StringEditMoveBefore(char* from, int len, char* to) : 
        StringEditMove(from, len, to)
{
    DEBUG_UTIL(cout << "SEMOV  StringEditMoveBefore(" << from << ", " << len << ", " << to << ")" << endl);
}

StringEditMoveBefore::~StringEditMoveBefore()
{
}


StringEditMove::StringEditMove(char* from, int len, char* to) : 
        from_(from), length_(len), to_(to)
{
    DEBUG_UTIL(cout << "SEMOV  StringEditMove(" << from << ", " << len << ", " << to << ")" << endl);
    DEBUG_UTIL(cout << "SEMOV  from_   = " << (void*)from_ << endl);
    DEBUG_UTIL(cout << "SEMOV  from_   = " << from_ << endl);
    DEBUG_UTIL(cout << "SEMOV  length_ = " << length_ << endl);
    DEBUG_UTIL(cout << "SEMOV  to_     = " << (void*)to_ << endl);
    DEBUG_UTIL(cout << "SEMOV  to_     = " << to_ << endl);
}

StringEditMove::~StringEditMove()
{
}

// void StringEditMove::apply(StringEditor* const editor, string* const s) const
// {
//     DEBUG_UTIL(cout << "SEMOV  StringEditMove::apply(" << *s << ")" << endl);

//     DEBUG_UTIL(cout << "SEMOV  from_    = " << (void*)from_ << endl);
//     DEBUG_UTIL(cout << "SEMOV  from_    = " << from_ << endl);
//     DEBUG_UTIL(cout << "SEMOV  length_  = " << length_ << endl);
//     DEBUG_UTIL(cout << "SEMOV  to_      = " << (void*)to_ << endl);
//     DEBUG_UTIL(cout << "SEMOV  to_      = " << to_ << endl);

//     // yank the string ...
//     int fstart = editor->index(from_);
//     int fend   = editor->index(from_ + length_);
//     // -1 is OK. That's the end of the string.

//     int tidx = toIndex(editor);
//     DEBUG_UTIL(cout << "SEMOV  to idx   = " << tidx << endl);

//     DEBUG_UTIL(cout << "SEMOV  fstart   = " << fstart << endl);
//     DEBUG_UTIL(cout << "SEMOV  fend     = " << fend << endl);

//     string fstr = s->substr(fstart, fend - fstart);
//     DEBUG_UTIL(cout << "SEMOV  fstr     = '" << fstr << "'" << endl);

//     if (from_ > to_) {
//         DEBUG_UTIL(cout << "SEMOV  from follows to" << endl);
        
//         // erase ...
//         s->erase(fstart, fend - fstart);
//         DEBUG_UTIL(cout << "SEMOV  s        = '" << *s << "'" << endl);

//         // then insert.
//         s->replace(tidx, 0, fstr);
//         DEBUG_UTIL(cout << "SEMOV  s        = '" << *s << "'" << endl);
//     }
//     else {
//         DEBUG_UTIL(cout << "SEMOV  to follows from" << endl);

//         // insert ...
//         s->replace(tidx, 0, fstr);
//         DEBUG_UTIL(cout << "SEMOV  s        = '" << *s << "'" << endl);

//         // then erase.
//         s->erase(fstart, fend - fstart);
//         DEBUG_UTIL(cout << "SEMOV  s        = '" << *s << "'" << endl);
//     }
// }

// void StringEditMove::execute(StringEditor* const editor, string* const s) const
// {
//     DEBUG_UTIL(cout << "SEMOV  StringEditMove::execute(" << *s << ")" << endl);
//     apply(editor, s);
//     editor->update(from_, length_, to_);
// }

void StringEditMove::execute(EditableString* const estr) const
{
    estr->moveInsert(from_, length_, to_);
}

char* StringEditMove::position() const
{
    return from_;
}

char* StringEditMove::endPosition() const
{
    return from_ + length_ - 1;
}


StringEditMulti::StringEditMulti(const vector<StringEdit*>& edits) : edits_(edits)
{
}

StringEditMulti::StringEditMulti()
{
}

StringEditMulti::~StringEditMulti()
{
}

void StringEditMulti::add(StringEdit* const edit)
{
    edits_.push_back(edit);
}

void StringEditMulti::apply(EditableString* const estr) const
{
    DEBUG_ANY(cout << "SEMUL  applying edits" << endl);

    // we'll actually be munging this string/editor during all of the changes in
    // this group of edits.
    EditableString copy(*estr);
    execute(&copy);
}

void StringEditMulti::execute(EditableString* const estr) const
{
    DEBUG_ANY(cout << "SEMUL  executing edits" << endl);
    vector<StringEdit*>::const_iterator stop = edits_.end();
    vector<StringEdit*>::const_iterator it   = edits_.begin();
    while (it != stop) {
        StringEdit* edit = *it;
        edit->execute(estr);
        ++it;
    }
}
    
char* StringEditMulti::position() const
{
    char* first = NULL;
    vector<StringEdit*>::const_iterator stop = edits_.end();
    vector<StringEdit*>::const_iterator it   = edits_.begin();
    while (it != stop) {
        StringEdit* edit = *it;
        char* p = edit->position();
        if (first == NULL || p < first) {
            first = p;
        }
        ++it;
    }
    return first;
}

char* StringEditMulti::endPosition() const
{
    char* last = NULL;
    vector<StringEdit*>::const_iterator stop = edits_.end();
    vector<StringEdit*>::const_iterator it   = edits_.begin();
    while (it != stop) {
        StringEdit* edit = *it;
        char* p = edit->position();
        if (last == NULL || p > last) {
            last = p;
        }
        ++it;
    }
    return last;
}


StringEditReplace::StringEditReplace(const string& str, char* pos, int len) : 
        StringEditSingle(pos), str_(str), len_(len)
{
}

StringEditReplace::~StringEditReplace()
{
}

string StringEditReplace::str() const
{
    return str_;
}

int StringEditReplace::length() const
{
    return len_;
}

void StringEditReplace::execute(EditableString* const estr) const
{
    DEBUG_ANY(cout << "SEREP  ::execute" << endl);
    DEBUG_ANY(cout << "SEREP  len = " << len_ << endl);
    DEBUG_ANY(cout << "SEREP  str = " << str_ << endl);
    estr->replace(position(), len_, str_);
}



StringEditSingle::StringEditSingle(char* const pos) : pos_(pos)
{
}

StringEditSingle::~StringEditSingle()
{
}

void StringEditSingle::execute(EditableString* const estr) const
{
//     apply(editor, s);
    
//     int newln = str().length();
//     int oldln = length();
//     int offset = oldln - newln;
//     char* pos = position();
//     DEBUG_UTIL(cout << "SESIN  pos = " << pos << endl);
//     DEBUG_UTIL(cout << "SESIN  offset = " << offset << endl);
//     DEBUG_UTIL(cout << "SESIN  edit.length = " << oldln << endl);
//     DEBUG_UTIL(cout << "SESIN  str.length = " << newln << endl);

//     editor->update(offset, pos);
}

char* StringEditSingle::position() const
{
    return pos_;
}

char* StringEditSingle::endPosition() const
{
    return pos_ + length() - 1;
}


StringEditor::StringEditor(char* str)
{
    init(str);
}

StringEditor::StringEditor(const StringEditor& rhs) : 
        estr_(rhs.estr_->clone()), orig_(rhs.orig_)
{
}

StringEditor::StringEditor()
{
}

StringEditor::~StringEditor()
{
    EACH(vector<StringEdit*>, edits_, eit) {
        StringEdit* edit = *eit;
        delete edit;
    }
    delete estr_;
}

void StringEditor::init(char* str)
{
    estr_ = new EditableString(str, str);
    orig_ = str;
}

string StringEditor::str() const
{
    return estr_->str();
}

void StringEditor::add(StringEdit* const edit)
{
    edits_.push_back(edit);
}

void StringEditor::execute()
{
    EACH(vector<StringEdit*>, edits_, eit) {
        StringEdit* edit = *eit;
        DEBUG_ANY(cout << "STRED  ::execute: executing edit" << endl);
        execute(*edit);
        delete edit;
    }
    edits_.clear();
}

void StringEditor::execute(const StringEdit& edit)
{
    edit.execute(estr_);
}

bool StringEditor::hasEdits() const
{
    return edits_.size() > 0;
}

int StringEditor::index(char* const pos) const
{
//     int idx = pos - estr_->head();
//     cout << "STRED  index(" << (void*)pos << ") = " << idx << endl;
    int eidx = estr_->index(pos);
//     cout << "STRED  estr index(" << (void*)pos << ") = " << eidx << endl;
    return eidx;
}

string StringEditor::preview(const StringEdit& edit)
{
    // use a copy, not the real string
    EditableString estrcp(*estr_);
    edit.execute(&estrcp);
    return estrcp.str();
}

void StringEditor::apply(const StringEdit& edit)
{
    DEBUG_UTIL(cout << "STRED  ::apply was invoked!" << endl);
    edit.execute(estr_);
}

string StringEditor::originalStr() const
{
    return string(orig_);
}
