#ifndef Test_h
#include "Test.h"
#endif

#ifndef Log_h
#include "Log.h"
#endif

#ifndef drj_stdio_h
#define drj_stdio_h
#include <stdio.h>
#endif

//----------------------------------------------------------------------
// test message
//----------------------------------------------------------------------

ostream& operator<<(ostream& os, const TestMessage& tm)
{
    return tm.print(os);
}

TestMessage::TestMessage(const string& file, 
                         int line, 
                         const string& funcName,
                         const string& msg, 
                         const string& status)
        : file_(file), line_(line), funcName_(funcName), msg_(msg), status_(status) 
{
}

TestMessage::~TestMessage()
{
}

ostream& TestMessage::print(ostream& os) const
{
#ifdef DEBUG
    char str[256];
    doctorj::Log::log(0, file_, line_, funcName_, msg_);
#else
    os << file_ << ":" << line_ << ": " << status_ << ": " << msg_ << endl;
#endif
    return os;
}

//----------------------------------------------------------------------
// test
//----------------------------------------------------------------------

ostream& operator<<(ostream& os, const Test& t)
{
    return os << (t.nerrors() == 0 ? "passed" : "failed");
}

Test::Test() : nerrors_(0), showSuccesses_(false)
{
}

Test::Test(bool showSuccesses) : nerrors_(0), showSuccesses_(showSuccesses)
{
}

Test::~Test()
{
}

void Test::test(bool condition, const string& msg, char file[], int line, const char* funcName)
{
    if (!condition) {
        fail(msg, file, line, funcName);
    }
    else if (showSuccesses_) {
        report(cout, file, line, funcName, "passed", msg);
    }
}

void Test::test(const string& condstr, bool condition, const string& msg, char file[], int line, const char* funcName)
{
#ifdef DEBUG
    char testmsg[256];
    sprintf(testmsg, "testing '%s'", condstr.c_str());
    doctorj::Log::log(0, file, line, funcName, testmsg);
#endif

    if (!condition) {
        fail(condstr + " (" + msg + ")", file, line, funcName);
    }
    else if (showSuccesses_) {
        report(cout, file, line, funcName, "passed", condstr + " (" + msg + ")");
    }
}

void Test::fail(const string& msg, char file[], int line, const char* funcName)
{
    report(cerr, file, line, funcName, "failed", msg);
    addError();
}

bool Test::passed() const
{
    return nerrors() == 0;
}

void Test::report(ostream& os, 
                  const string& file, 
                  int line, 
                  const string& funcName,
                  const string& status,
                  const string& msg)
{
#ifdef DEBUG
    char testmsg[256];
    doctorj::Log::log(0, file, line, funcName, msg);
#else
    os << file << ":" << line << ": " << status << ": " << msg << endl;
#endif
    TestMessage tm(file, line, funcName, msg, status);
    messages_.push_back(tm);
}

int Test::nerrors() const
{
    return nerrors_;
}

void Test::addError()
{
    nerrors_++;
}

void Test::writeReport(ostream& os) const
{
    for (vector<TestMessage>::const_iterator it = messages_.begin(); 
         it != messages_.end(); 
         ++it) {
        TestMessage tm = *it;
        tm.print(os);
    }
}

