#ifndef Test_h
#define Test_h

#ifndef testString
#define testString
#include <string>
#endif

#ifdef HAVE_SSTREAM
    #ifndef std_sstream
    #define std_sstream
    #include <sstream>
    #endif
#else
    #ifndef std_strstream
    #define std_strstream
    #include <strstream>
    #endif
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

using namespace std;

/**
 * Macro that expands the test condition to include the file and line where
 * used. Also stringifies the condition, so that "i == 4" is retained and
 * displayed. Note that the TEST macro is useful only within a class derived
 * from Test.
 */
#define TEST(Condition, Message) \
test(#Condition, Condition, Message, __FILE__, __LINE__, __PRETTY_FUNCTION__)

/**
 * Macro to do equality comparisons.
 */
#define TESTEQ(Var1, Var2, Message) \
testEq(#Var1, Var1, #Var2, Var2, Message, __FILE__, __LINE__, __PRETTY_FUNCTION__)

/**
 * Macro to do inequality comparisons.
 */
#define TESTNE(Var1, Var2, Message) \
testNe(#Var1, Var1, #Var2, Var2, Message, __FILE__, __LINE__, __PRETTY_FUNCTION__)

/**
 * Macro that reports a failure, e.g., a line that should not be reached. Note
 * that the FAIL macro is useful only within a class derived from Test.
 */
#define FAIL(Message) fail(Message, __FILE__, __LINE__, __PRETTY_FUNCTION__)

/**
 * A message associating a message and a status with a test at a file and line.
 */
class TestMessage 
{
public:

    friend ostream& operator<<(ostream& os, const TestMessage& tm);

    TestMessage(const string& file, 
                int line, 
                const string& funcName,
                const string& msg, 
                const string& status);

    virtual ~TestMessage();

    /**
     * Writes the message to the output stream, in the format "file:line:
     * status: message".
     */
    virtual ostream& print(ostream& os) const;

private:

    string file_;

    int line_;

    string funcName_;

    string msg_;

    string status_;

};

/**
 * Runs a series of tests, and reports failures and successes.
 */
class Test 
{
public:
    /**
     * Output operator.
     */
    friend ostream& operator<<(ostream& os, const Test& t);

    /**
     * Default constructor. Sets the passed status to true.
     */
    Test();

    /**
     * Takes the parameter of displaying successful tests.
     */
    Test(bool showSuccesses);

    /**
     * Destructor.
     */
    virtual ~Test();

    /**
     * Runs the entire test.
     */
    virtual void run() = 0;

    /**
     * Runs a single test and reports the condition.
     */
    virtual void test(bool condition, const string& msg, char file[], int line, const char* funcName);

    /**
     * Runs a single test and reports the condition.
     */
    virtual void test(const string& condstr, bool condition, const string& msg, char file[], int line, const char* funcName);
    
    /**
     * Tests for equality between the variables, and reports their status. There
     * must be an operator==(const T1&, const T2&) defined.
     */
    template <class T1, class T2>
    void testEq(const string& var1name, 
                const T1& var1,
                const string& var2name, 
                const T2& var2,
                const string& msg, 
                char file[], 
                int line, 
                const char* funcName);
    
    /**
     * Tests for inequality between the variables, and reports their status. There
     * must be an operator!=(const T1&, const T2&) defined.
     */
    template <class T1, class T2>
    void testNe(const string& var1name, 
                const T1& var1,
                const string& var2name, 
                const T2& var2,
                const string& msg, 
                char file[], 
                int line, 
                const char* funcName);
    
    /**
     * Adds the condition as having failed, and writes the message to the
     * standard error stream. This is primarily for locations in code that
     * should not have been reached, for example:<p>
     *
     * <pre>
     *     try {
     *         somethingIllegal();
     *         fail("should have thrown IllegalOperationException", _FILE_, _LINE_);
     *     }
     *     catch (const IllegalOperationException& e) {
     *         // expected.
     *     }
     */
    virtual void fail(const string& msg, char file[], int line, const char* funcName);

    /**
     * Returns whether this test passed. For more precision, use
     * <code>nerrors</code>, for the number of errors.
     */
    virtual bool passed() const;

    /**
     * Returns the number of errors.
     */
    virtual int nerrors() const;

    /**
     * Add one to the number of errors.
     */
    virtual void addError();

    /**
     * Writes the full report of tests conducted to the given output stream. If
     * <code>showSuccesses</code> is true, all tests will be printed.
     * Otherwise, only failures will be printed.
     */
    virtual void writeReport(ostream& os) const;

protected:

    /**
     * Writes a message to the given stream, in the standard compiler/grep/emacs
     * format (file:line: status: message). Also stores the message for later
     * display.
     */
    virtual void report(ostream& os, 
                        const string& file, 
                        int line, 
                        const string& funcName,
                        const string& status,
                        const string& msg);
    
    /**
     * Returns v as a string.
     */
    template <class T>
    static string toString(T v);
    
    /**
     * Passes back v converted from a string. Class T must have the
     * following defined:
     *
     *     T()                         // (no-args constructor)
     *     operator>>(strstream&, T&)
     *     operator=(int)
     */
    template <class T>
    static T fromString(const string& s);
    
private:

    /**
     * The number of errors.
     */
    int nerrors_;

    /**
     * Whether to show the successful tests.
     */
    bool showSuccesses_;

    /**
     * The tests conducted.
     */
    vector<TestMessage> messages_;

};

template <class T>
string Test::toString(T v)
{
#ifdef HAVE_SSTREAM
    ostringstream oss;
    oss << v;
    string sstr = oss.str();
#else
    ostrstream oss;
    oss << v << ends;
    string sstr = oss.str();

    // free the char* returned
    oss.freeze(false);
#endif

    return sstr;
}

template <class T>
T Test::fromString(const string& s)
{
    T v;
    if (s.length() > 0) {
#ifdef HAVE_SSTREAM
        istringstream iss(s.c_str());
#else
        istrstream iss(s.c_str());
#endif
        iss >> v;
    }
    else {
        // pretty much everything has a zero (note that string is handled
        // separately)
        v = 0;
    }
    return v;
}

template <class T1, class T2>
void Test::testEq(const string& var1name, 
                  const T1& var1,
                  const string& var2name, 
                  const T2& var2,
                  const string& msg, 
                  char file[], 
                  int line,
                  const char* funcName)
{
    string condstr = var1name;
    condstr += string(" (");
    condstr += toString(var1);
    condstr += string(") == ");
    condstr += var2name;
    condstr += string(" (");
    condstr += toString(var2);
    condstr += string(")");
    
    test(condstr, var1 == var2, msg, file, line, funcName);
}

template <class T1, class T2>
void Test::testNe(const string& var1name, 
                  const T1& var1,
                  const string& var2name, 
                  const T2& var2,
                  const string& msg, 
                  char file[], 
                  int line,
                  const char* funcName)
{
    string condstr = var1name;
    condstr += string(" (");
    condstr += toString(var1);
    condstr += string(") != ");
    condstr += var2name;
    condstr += string(" (");
    condstr += toString(var2);
    condstr += string(")");
    
    test(condstr, var1 != var2, msg, file, line, funcName);
}

#endif // !Test_h
