# Authors:
# Matthew Harmsen <mharmsen@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Copyright (C) 2012 Red Hat, Inc.
# All rights reserved.
#

# System Imports
from __future__ import absolute_import
import os

# PKI Deployment Imports
from .. import pkiconfig as config
from .. import pkimessages as log
from .. import pkiscriptlet


# PKI Deployment Instance Layout Scriptlet
class PkiScriptlet(pkiscriptlet.AbstractBasePkiScriptlet):

    def spawn(self, deployer):

        if config.str2bool(deployer.mdict['pki_skip_installation']):
            config.pki_log.info(log.SKIP_INSTANCE_SPAWN_1, __name__,
                                extra=config.PKI_INDENTATION_LEVEL_1)
            return

        config.pki_log.info(log.INSTANCE_SPAWN_1, __name__,
                            extra=config.PKI_INDENTATION_LEVEL_1)

        # if this is not the first subsystem, skip
        if len(deployer.instance.tomcat_instance_subsystems()) != 1:
            return

        # establish instance logs
        deployer.directory.create(deployer.mdict['pki_instance_log_path'])

        # copy /usr/share/pki/server/conf tree into
        # /var/lib/pki/<instance>/conf
        # except common ldif files and theme deployment descriptor
        deployer.directory.copy(
            deployer.mdict['pki_source_server_path'],
            deployer.mdict['pki_instance_configuration_path'],
            ignore_cb=file_ignore_callback_src_server)

        # Copy /usr/share/pki/server/conf/server.xml
        # to /etc/pki/<instance>/server.xml.
        deployer.file.copy_with_slot_substitution(
            deployer.mdict['pki_source_server_xml'],
            deployer.mdict['pki_target_server_xml'],
            overwrite_flag=True)

        # Link /etc/pki/<instance>/catalina.properties
        # to /usr/share/pki/server/conf/catalina.properties.
        deployer.symlink.create(
            os.path.join(deployer.mdict['pki_source_server_path'],
                         "catalina.properties"),
            os.path.join(deployer.mdict['pki_instance_configuration_path'],
                         "catalina.properties"))

        # Link /etc/pki/<instance>/ciphers.info
        # to /usr/share/pki/server/conf/ciphers.info.
        deployer.symlink.create(
            os.path.join(deployer.mdict['pki_source_server_path'],
                         "ciphers.info"),
            os.path.join(deployer.mdict['pki_instance_configuration_path'],
                         "ciphers.info"))

        # Link /etc/pki/<instance>/context.xml
        # to /usr/share/tomcat/conf/context.xml.
        deployer.symlink.create(
            os.path.join(deployer.mdict['CATALINA_HOME'],
                         "conf",
                         "context.xml"),
            os.path.join(deployer.mdict['pki_instance_configuration_path'],
                         "context.xml"))

        # Link /etc/pki/<instance>/logging.properties
        # to /usr/share/pki/server/conf/logging.properties.
        deployer.symlink.create(
            os.path.join(deployer.mdict['pki_source_server_path'],
                         "logging.properties"),
            os.path.join(deployer.mdict['pki_instance_configuration_path'],
                         "logging.properties"))

        # Link /etc/pki/<instance>/web.xml
        # to /usr/share/tomcat/conf/web.xml.
        deployer.symlink.create(
            os.path.join(deployer.mdict['CATALINA_HOME'],
                         "conf",
                         "web.xml"),
            os.path.join(deployer.mdict['pki_instance_configuration_path'],
                         "web.xml"))

        # create /etc/sysconfig/<instance>
        deployer.file.copy_with_slot_substitution(
            deployer.mdict['pki_source_tomcat_conf'],
            deployer.mdict['pki_target_tomcat_conf_instance_id'],
            uid=0, gid=0, overwrite_flag=True)

        # create /var/lib/pki/<instance>/conf/tomcat.conf
        deployer.file.copy_with_slot_substitution(
            deployer.mdict['pki_source_tomcat_conf'],
            deployer.mdict['pki_target_tomcat_conf'],
            overwrite_flag=True)

        # Deploy ROOT web application
        deployer.deploy_webapp(
            "ROOT",
            os.path.join(
                config.PKI_DEPLOYMENT_SOURCE_ROOT,
                "server",
                "webapps",
                "ROOT"),
            os.path.join(
                deployer.mdict['pki_source_server_path'],
                "Catalina",
                "localhost",
                "ROOT.xml"))

        # Deploy pki web application which includes themes,
        # admin templates, and JS libraries
        deployer.deploy_webapp(
            "pki",
            os.path.join(
                config.PKI_DEPLOYMENT_SOURCE_ROOT,
                "server",
                "webapps",
                "pki"),
            os.path.join(
                deployer.mdict['pki_source_server_path'],
                "Catalina",
                "localhost",
                "pki.xml"))

        # Create Tomcat instance library
        deployer.directory.create(deployer.mdict['pki_instance_lib'])
        for name in os.listdir(deployer.mdict['pki_tomcat_lib_path']):
            deployer.symlink.create(
                os.path.join(
                    deployer.mdict['pki_tomcat_lib_path'],
                    name),
                os.path.join(
                    deployer.mdict['pki_instance_lib'],
                    name))

        for name in os.listdir('/usr/share/pki/server/lib'):
            deployer.symlink.create(
                os.path.join(
                    '/usr/share/pki/server/lib',
                    name),
                os.path.join(
                    deployer.mdict['pki_instance_lib'],
                    name))

        # Link /var/lib/pki/<instance>/lib/log4j.properties
        # to /usr/share/pki/server/conf/log4j.properties.
        deployer.symlink.create(
            os.path.join(deployer.mdict['pki_source_server_path'],
                         "log4j.properties"),
            deployer.mdict['pki_instance_lib_log4j_properties'])

        # Link /var/lib/pki/<instance>/common to /usr/share/pki/server/common
        deployer.symlink.create(
            '/usr/share/pki/server/common',
            deployer.mdict['pki_tomcat_common_path'])

        deployer.directory.create(deployer.mdict['pki_tomcat_tmpdir_path'])

        deployer.directory.create(deployer.mdict['pki_tomcat_work_path'])
        deployer.directory.create(
            deployer.mdict['pki_tomcat_work_catalina_path'])
        deployer.directory.create(
            deployer.mdict['pki_tomcat_work_catalina_host_path'])
        deployer.directory.create(
            deployer.mdict['pki_tomcat_work_catalina_host_run_path'])
        deployer.directory.create(
            deployer.mdict['pki_tomcat_work_catalina_host_subsystem_path'])
        # establish Tomcat instance logs
        # establish Tomcat instance registry
        # establish Tomcat instance convenience symbolic links
        deployer.symlink.create(
            deployer.mdict['pki_tomcat_bin_path'],
            deployer.mdict['pki_tomcat_bin_link'])

        # create systemd links
        deployer.symlink.create(
            deployer.mdict['pki_tomcat_systemd'],
            deployer.mdict['pki_instance_systemd_link'],
            uid=0, gid=0)
        user = deployer.mdict['pki_user']
        group = deployer.mdict['pki_group']
        if user != 'pkiuser' or group != 'pkiuser':
            deployer.systemd.set_override(
                'Service', 'User', user, 'user.conf')
            deployer.systemd.set_override(
                'Service', 'Group', group, 'user.conf')
        deployer.systemd.write_overrides()
        deployer.systemd.daemon_reload()

        # establish shared NSS security databases for this instance
        deployer.directory.create(deployer.mdict['pki_database_path'])
        # establish instance convenience symbolic links
        deployer.symlink.create(
            deployer.mdict['pki_database_path'],
            deployer.mdict['pki_instance_database_link'])
        deployer.symlink.create(
            deployer.mdict['pki_instance_configuration_path'],
            deployer.mdict['pki_instance_conf_link'])
        deployer.symlink.create(
            deployer.mdict['pki_instance_log_path'],
            deployer.mdict['pki_instance_logs_link'])

        # create Tomcat instance systemd service link
        deployer.symlink.create(deployer.mdict['pki_systemd_service'],
                                deployer.mdict['pki_systemd_service_link'])

        # create instance registry
        deployer.file.copy_with_slot_substitution(
            deployer.mdict['pki_source_registry'],
            os.path.join(deployer.mdict['pki_instance_registry_path'],
                         deployer.mdict['pki_instance_name']),
            overwrite_flag=True)

    def destroy(self, deployer):

        config.pki_log.info(log.INSTANCE_DESTROY_1, __name__,
                            extra=config.PKI_INDENTATION_LEVEL_1)

        # if this is not the last subsystem, skip
        if len(deployer.instance.tomcat_instance_subsystems()) != 0:
            return

        # remove Tomcat instance systemd service link
        deployer.symlink.delete(deployer.systemd.systemd_link)

        # delete systemd override directories
        if deployer.directory.exists(deployer.systemd.base_override_dir):
            deployer.directory.delete(deployer.systemd.base_override_dir)
        if deployer.directory.exists(deployer.systemd.nuxwdog_override_dir):
            deployer.directory.delete(deployer.systemd.nuxwdog_override_dir)
        deployer.systemd.daemon_reload()

        # remove Tomcat instance base
        deployer.directory.delete(deployer.mdict['pki_instance_path'])
        # remove Tomcat instance logs
        deployer.directory.delete(deployer.mdict['pki_instance_log_path'])
        # remove shared NSS security database path for this instance
        deployer.directory.delete(deployer.mdict['pki_database_path'])
        # remove Tomcat instance configuration
        deployer.directory.delete(
            deployer.mdict['pki_instance_configuration_path'])
        # remove PKI 'tomcat.conf' instance file
        deployer.file.delete(
            deployer.mdict['pki_target_tomcat_conf_instance_id'])
        # remove Tomcat instance registry
        deployer.directory.delete(
            deployer.mdict['pki_instance_registry_path'])


# Callback only when the /usr/share/pki/server/conf directory
# Is getting copied to the etc tree.
# Don't copy the shared ldif files:
# schema.ldif, manager.ldif, database.ldif
def file_ignore_callback_src_server(src, names):
    config.pki_log.info(log.FILE_EXCLUDE_CALLBACK_2, src, names,
                        extra=config.PKI_INDENTATION_LEVEL_1)

    return {
        'catalina.properties',
        'ciphers.info',
        'schema.ldif',
        'database.ldif',
        'manager.ldif',
        'pki.xml',
        'log4j.properties',
        'logging.properties',
        'serverCertNick.conf'
    }
