"""
Dual class
"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# Last changed: 2010-05-09
from ufl.algorithms import preprocess

from dolfin import replace, lhs, rhs, homogenize, assemble, plot, interactive
from dolfin import VariationalProblem, Function

from dolfin.cpp import solve, MeshFunction

from dolfin.adaptivity.formmanipulation import construct_dual_form, extract_mesh

__all__ = ["Dual"]

class Dual:
    """
    Primal problem: Find u such that

      F(v; u) = 0,   for all v

    Linearized dual problem with goal M = M(u): Find du such that

    (dF/du)^*(v, du; u) - dM/du (v; u) = 0,   for all v

    Linear example:

    Primal problem:

      F(v; u) = a(v, u) - l(v)
      M(u) = M(u)

    Dual problem:

      a^*(v, du) = M(v)

    """
    def __init__(self, F, bcs, goal, u=None, exterior_facets=None):
        """
        For now (and in the foreseeable future), we can represent the
        dual problem as a linear, stationary variational problem.
        """
        self.F = construct_dual_form(F, goal, u)
        self.bcs = homogenize(bcs)
        self.u = u
        self.exterior_facets = exterior_facets


    def solve(self, u_h=None):
        """
        Solving the dual problem is slightly trickier than it seems.

        The dual problem reads: for given u, find w such that

          a'*(v, w; u) = M'(v; u),   for all v

        The form a is initialized with some Function u. This function
        u is the one we have to use to take the Frechet derivative of
        a.

        However, we actually want to solve with u = u_h:

          a'*(v, w; u_h) = M'(v; u_h),   for all v

        This seems to be magically taken care of by dolfin but should
        be kept in mind.

        """

        # Is this necessary?
        if u_h is None:
            F = self.F
        else:
            F = replace(self.F, {self.u: u_h})

        a, L = lhs(F), rhs(F)


        A = assemble(a)
        b = assemble(L, exterior_facet_domains=self.exterior_facets)

        if self.bcs is not None:
            for bc in self.bcs:
                bc.apply(A, b)

        preprocessed = preprocess(F)
        V = preprocessed.form_data().original_arguments[0].function_space()
        solution = Function(V)
        solve(A, solution.vector(), b)
        return solution
