<?php
/* Copyright (C) 2015   Jean-François Ferry     <jfefe@aternatik.fr>
 * Copyright (C) 2016	Laurent Destailleur		<eldy@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

 use Luracast\Restler\RestException;

 require_once DOL_DOCUMENT_ROOT.'/projet/class/project.class.php';
 require_once DOL_DOCUMENT_ROOT.'/projet/class/task.class.php';

/**
 * API class for projects
 *
 * @access protected
 * @class  DolibarrApiAccess {@requires user,external}
 */
class Projects extends DolibarrApi
{

    /**
     * @var array   $FIELDS     Mandatory fields, checked when create and update object
     */
    static $FIELDS = array(
        'ref',
        'title'
    );

    /**
     * @var Project $project {@type Project}
     */
    public $project;

    /**
     * Constructor
     */
    function __construct()
    {
		global $db, $conf;
		$this->db = $db;
        $this->project = new Project($this->db);
        $this->task = new Task($this->db);
    }

    /**
     * Get properties of a project object
     *
     * Return an array with project informations
     *
     * @param       int         $id         ID of project
     * @return 	array|mixed data without useless information
	 *
     * @throws 	RestException
     */
    function get($id)
    {
		if(! DolibarrApiAccess::$user->rights->projet->lire) {
			throw new RestException(401);
		}

        $result = $this->project->fetch($id);
        if( ! $result ) {
            throw new RestException(404, 'Project not found');
        }

		if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
		}

        $this->project->fetchObjectLinked();
		return $this->_cleanObjectDatas($this->project);
    }



    /**
     * List projects
     *
     * Get a list of projects
     *
     * @param string	       $sortfield	        Sort field
     * @param string	       $sortorder	        Sort order
     * @param int		       $limit		        Limit for list
     * @param int		       $page		        Page number
     * @param string   	       $thirdparty_ids	    Thirdparty ids to filter projects of. {@example '1' or '1,2,3'} {@pattern /^[0-9,]*$/i}
     * @param string           $sqlfilters          Other criteria to filter answers separated by a comma. Syntax example "(t.ref:like:'SO-%') and (t.date_creation:<:'20160101')"
     * @return  array                               Array of project objects
     */
    function index($sortfield = "t.rowid", $sortorder = 'ASC', $limit = 100, $page = 0, $thirdparty_ids = '', $sqlfilters = '') {
        global $db, $conf;

        $obj_ret = array();

        // case of external user, $thirdparty_ids param is ignored and replaced by user's socid
        $socids = DolibarrApiAccess::$user->societe_id ? DolibarrApiAccess::$user->societe_id : $thirdparty_ids;

        // If the internal user must only see his customers, force searching by him
        $search_sale = 0;
        if (! DolibarrApiAccess::$user->rights->societe->client->voir && !$socids) $search_sale = DolibarrApiAccess::$user->id;

        $sql = "SELECT t.rowid";
        if ((!DolibarrApiAccess::$user->rights->societe->client->voir && !$socids) || $search_sale > 0) $sql .= ", sc.fk_soc, sc.fk_user"; // We need these fields in order to filter by sale (including the case where the user can only see his prospects)
        $sql.= " FROM ".MAIN_DB_PREFIX."projet as t";

        if ((!DolibarrApiAccess::$user->rights->societe->client->voir && !$socids) || $search_sale > 0) $sql.= ", ".MAIN_DB_PREFIX."societe_commerciaux as sc"; // We need this table joined to the select in order to filter by sale

        $sql.= ' WHERE t.entity IN ('.getEntity('project', 1).')';
        if ((!DolibarrApiAccess::$user->rights->societe->client->voir && !$socids) || $search_sale > 0) $sql.= " AND t.fk_soc = sc.fk_soc";
        if ($socids) $sql.= " AND t.fk_soc IN (".$socids.")";
        if ($search_sale > 0) $sql.= " AND t.rowid = sc.fk_soc";		// Join for the needed table to filter by sale
        // Insert sale filter
        if ($search_sale > 0)
        {
            $sql .= " AND sc.fk_user = ".$search_sale;
        }
        // Add sql filters
        if ($sqlfilters)
        {
            if (! DolibarrApi::_checkFilters($sqlfilters))
            {
                throw new RestException(503, 'Error when validating parameter sqlfilters '.$sqlfilters);
            }
	        $regexstring='\(([^:\'\(\)]+:[^:\'\(\)]+:[^:\(\)]+)\)';
            $sql.=" AND (".preg_replace_callback('/'.$regexstring.'/', 'DolibarrApi::_forge_criteria_callback', $sqlfilters).")";
        }

        $sql.= $db->order($sortfield, $sortorder);
        if ($limit)	{
            if ($page < 0)
            {
                $page = 0;
            }
            $offset = $limit * $page;

            $sql.= $db->plimit($limit + 1, $offset);
        }

        dol_syslog("API Rest request");
        $result = $db->query($sql);

        if ($result)
        {
            $num = $db->num_rows($result);
            while ($i < min($num, ($limit <= 0 ? $num : $limit)))
            {
                $obj = $db->fetch_object($result);
                $project_static = new Project($db);
                if($project_static->fetch($obj->rowid)) {
                    $obj_ret[] = $this->_cleanObjectDatas($project_static);
                }
                $i++;
            }
        }
        else {
            throw new RestException(503, 'Error when retrieve project list : '.$db->lasterror());
        }
        if( ! count($obj_ret)) {
            throw new RestException(404, 'No project found');
        }
		return $obj_ret;
    }

    /**
     * Create project object
     *
     * @param   array   $request_data   Request data
     * @return  int     ID of project
     */
    function post($request_data = NULL)
    {
      if(! DolibarrApiAccess::$user->rights->projet->creer) {
			  throw new RestException(401, "Insuffisant rights");
		  }
        // Check mandatory fields
        $result = $this->_validate($request_data);

        foreach($request_data as $field => $value) {
            $this->project->$field = $value;
        }
        /*if (isset($request_data["lines"])) {
          $lines = array();
          foreach ($request_data["lines"] as $line) {
            array_push($lines, (object) $line);
          }
          $this->project->lines = $lines;
        }*/
        if ($this->project->create(DolibarrApiAccess::$user) < 0) {
            throw new RestException(500, "Error creating project", array_merge(array($this->project->error), $this->project->errors));
        }

        return $this->project->id;
    }

    /**
     * Get tasks of a project
     *
     * @param int   $id                     Id of project
     * @param int   $includetimespent       0=Return only list of tasks. 1=Include a summary of time spent, 2=Include details of time spent lines (2 is no implemented yet)
     * @return int
     *
     * @url	GET {id}/tasks
     */
    function getLines($id, $includetimespent=0) {
      if(! DolibarrApiAccess::$user->rights->projet->lire) {
		  	throw new RestException(401);
		  }

      $result = $this->project->fetch($id);
      if( ! $result ) {
         throw new RestException(404, 'Project not found');
      }

		  if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			  throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
      }
      $this->project->getLinesArray(DolibarrApiAccess::$user);
      $result = array();
      foreach ($this->project->lines as $line)      // $line is a task
      {
          if ($includetimespent == 1)
          {
              $timespent = $line->getSummaryOfTimeSpent(0);
          }
          if ($includetimespent == 1)
          {
                // TODO
                // Add class for timespent records and loop and fill $line->lines with records of timespent
          }
          array_push($result,$this->_cleanObjectDatas($line));
      }
      return $result;
    }


    /**
     * Get roles a user is assigned to a project with
     *
     * @param   int   $id             Id of project
     * @param   int   $userid         Id of user (0 = connected user)
     *
     * @url	GET {id}/roles
     *
     * @return int
     */
    function getRoles($id, $userid=0) {
        global $db;

        if(! DolibarrApiAccess::$user->rights->projet->lire) {
            throw new RestException(401);
        }

        $result = $this->project->fetch($id);
        if( ! $result ) {
            throw new RestException(404, 'Project not found');
        }

        if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
            throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
        }

        require_once DOL_DOCUMENT_ROOT.'/projet/class/task.class.php';
        $taskstatic=new Task($this->db);
        $userp = DolibarrApiAccess::$user;
        if ($userid > 0)
        {
            $userp = new User($this->db);
            $userp->fetch($userid);
        }
        $this->project->roles = $taskstatic->getUserRolesForProjectsOrTasks($userp, 0, $id, 0);
        $result = array();
        foreach ($this->project->roles as $line) {
            array_push($result,$this->_cleanObjectDatas($line));
        }
        return $result;
    }


    /**
     * Add a task to given project
     *
     * @param int   $id             Id of project to update
     * @param array $request_data   Projectline data
     *
     * @url	POST {id}/tasks
     *
     * @return int
     */
    /*
    function postLine($id, $request_data = NULL) {
      if(! DolibarrApiAccess::$user->rights->projet->creer) {
		  	throw new RestException(401);
		  }

      $result = $this->project->fetch($id);
      if( ! $result ) {
         throw new RestException(404, 'Project not found');
      }

		  if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			  throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
      }
			$request_data = (object) $request_data;
      $updateRes = $this->project->addline(
                        $request_data->desc,
                        $request_data->subprice,
                        $request_data->qty,
                        $request_data->tva_tx,
                        $request_data->localtax1_tx,
                        $request_data->localtax2_tx,
                        $request_data->fk_product,
                        $request_data->remise_percent,
                        $request_data->info_bits,
                        $request_data->fk_remise_except,
                        'HT',
                        0,
                        $request_data->date_start,
                        $request_data->date_end,
                        $request_data->product_type,
                        $request_data->rang,
                        $request_data->special_code,
                        $fk_parent_line,
                        $request_data->fk_fournprice,
                        $request_data->pa_ht,
                        $request_data->label,
                        $request_data->array_options,
                        $request_data->fk_unit,
                        $this->element,
                        $request_data->id
      );

      if ($updateRes > 0) {
        return $this->get($id)->line->rowid;

      }
      return false;
    }
    */

    /**
     * Update a task to given project
     *
     * @param int   $id             Id of project to update
     * @param int   $taskid         Id of task to update
     * @param array $request_data   Projectline data
     *
     * @url	PUT {id}/tasks/{taskid}
     *
     * @return object
     */
    /*
    function putLine($id, $lineid, $request_data = NULL) {
      if(! DolibarrApiAccess::$user->rights->projet->creer) {
		  	throw new RestException(401);
		  }

      $result = $this->project->fetch($id);
      if( ! $result ) {
         throw new RestException(404, 'Project not found');
      }

		  if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			  throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
      }
			$request_data = (object) $request_data;
      $updateRes = $this->project->updateline(
                        $lineid,
                        $request_data->desc,
                        $request_data->subprice,
                        $request_data->qty,
                        $request_data->remise_percent,
                        $request_data->tva_tx,
                        $request_data->localtax1_tx,
                        $request_data->localtax2_tx,
                        'HT',
                        $request_data->info_bits,
                        $request_data->date_start,
                        $request_data->date_end,
                        $request_data->product_type,
                        $request_data->fk_parent_line,
                        0,
                        $request_data->fk_fournprice,
                        $request_data->pa_ht,
                        $request_data->label,
                        $request_data->special_code,
                        $request_data->array_options,
                        $request_data->fk_unit
      );

      if ($updateRes > 0) {
        $result = $this->get($id);
        unset($result->line);
        return $this->_cleanObjectDatas($result);
      }
      return false;
    }*/



    /**
     * Update project general fields (won't touch lines of project)
     *
     * @param int   $id             Id of project to update
     * @param array $request_data   Datas
     *
     * @return int
     */
    function put($id, $request_data = NULL) {
      if(! DolibarrApiAccess::$user->rights->projet->creer) {
		  	throw new RestException(401);
		  }

        $result = $this->project->fetch($id);
        if ($result <= 0) {
            throw new RestException(404, 'Project not found');
        }

		if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
		}
        foreach($request_data as $field => $value) {
            if ($field == 'id') continue;
            $this->project->$field = $value;
        }

        if($this->project->update(DolibarrApiAccess::$user, 0) > 0)
        {
            return $this->get($id);
        }
        else
        {
            throw new RestException(500, $this->project->error);
        }
    }

    /**
     * Delete project
     *
     * @param   int     $id         Project ID
     *
     * @return  array
     */
    function delete($id)
    {
        if(! DolibarrApiAccess::$user->rights->projet->supprimer) {
			throw new RestException(401);
		}
        $result = $this->project->fetch($id);
        if( ! $result ) {
            throw new RestException(404, 'Project not found');
        }

		if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
		}

        if( ! $this->project->delete(DolibarrApiAccess::$user)) {
            throw new RestException(500, 'Error when delete project : '.$this->project->error);
        }

        return array(
            'success' => array(
                'code' => 200,
                'message' => 'Project deleted'
            )
        );

    }

    /**
     * Validate a project.
     * You can test this API with the following input message
     * { "notrigger": 0 }
     *
     * @param   int $id             Project ID
     * @param   int $notrigger      1=Does not execute triggers, 0= execute triggers
     *
     * @url POST    {id}/validate
     *
     * @return  array
     * FIXME An error 403 is returned if the request has an empty body.
     * Error message: "Forbidden: Content type `text/plain` is not supported."
     * Workaround: send this in the body
     * {
     *   "notrigger": 0
     * }
     */
    function validate($id, $notrigger=0)
    {
        if(! DolibarrApiAccess::$user->rights->projet->creer) {
			throw new RestException(401);
		}
        $result = $this->project->fetch($id);
        if( ! $result ) {
            throw new RestException(404, 'Project not found');
        }

		if( ! DolibarrApi::_checkAccessToResource('project',$this->project->id)) {
			throw new RestException(401, 'Access not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$result = $this->project->setValid(DolibarrApiAccess::$user, $notrigger);
		if ($result == 0) {
		    throw new RestException(500, 'Error nothing done. May be object is already validated');
		}
		if ($result < 0) {
		    throw new RestException(500, 'Error when validating Project: '.$this->project->error);
		}

        return array(
            'success' => array(
                'code' => 200,
                'message' => 'Project validated'
            )
        );
    }


    /**
     * Clean sensible object datas
     *
     * @param   object  $object    Object to clean
     * @return    array    Array of cleaned object properties
     */
    function _cleanObjectDatas($object) {

        $object = parent::_cleanObjectDatas($object);

        unset($object->titre);
        unset($object->datec);
        unset($object->datem);
        unset($object->barcode_type);
        unset($object->barcode_type_code);
        unset($object->barcode_type_label);
        unset($object->barcode_type_coder);
        unset($object->cond_reglement_id);
        unset($object->cond_reglement);
        unset($object->fk_delivery_address);
        unset($object->shipping_method_id);
        unset($object->fk_account);
        unset($object->note);
        unset($object->fk_incoterms);
        unset($object->libelle_incoterms);
        unset($object->location_incoterms);
        unset($object->name);
        unset($object->lastname);
        unset($object->firstname);
        unset($object->civility_id);
        unset($object->mode_reglement_id);
        unset($object->country);
        unset($object->country_id);
        unset($object->country_code);

        unset($object->weekWorkLoad);
        unset($object->weekWorkLoad);

        //unset($object->lines);            // for task we use timespent_lines, but for project we use lines

        unset($object->total_ht);
        unset($object->total_tva);
        unset($object->total_localtax1);
        unset($object->total_localtax2);
        unset($object->total_ttc);

        return $object;
    }

    /**
     * Validate fields before create or update object
     *
     * @param   array           $data   Array with data to verify
     * @return  array
     * @throws  RestException
     */
    function _validate($data)
    {
        $object = array();
        foreach (self::$FIELDS as $field) {
            if (!isset($data[$field]))
                throw new RestException(400, "$field field missing");
            $object[$field] = $data[$field];

        }
        return $object;
    }


    // TODO
    // getSummaryOfTimeSpent
}
