/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "bookmarkselector.h"
#include <qpopupmenu.h>
#include <kiconloader.h>
#include <kglobalsettings.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <assert.h>

#include "bookmark.h"
#include "dolphinsettings.h"
#include "dolphinview.h"
#include "dolphin.h"
#include "urlnavigator.h"

BookmarkSelector::BookmarkSelector(URLNavigator* parent) :
    URLButton(parent),
    m_selectedIndex(0)
{
    setFocusPolicy(QWidget::NoFocus);

    m_bookmarksMenu = new QPopupMenu(this);
    const QValueList<Bookmark>& bookmarks = DolphinSettings::instance().bookmarks();

    // The last bookmark entry is reserved as 'Custom Path' entry, hence only add
    // the previous bookmarks to the menu.
    bool isEditable = false;
    const int count = bookmarks.count() - 1;
    for (int i = 0; i < count; ++i) {
        const Bookmark& bookmark = bookmarks[i];
        if (bookmark.isEditable() != isEditable) {
            isEditable = bookmark.isEditable();
            m_bookmarksMenu->insertSeparator();
        }
        m_bookmarksMenu->insertItem(SmallIcon(bookmark.icon()),
                                    bookmark.name(),
                                    i);
    }
    connect(m_bookmarksMenu, SIGNAL(activated(int)),
            this, SLOT(slotBookmarkActivated(int)));

    QPixmap pixmap = SmallIcon(bookmarks[m_selectedIndex].icon());
    setPixmap(pixmap);
    setMinimumWidth(pixmap.width() + 2);
    setPopup(m_bookmarksMenu);
}


BookmarkSelector::~BookmarkSelector()
{
}

void BookmarkSelector::updateSelection(const KURL& url)
{
    const QValueList<Bookmark>& bookmarks = DolphinSettings::instance().bookmarks();
    const int count = bookmarks.count();

    int maxLength = 0;
    m_selectedIndex = -1;

    // Search the bookmark which is equal to the URL or at least is a parent URL.
    // If there are more than one possible parent URL candidates, choose the bookmark
    // which covers the bigger range of the URL.
    for (int i = 0; i < count; ++i) {
        const KURL& bookmarkURL = bookmarks[i].url();
        if (bookmarkURL.isParentOf(url)) {
            const int length = bookmarkURL.prettyURL().length();
            if (length > maxLength) {
                m_selectedIndex = i;
                maxLength = length;
            }
        }
    }

    if (m_selectedIndex < 0) {
        // No matching URL has been found. Use the last URL of the bookmarks
        // as it is defined as 'Custom Path'.
        m_selectedIndex = count - 1;
    }

    updateIcon();
}

Bookmark BookmarkSelector::selectedBookmark() const
{
    const QValueList<Bookmark>& bookmarks = DolphinSettings::instance().bookmarks();
    assert(m_selectedIndex < static_cast<int>(bookmarks.count()));
    return bookmarks[m_selectedIndex];
}

void BookmarkSelector::drawButton(QPainter* painter)
{
    const int buttonWidth  = width();
    const int buttonHeight = height();

    QColor backgroundColor;
    QColor foregroundColor;
    const bool isHighlighted = isDisplayHintEnabled(EnteredHint) ||
        isDisplayHintEnabled(DraggedHint);
    if (isHighlighted) {
        backgroundColor = KGlobalSettings::highlightColor();
        foregroundColor = KGlobalSettings::highlightedTextColor();
    }
    else {
        backgroundColor = colorGroup().background();
        foregroundColor = KGlobalSettings::buttonTextColor();
    }

    // dimm the colors if the parent view does not have the focus
    const DolphinView* parentView = urlNavigator()->dolphinView();
    const Dolphin& dolphin = Dolphin::mainWin();

    const bool isActive = (dolphin.activeView() == parentView);
    if (!isActive) {
        QColor dimmColor(colorGroup().background());
        foregroundColor = mixColors(foregroundColor, dimmColor);
        if (isHighlighted) {
            backgroundColor = mixColors(backgroundColor, dimmColor);
        }
    }

    if (!(isDisplayHintEnabled(ActivatedHint) && isActive) && !isHighlighted) {
        // dimm the foreground color by mixing it with the background
        foregroundColor = mixColors(foregroundColor, backgroundColor);
        painter->setPen(foregroundColor);
    }

    // draw button backround
    painter->setPen(NoPen);
    painter->setBrush(backgroundColor);
    painter->drawRect(0, 0, buttonWidth, buttonHeight);

    if (!isHighlighted) {
        painter->setPen(foregroundColor);
        painter->drawLine(buttonWidth - 1, 2, buttonWidth - 1, buttonHeight - 3);
    }

    // draw icon
    const QPixmap* icon = pixmap();
    if (icon != 0) {
        const int x = (buttonWidth - icon->width()) / 2;
        const int y = (buttonHeight - icon->height()) / 2;
        painter->drawPixmap(x, y, *icon);
    }
}

void BookmarkSelector::slotBookmarkActivated(int index)
{
    m_selectedIndex = index;
    updateIcon();
    emit bookmarkActivated(index);
}

void BookmarkSelector::updateIcon()
{
    const QValueList<Bookmark>& bookmarks = DolphinSettings::instance().bookmarks();
    assert(m_selectedIndex < static_cast<int>(bookmarks.count()));
    setPixmap(SmallIcon(bookmarks[m_selectedIndex].icon()));
}

#include "bookmarkselector.moc"

