/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "dolphincontextmenu.h"
#include <kactioncollection.h>
#include <qvaluevector.h>
#include <ktrader.h>
#include <klocale.h>
#include <krun.h>
#include <qdir.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kiconloader.h>
#include <kaction.h>
#include <kpropertiesdialog.h>
#include <kdesktopfile.h>
#include <assert.h>
#include <ksortablevaluelist.h>

#include "dolphin.h"
#include "dolphinview.h"

DolphinContextMenu::DolphinContextMenu(DolphinView* parent,
                                       KFileItem* fileInfo,
                                       const QPoint& pos) :
   m_dolphinView(parent),
   m_fileInfo(fileInfo),
   m_pos(pos)
{
}

void DolphinContextMenu::open()
{
    if (m_fileInfo == 0) {
        openViewportContextMenu();
    }
    else {
        openItemContextMenu();
    }
}

DolphinContextMenu::~DolphinContextMenu()
{
}

void DolphinContextMenu::openViewportContextMenu()
{
    // Parts of the following code have been taken
    // from the class KonqOperations located in
    // libqonq/konq_operations.h of Konqueror.
    // (Copyright (C) 2000  David Faure <faure@kde.org>)

    assert(m_fileInfo == 0);
    const int propertiesID = 100;

    KPopupMenu* popup = new KPopupMenu(m_dolphinView);
    Dolphin& dolphin = Dolphin::mainWin();

    // setup 'Create New' menu
    KPopupMenu* createNewMenu = new KPopupMenu();

    KAction* createFolderAction = dolphin.actionCollection()->action("create_folder");
    if (createFolderAction != 0) {
        createFolderAction->plug(createNewMenu);
    }

    createNewMenu->insertSeparator();

    KAction* action = 0;

    QPtrListIterator<KAction> fileGrouptIt(dolphin.fileGroupActions());
    while ((action = fileGrouptIt.current()) != 0) {
        action->plug(createNewMenu);
        ++fileGrouptIt;
    }

    // TODO: not used yet. See documentation of Dolphin::linkGroupActions()
    // and Dolphin::linkToDeviceActions() in the header file for details.
    //
    //createNewMenu->insertSeparator();
    //
    //QPtrListIterator<KAction> linkGroupIt(dolphin.linkGroupActions());
    //while ((action = linkGroupIt.current()) != 0) {
    //    action->plug(createNewMenu);
    //    ++linkGroupIt;
    //}
    //
    //KPopupMenu* linkToDeviceMenu = new KPopupMenu();
    //QPtrListIterator<KAction> linkToDeviceIt(dolphin.linkToDeviceActions());
    //while ((action = linkToDeviceIt.current()) != 0) {
    //    action->plug(linkToDeviceMenu);
    //    ++linkToDeviceIt;
    //}
    //
    //createNewMenu->insertItem(i18n("Link to Device"), linkToDeviceMenu);

    popup->insertItem(SmallIcon("filenew"), i18n("Create New"), createNewMenu);
    popup->insertSeparator();

    KAction* pasteAction = dolphin.actionCollection()->action(KStdAction::stdName(KStdAction::Paste));
    pasteAction->plug(popup);

    // setup 'View Mode' menu
    KPopupMenu* viewModeMenu = new KPopupMenu();

    KAction* iconsMode = dolphin.actionCollection()->action("icons");
    iconsMode->plug(viewModeMenu);

    KAction* detailsMode = dolphin.actionCollection()->action("details");
    detailsMode->plug(viewModeMenu);

    KAction* previewsMode = dolphin.actionCollection()->action("previews");
    previewsMode->plug(viewModeMenu);

    popup->insertItem(i18n("View Mode"), viewModeMenu);
    popup->insertSeparator();

    popup->insertItem(i18n("Properties..."), propertiesID);

    int id = popup->exec(m_pos);
    if (id == propertiesID) {
        new KPropertiesDialog(dolphin.activeView()->url());
    }

    popup->deleteLater();
}

void DolphinContextMenu::openItemContextMenu()
{
    // Parts of the following code have been taken
    // from the class KonqOperations located in
    // libqonq/konq_operations.h of Konqueror.
    // (Copyright (C) 2000  David Faure <faure@kde.org>)

    assert(m_fileInfo != 0);
    const int submenuID = 90;
    const int openWithIDStart = 100;
    const int actionsIDStart = 1000;

    KPopupMenu* popup = new KPopupMenu(m_dolphinView);
    Dolphin& dolphin = Dolphin::mainWin();

    // insert 'Cut', 'Copy' and 'Paste'
    const KStdAction::StdAction actionNames[] = { KStdAction::Cut, KStdAction::Copy, KStdAction::Paste };
    const int count = sizeof(actionNames) / sizeof(KStdAction::StdAction);
    for (int i = 0; i < count; ++i) {
        KAction* action = dolphin.actionCollection()->action(KStdAction::stdName(actionNames[i]));
        if (action != 0) {
            action->plug(popup);
        }
    }
    popup->insertSeparator();

    // insert 'Rename'
    KAction* renameAction = dolphin.actionCollection()->action("rename");
    renameAction->plug(popup);

    // insert 'Move to Trash' for local URLs, otherwise insert 'Delete'
    const KURL& url = dolphin.activeView()->url();
    if (url.isLocalFile()) {
        KAction* moveToTrashAction = dolphin.actionCollection()->action("move_to_trash");
        moveToTrashAction->plug(popup);
    }
    else {
        KAction* deleteAction = dolphin.actionCollection()->action("delete");
        deleteAction->plug(popup);
    }
    popup->insertSeparator();

    // Prepare 'Open With' sub menu. Usually a sub menu is created, where all applications
    // are listed which are registered to open the item. As last entry "Other..." will be
    // attached which allows to select a custom application. If no applications are registered
    // no sub menu is created at all, only "Open With..." will be offered.
    QValueVector<KService::Ptr> openWithVector;
    KMimeType::Ptr mimePtr = KMimeType::findByURL(m_fileInfo->url());
    const QString mimeComment(mimePtr->comment());

    // Check whether all selected items have the same mime type. As soon as one
    // item has a different MIME type, disable the 'Open With' entry.
    bool enableOpenWithItem = true;
    KURL::List urls = m_dolphinView->selectedURLs();
    for(KURL::List::ConstIterator it = urls.begin(); it != urls.end(); ++it) {
        if (mimeComment != KMimeType::findByURL((*it).url())->comment()) {
            enableOpenWithItem = false;
            break;
        }
    }

    int openWithID = -1;
    if (enableOpenWithItem) {
        // fill the 'Open with' sub menu with application types
        KTrader::OfferList offers = KTrader::self()->query(mimePtr->name(),
                                                           "Type == 'Application'");
        int index = openWithIDStart;
        if (offers.count() > 0) {
            KTrader::OfferList::Iterator it;
            KPopupMenu* openWithMenu = new KPopupMenu();
            for(it = offers.begin(); it != offers.end(); ++it) {
                // The offer list from the KTrader returns duplicate
                // application entries. Although this seems to be a configuration
                // problem outside the scope of Dolphin, duplicated entries just
                // will be skipped here.
                const QString appName((*it)->name());
                if (!containsEntry(openWithMenu, appName)) {
                    openWithMenu->insertItem((*it)->pixmap(KIcon::Small),
                                            appName, index);
                    openWithVector.append(*it);
                    ++index;
                }
            }

            openWithMenu->insertSeparator();
            openWithMenu->insertItem(i18n("&Other..."), index);
            popup->insertItem(i18n("Open With"), openWithMenu);
        }
        else {
            // No applications are registered, hence just offer
            // a "Open With..." item instead of a sub menu containing
            // only one entry.
            popup->insertItem(i18n("Open With..."), openWithIDStart);
        }
        openWithID = index;
    }
    else {
        // At least one of the selected items has a different MIME type. In this case
        // just show a disabled "Open With..." entry.
        popup->insertItem(i18n("Open With..."), openWithIDStart);
        popup->setItemEnabled(openWithIDStart, false);
    }


    popup->setItemEnabled(openWithID, enableOpenWithItem);

    // Insert 'Actions' sub menu
    KPopupMenu* actionsMenu = new KPopupMenu();

    QValueVector<KDEDesktopMimeType::Service> actionsVector;
    int actionsIndex = 0;

    const QString mimeType(mimePtr->name());
    const QString mimeGroup(mimeType.left(mimeType.find('/')));
    QStringList dirs = KGlobal::dirs()->findDirs("data", "dolphin/servicemenus/");

    KPopupMenu* menu = 0;
    for (QStringList::ConstIterator dirIt = dirs.begin(); dirIt != dirs.end(); ++dirIt ) {
        QDir dir(*dirIt);
        QStringList entries = dir.entryList("*.desktop", QDir::Files);

        for (QStringList::ConstIterator entryIt = entries.begin(); entryIt != entries.end(); ++entryIt) {
            KSimpleConfig cfg(*dirIt + *entryIt, true);
            cfg.setDesktopGroup();
            if ((cfg.hasKey("Actions") || cfg.hasKey("X-KDE-GetActionMenu"))  && cfg.hasKey("ServiceTypes")) {
                const QStringList types = cfg.readListEntry("ServiceTypes");
                for (QStringList::ConstIterator it = types.begin(); it != types.end(); ++it) {
                    // check whether the mime type is equal or whether the
                    // mimegroup (e. g. image/*) is supported
                    const bool insert = (*it == mimeType) ||
                                        ((*it).right(1) == "*") && ((*it).left((*it).find('/')) == mimeGroup);
                    if (insert) {
                        menu = actionsMenu;

                        const QString submenuName = cfg.readEntry( "X-KDE-Submenu" );
                        if (!submenuName.isEmpty()) {
                            menu = new KPopupMenu();
                            actionsMenu->insertItem(submenuName, menu, submenuID);
                        }

                        QValueList<KDEDesktopMimeType::Service> userServices =
                            KDEDesktopMimeType::userDefinedServices(*dirIt + *entryIt, true);

                        QValueList<KDEDesktopMimeType::Service>::Iterator serviceIt;
                        for (serviceIt = userServices.begin(); serviceIt != userServices.end(); ++serviceIt) {
                            KDEDesktopMimeType::Service service = (*serviceIt);
                            menu->insertItem(SmallIcon(service.m_strIcon),
                                             service.m_strName,
                                             actionsIDStart + actionsIndex);
                            actionsVector.append(service);
                            actionsIndex++;
                        }
                    }
                }
            }
        }
    }

    const int itemsCount = actionsMenu->count();
    if (itemsCount == 0) {
        // no actions are available at all, hence show the "Actions"
        // submenu disabled
        actionsMenu->setEnabled(false);
    }

    if (itemsCount == 1) {
        // Exactly one item is available. Instead of showing a sub menu with
        // only one item, show the item directly in the root menu.
        if (menu == actionsMenu) {
            // The item is an action, hence show the action in the root menu.
            const QString text(actionsMenu->text(actionsIDStart));
            const QIconSet* iconSet = actionsMenu->iconSet(actionsIDStart);
            if (iconSet == 0) {
                popup->insertItem(text, actionsIDStart);
            }
            else {
                popup->insertItem(*iconSet, text, actionsIDStart);
            }
        }
        else {
            // The item is a sub menu, hence show the sub menu in the root menu.
            popup->insertItem(actionsMenu->text(submenuID), menu);
        }
        actionsMenu->deleteLater();
        actionsMenu = 0;
    }
    else {
        popup->insertItem(i18n("Actions"), actionsMenu);
    }

    // insert 'Properties...' entry
    popup->insertSeparator();
    KAction* propertiesAction = dolphin.actionCollection()->action("properties");
    propertiesAction->plug(popup);

    int id = popup->exec(m_pos);

    if (id >= actionsIDStart) {
        // one of the 'Actions' items has been selected
        KDEDesktopMimeType::executeService(urls, actionsVector[id - actionsIDStart]);
    }
    else if (id >= openWithIDStart) {
        // one of the 'Open With' items has been selected
        if (id == openWithID) {
            // the item 'Other...' has been selected
            KRun::displayOpenWithDialog(urls);
        }
        else {
            KService::Ptr servicePtr = openWithVector[id - openWithIDStart];
            KRun::run(*servicePtr, urls);
        }
    }

    openWithVector.clear();
    actionsVector.clear();
    popup->deleteLater();
}

bool DolphinContextMenu:: containsEntry(const KPopupMenu* menu,
                                        const QString& entryName) const
{
    assert(menu != 0);

    const uint count = menu->count();
    for (uint i = 0; i < count; ++i) {
        const int id = menu->idAt(i);
        if (menu->text(id) == entryName) {
            return true;
        }
    }

    return false;
}

