/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz                                      *
 *   peter.penz@gmx.at                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "undomanager.h"
#include <klocale.h>
#include <kio/netaccess.h>
#include <qtimer.h>
#include <assert.h>

#include "dolphin.h"
#include "dolphinstatusbar.h"
#include "progressindicator.h"

DolphinCommand::DolphinCommand() :
    m_type(Copy)
{
    // Implementation note: DolphinCommands are stored in a QValueList, whereas
    // QValueList requires a default constructor of the added class.
    // Instead of expressing this implementation detail to the interface by adding a
    // Type::Undefined just Type::Copy is used to assure that all members have
    // a defined state.
}

DolphinCommand::DolphinCommand(Type type,
                              const KURL::List& source,
                              const KURL& dest) :
    m_type(type),
    m_source(source),
    m_dest(dest)
{
}

DolphinCommand::~DolphinCommand()
{
}

DolphinCommand& DolphinCommand::operator = (const DolphinCommand& command)
{
    m_type = command.m_type;
    m_source = command.m_source;
    m_dest = command.m_dest;
    return *this;
}

UndoManager& UndoManager::instance()
{
    static UndoManager* instance = 0;
    if (instance == 0) {
        instance = new UndoManager();
    }
    return *instance;
}

void UndoManager::addCommand(const DolphinCommand& command)
{
    ++m_historyIndex;
    m_history.insert(m_history.at(m_historyIndex), command);
    emit undoAvailable(true);
    emit undoTextChanged(i18n("Undo: %1").arg(commandText(command)));

    // Prevent an endless growing of the Undo history: remembering
    // 100 operations should be enough...
    if (m_historyIndex > 100) {
        m_history.erase(m_history.begin());
        --m_historyIndex;
    }
}

void UndoManager::undo()
{
    if (m_historyIndex < 0) {
        return;
    }

    m_progressIndicator = new ProgressIndicator(i18n("Executing undo operation..."),
                                                i18n("Executed undo operation."),
                                                m_history[m_historyIndex].source().count());

    const DolphinCommand command = m_history[m_historyIndex];
    --m_historyIndex;
    if (m_historyIndex < 0) {
        emit undoAvailable(false);
        emit undoTextChanged(i18n("Undo"));
    }
    else {
        emit undoTextChanged(i18n("Undo: %1").arg(commandText(m_history[m_historyIndex])));
    }

    if (m_historyIndex < static_cast<int>(m_history.count()) - 1) {
        emit redoAvailable(true);
        emit redoTextChanged(i18n("Redo: %1").arg(commandText(command)));
    }
    else {
        emit redoAvailable(false);
        emit redoTextChanged(i18n("Redo"));
    }

    KURL::List sourceURLs = command.source();
    KURL::List::Iterator it = sourceURLs.begin();
    const KURL::List::Iterator end = sourceURLs.end();
    const QString destURL(command.destination().prettyURL(+1));

    KIO::Job* job = 0;
    switch (command.type()) {
        case DolphinCommand::Link:
        case DolphinCommand::Copy: {
            KURL::List list;
            while (it != end) {
                const KURL deleteURL(destURL + (*it).filename());
                list.append(deleteURL);
                ++it;
            }
            job = KIO::del(list, false, false);
            break;
        }

        case DolphinCommand::Move: {
            KURL::List list;
            const KURL newDestURL((*it).directory());
            while (it != end) {
                const KURL newSourceURL(destURL + (*it).filename());
                list.append(newSourceURL);
                ++it;
            }
            job = KIO::move(list, newDestURL, false);
            break;
        }

        case DolphinCommand::Rename: {
            assert(sourceURLs.count() == 1);
            KIO::NetAccess::move(command.destination(), (*it));
            break;
        }

        case DolphinCommand::Trash: {
            while (it != end) {
                m_progressIndicator->execOperation();

                // TODO: use KIO::special for accessing the trash protocol. See
                // also Dolphin::slotJobResult() for further details.
                const QString originalFileName((*it).filename().section('-', 1));
                KURL newDestURL(destURL + originalFileName);
                KIO::NetAccess::move(*it, newDestURL);
                ++it;
            }
            break;
        }

        case DolphinCommand::CreateFolder:
        case DolphinCommand::CreateFile: {
            KIO::NetAccess::del(command.destination(), &Dolphin::mainWin());
            break;
        }
    }

    if (job != 0) {
        // Execute the jobs in a synchronous manner and forward the progress
        // information to the Dolphin statusbar.
        connect(job, SIGNAL(percent(KIO::Job*, unsigned long)),
                this, SLOT(slotPercent(KIO::Job*, unsigned long)));
        KIO::NetAccess::synchronousRun(job, &Dolphin::mainWin());
    }

    delete m_progressIndicator;
    m_progressIndicator = 0;
}

void UndoManager::redo()
{
    const int maxHistoryIndex = m_history.count() - 1;
    if (m_historyIndex >= maxHistoryIndex) {
        return;
    }
    ++m_historyIndex;

    m_progressIndicator = new ProgressIndicator(i18n("Executing redo operation..."),
                                                i18n("Executed redo operation."),
                                                m_history[m_historyIndex].source().count());

    const DolphinCommand command = m_history[m_historyIndex];
    if (m_historyIndex >= maxHistoryIndex) {
        emit redoAvailable(false);
        emit redoTextChanged(i18n("Redo"));
    }
    else {
        emit redoTextChanged(i18n("Redo: %1").arg(commandText(m_history[m_historyIndex + 1])));
    }

    emit undoAvailable(true);
    emit undoTextChanged(i18n("Undo: %1").arg(commandText(command)));

    Dolphin& dolphin = Dolphin::mainWin();

    KURL::List sourceURLs = command.source();
    KURL::List::Iterator it = sourceURLs.begin();
    const KURL::List::Iterator end = sourceURLs.end();

    KIO::Job* job = 0;
    switch (command.type()) {
        case DolphinCommand::Link: {
            job = KIO::link(sourceURLs, command.destination(), false);
            break;
        }

        case DolphinCommand::Copy: {
            job = KIO::copy(sourceURLs, command.destination(), false);
            break;
        }

        case DolphinCommand::Rename:
        case DolphinCommand::Move: {
            job = KIO::move(sourceURLs, command.destination(), false);
            break;
        }

        case DolphinCommand::Trash: {
            const QString destURL(command.destination().prettyURL());
            while (it != end) {
                m_progressIndicator->execOperation();

                // TODO: use KIO::special for accessing the trash protocol. See
                // also Dolphin::slotJobResult() for further details.
                const QString originalFileName((*it).filename().section('-', 1));
                KURL originalSourceURL(destURL + "/" + originalFileName);
                KIO::Job* moveToTrashJob = KIO::trash(originalSourceURL);
                KIO::NetAccess::synchronousRun(moveToTrashJob, &dolphin);
                ++it;
            }
            break;
        }

        case DolphinCommand::CreateFolder: {
            m_progressIndicator->execOperation();
            KIO::NetAccess::mkdir(command.destination(), &dolphin);
            break;
        }

        case DolphinCommand::CreateFile: {
            m_progressIndicator->execOperation();
            KURL::List::Iterator it = sourceURLs.begin();
            assert(sourceURLs.count() == 1);
            KIO::CopyJob* copyJob = KIO::copyAs(*it, command.destination(), false);
            copyJob->setDefaultPermissions(true);
            job = copyJob;
            break;
        }
    }

    if (job != 0) {
        // Execute the jobs in a synchronous manner and forward the progress
        // information to the Dolphin statusbar.
        connect(job, SIGNAL(percent(KIO::Job*, unsigned long)),
                this, SLOT(slotPercent(KIO::Job*, unsigned long)));
        KIO::NetAccess::synchronousRun(job, &dolphin);
    }

    delete m_progressIndicator;
    m_progressIndicator = 0;
}

UndoManager::UndoManager()
 :  m_historyIndex(-1),
    m_progressIndicator(0)
{
}

UndoManager::~UndoManager()
{
    delete m_progressIndicator;
    m_progressIndicator = 0;
}

QString UndoManager::commandText(const DolphinCommand& command) const
{
    QString text;
    switch (command.type()) {
        case DolphinCommand::Copy:         text = i18n("Copy"); break;
        case DolphinCommand::Move:         text = i18n("Move"); break;
        case DolphinCommand::Link:         text = i18n("Link"); break;
        case DolphinCommand::Rename:       text = i18n("Rename"); break;
        case DolphinCommand::Trash:        text = i18n("Move to Trash"); break;
        case DolphinCommand::CreateFolder: text = i18n("Create New Folder"); break;
        case DolphinCommand::CreateFile:   text = i18n("Create New File"); break;
        default: break;
    }
    return text;
}

void UndoManager::slotPercent(KIO::Job* /* job */, unsigned long /* percent */)
{
    // It is not allowed to update the progress indicator in the context
    // of this slot, hence do an asynchronous triggering.
    QTimer::singleShot(0, this, SLOT(updateProgress()));
}

void UndoManager::updateProgress()
{
    m_progressIndicator->execOperation();
}

#include "undomanager.moc"


