#!/usr/bin/python
#
# Copyright (c) 2018  Peter Pentchev <roam@ringlet.net>
#
# This file conforms to the GNU General Public License version 2.

import os
import pty
import unittest

import ddt
import six


def read_prompt(fd, check=True):
    """
    Read a series of characters ending in ": " and return it.
    If `check` is false, do not look for ": ", but read to
    the end of the stream and return everything.
    """
    buf = ''
    while True:
        try:
            s = os.read(fd, 1024).decode()
        except OSError:
            return buf
        buf = buf + s
        if check and buf.endswith(': '):
            return buf


@ddt.ddt
class TestDonkey(unittest.TestCase):
    @ddt.data(
        {
            'argv': ['donkey', '-h'],
            'dialog': [
            ],
            'result': 'usage:\tdonkey',
        },

        {
            'argv': ['donkey', '--help'],
            'dialog': [
            ],
            'result': 'usage:\tdonkey',
        },

        {
            'argv': ['donkey', '-v'],
            'dialog': [
            ],
            'result': 'donkey version ',
        },

        {
            'argv': ['donkey', '--version'],
            'dialog': [
            ],
            'result': 'donkey version ',
        },

        {
            'argv': ['donkey', '--features'],
            'dialog': [
            ],
            'result': 'Features: donkey=',
        },

        {
            'argv': ['donkey', '-X'],
            'dialog': [
            ],
            'result': 'usage:\tdonkey',
            'exit_code': 1,
        },

        {
            'argv': ['donkey', '-i'],
            'dialog': [
                ('Enter login name', 'roam'),
                ('Enter sequence', '99'),
                ('Enter new seed', 'st66708'),
                ('Enter passphrase', 'meowmeow'),
                ('Re-enter passphrase', 'meowmeow'),
            ],
            'result': 'FOLD BAWD AS WARM FOUL BARN',
        },

        {
            'argv': ['donkey', '24', 'st66708'],
            'dialog': [
                ('Enter passphrase', 'meowmeow'),
            ],
            'result': 'INCH CLAD LOCK VIE PAL WINK',
        },

        {
            'argv': ['donkey', '-n', '10', '77', 'st66708'],
            'dialog': [
                ('Enter passphrase', 'meowmeow'),
            ],
            'result': '73: HYMN GRAD THEY MANY MUTT BONG',
        },

        {
            'argv': ['donkey', '-i', '-f', 'md4'],
            'dialog': [
                ('Enter login name', 'roam'),
                ('Enter sequence', '99'),
                ('Enter new seed', 'st66708'),
                ('Enter passphrase', 'meowmeow'),
                ('Re-enter passphrase', 'meowmeow'),
            ],
            'result': 'FOLD BAWD AS WARM FOUL BARN',
        },

        {
            'argv': ['donkey', '-f', 'md4', '24', 'st66708'],
            'dialog': [
                ('Enter passphrase', 'meowmeow'),
            ],
            'result': 'INCH CLAD LOCK VIE PAL WINK',
        },

        {
            'argv': ['donkey', '-n', '10', '-f', 'md4', '77', 'st66708'],
            'dialog': [
                ('Enter passphrase', 'meowmeow'),
            ],
            'result': '73: HYMN GRAD THEY MANY MUTT BONG',
        },

        {
            'argv': ['donkey', '-i', '-f', 'md5'],
            'dialog': [
                ('Enter login name', 'roam'),
                ('Enter sequence', '99'),
                ('Enter new seed', 'st66708'),
                ('Enter passphrase', 'meowmeow'),
                ('Re-enter passphrase', 'meowmeow'),
            ],
            'result': 'JILT LURA LULU HOME WERE SWAM',
        },

        {
            'argv': ['donkey', '-f', 'md5', '24', 'st66708'],
            'dialog': [
                ('Enter passphrase', 'meowmeow'),
            ],
            'result': 'SEND OR LOOT BETH TONY WERT',
        },

        {
            'argv': ['donkey', '-n', '10', '-f', 'md5', '77', 'st66708'],
            'dialog': [
                ('Enter passphrase', 'meowmeow'),
            ],
            'result': '73: BUDD DOLE LIT BUNT BLOW ADD',
        },
    )
    def test_dialog(self, data):
        """
        Start the test program with the options given in the test data,
        carry out the dialog, then check the result.
        """
        (pid, fd) = pty.fork()
        self.assertIsNotNone(pid)
        if pid == 0:
            os.execlp(os.environ.get('TEST_PROG', './donkey'),
                      *data['argv'])
            os._exit(42)

        self.assertIsNotNone(fd)

        for (prompt, response) in data['dialog']:
            p = read_prompt(fd)
            self.assertIn(prompt, p)
            os.write(fd, six.b(response + '\n'))

        p = read_prompt(fd, False)
        self.assertIn(data['result'], p)

        (npid, status) = os.waitpid(pid, 0)
        self.assertEqual(pid, npid)
        self.assertEqual(status, 256 * data.get('exit_code', 0))


if __name__ == '__main__':
    unittest.main()
