/* doscan - Denial Of Service Capable Auditing of Networks
 * Copyright (C) 2003, 2004 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This code implements a scanner for one incarnation of the MS04-007
 * vulnerability:
 *
 *  <http://www.microsoft.com/technet/security/bulletin/ms04-007.asp>
 *
 * It is based on Renaud Deraison's Nessus plugin (script ID 12055,
 * revision 1.2).
 */

#include "config.h"
#include "half_duplex.h"
#include "opt.h"
#include "proto.h"
#include "results.h"
#include "rx.h"
#include "scan_tcp.h"
#include "scan_trigger.h"
#include "subnets.h"
#include "utils.h"

#include <arpa/inet.h>
#include <cerrno>
#include <cstdio>
#include <memory>
#include <netinet/in.h>
#include <string>
#include <sys/types.h>
#include <sys/socket.h>

static bool asn1_start(subnets&);
static void asn1_open(scan_host_t *);

void
proto_ms_04_007_http_register()
{
  proto_register ("ms_ms04-007_http", asn1_start, asn1_open);
}

static void
asn1_open (scan_host_t *)
{
}

namespace {

class http_client : public tcp_half_duplex_handler {
  virtual void ready(int state, int error, const std::string& data);

  void make_get_request(std::string&);

public:
  http_client(event_queue&, ipv4_t);
};

http_client::http_client(event_queue& q, ipv4_t host)
  : tcp_half_duplex_handler(q, host, opt_port)
{
  set_relative_timeout(opt_connect_timeout);
}

void
http_client::make_get_request(std::string& req)
{
  static const char* request_start = "GET / HTTP/1.1\r\n"
    "Authorization: Negotiate YIQAAABiBoMAAAYrBgEFBQKgggBTMIFQoA4"
    "wDAYKKwYBBAGCNwICCqM+MDygMDsuBIEBJSSBJwQBBSQiJCAkGCQWJBQkEiQ"
    "QJA4kDCQKJAgkBiQEJAIEAASCAAI5JaEIBAblxfnlIAQ=\r\n"
    "Host: ";

  ipv4_string_t host_str;
  ipv4_address_to_string(host(), host_str);

  req.assign(request_start);
  req.append(host_str);         // Add IP address to Host: header.
  req.append("\r\n\r\n");
}

void
http_client::ready(int state, int error, const std::string& data)
{
  std::string request;

  if (error) {
    return;
  }

  switch (state) {
  case 0:
    if (get_error()) {
      return;
    }
    make_get_request(request);
    send(20, request);
    return;

  case 20:
    if (data.find("\r\n\r\n") != std::string::npos) {
      // We have received the complete header.

      // FIXME: These checks are extremely crude.
      bool vulnerable = (data.find("\r\nWWW-Authenticate: Negotiate ")
                         != std::string::npos);
      bool patched = (data.find("\r\nWWW-Authenticate: Negotiate\r\n")
                      != std::string::npos);
      bool iis = (data.find("\r\nServer: Microsoft-IIS") != std::string::npos);

      if (vulnerable) {
        if (iis) {
          results_add_unquoted(host(), "vulnerable to MS04-007");
        } else {
          results_add_unquoted(host(), "vulnerable to MS04-007 (no IIS?)");
        }
      } else {
        if (patched) {
          if (opt_verbose) {
            if (iis) {
              results_add_unquoted(host(), "not vulnerable to MS04-007 (IIS)");
            } else {
              results_add_unquoted(host(), "not vulnerable to MS04-007 (IIS)");
            }
          }
        } else {
          if (iis) {
            results_add_unquoted(host(), "unclear MS04-007 status: "
                                 + quote(data));
          }
        }
      }
    } else {
      request_more_data(state);
    }
    return;
  }

  abort();
}

} // anonymous namespace

static bool
asn1_start(subnets& nets)
{
  if (opt_send && opt_send[0]) {
    fprintf (stderr, "%s: --send is not supported by this module.\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  if (opt_receive && opt_receive[0]) {
    fprintf (stderr, "%s: --receive is not supported by this module.\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  if (opt_banner_size) {
    fprintf (stderr, "%s: --banner is not supported by this module.\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  std::auto_ptr<event_queue> q(event_queue::create(opt_fd_count));
  scan_trigger::default_handler<http_client> th;
  scan_trigger t(*q, nets, th, opt_fd_count, opt_add_timeout, opt_add_burst);

  q->run();

  return false;
}
