/* doscan - Denial Of Service Capable Auditing of Networks
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This protocol module scans for hosts which are vulnerable to
 * CAN-2003-0352 (Microsoft Security Bulletin MS03-026, "Buffer Overrun
 * In RPC Interface Could Allow Code Execution").
 *
 * For more information on the vulnerability, see:
 *
 *   <http://www.microsoft.com/technet/security/bulletin/MS03-026.asp>
 *
 * This protocol module is based on a Nessus plugin by KK Liu:
 *
 *   <http://cgi.nessus.org/plugins/dump.php3?id=11808>
 *
 * It seems that this remote detection works because the response to
 * the IRemoteActivation request contains different OXIDs, probably
 * because Microsoft changed something else in the security patch.
 * (This mostly speculation, there does not seem to be a DCOM protocol
 * analyzer which is readily available.)
 *
 */

#include "config.h"
#include "engine_tcp.h"
#include "opt.h"
#include "proto.h"
#include "results.h"
#include "scan.h"

#include <cstdio>
#include <cstring>

static bool rpc_start (subnets&);
static void rpc_open (scan_host_t *);
static void rpc_send_bind (scan_host_t *);
static void rpc_send_bind_cmpl (scan_host_t *);
static void rpc_send_request (scan_host_t *, const char*, unsigned);
static void rpc_send_request_cmpl (scan_host_t *);
static void rpc_receive_version (scan_host_t *, const char *, unsigned);

static int show_all;

void
proto_ms_rpc_dcom_old_register (void)
{
  proto_register ("ms_rpc_dcom_old", rpc_start, rpc_open);
}

static bool
rpc_start (subnets&)
{
  if (opt_banner_size != 0) {
    fprintf (stderr, "%s: ms_rpc protocol module does not support --banner\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  if (opt_send && (opt_send[0] != '\0')) {
    fprintf (stderr, "%s: ms_rpc protocol module does not support --send\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  show_all = 0;
  if (opt_receive && (opt_receive[0] != '\0')) {
    if (strcmp (opt_receive, "all") == 0) {
      show_all = 1;
    } else {
      fprintf (stderr, "%s: --receive argument '%s' not recognized\n",
             opt_program, opt_receive);
      exit (EXIT_FAILURE);
    }
  }

  return true;
}

static void
rpc_open (scan_host_t *s)
{
  if (!s->state) {
    s->state = malloc (sizeof (engine_tcp_t));
  }
  engine_tcp_open (s, rpc_send_bind, 0);
}

static void
rpc_send_bind (scan_host_t *s)
{
  static const char bind_data[] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
    0x00, 0x53, 0x53, 0x56, 0x41, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xb8,
    0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11, 0x86, 0x1e, 0x00, 0x20,
    0xaf, 0x6e, 0x7c, 0x57, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
    0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
    0x48, 0x60, 0x02, 0x00, 0x00, 0x00
  };
  static const unsigned bind_data_len = sizeof (bind_data);

  engine_tcp_send (s, bind_data, bind_data_len, ENGINE_TCP_NO_COPY,
                   rpc_send_bind_cmpl);
}

static void
rpc_send_bind_cmpl (scan_host_t *s)
{
  engine_tcp_receive_packet (s, 128, rpc_send_request);
}

static void
rpc_send_request (scan_host_t *s, const char * buffer, unsigned size)
{
  static const char request_data[] = {
    0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x5b, 0x4e, 0x45, 0x53, 0x53, 0x55, 0x53, 0x5d,
    0x5b, 0x4e, 0x45, 0x53, 0x53, 0x55, 0x53, 0x5d, 0x00, 0x00, 0x00,
    0x00, 0x53, 0x53, 0x56, 0x41, 0x32, 0x30, 0x30, 0x33, 0x53, 0x53,
    0x56, 0x41, 0x32, 0x30, 0x30, 0x33, 0x68, 0x0f, 0x0b, 0x00, 0x1e,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00,
    0x5c, 0x00, 0x5c, 0x00, 0x53, 0x4f, 0x43, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x63, 0x00, 0x24, 0x00, 0x5c, 0x00, 0x53, 0x00, 0x53, 0x00,
    0x56, 0x00, 0x41, 0x00, 0x5f, 0x00, 0x32, 0x00, 0x30, 0x00, 0x30,
    0x00, 0x33, 0x00, 0x5f, 0x00, 0x4e, 0x00, 0x45, 0x00, 0x53, 0x00,
    0x53, 0x00, 0x45, 0x00, 0x53, 0x00, 0x2e, 0x00, 0x74, 0x00, 0x78,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xb8,
    0xeb, 0x0b, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00
  };
  static const unsigned request_data_len = sizeof (request_data);

  engine_tcp_send (s, request_data, request_data_len, ENGINE_TCP_NO_COPY,
                   rpc_send_request_cmpl);
}

static void
rpc_send_request_cmpl (scan_host_t *s)
{
  engine_tcp_receive_packet (s, 128, rpc_receive_version);
}


static void
rpc_receive_version (scan_host_t *s, const char *buffer, unsigned size)
{
  if (size < 26) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 buffer, size);

  } else {
    const char *p = buffer + size - 25;
    unsigned j, len;
    int status;
    char buf[100];

    typedef struct {
      unsigned char data[4];
      unsigned size;
      int vulnerable;
    } signature_t;

    /* We get some responses from other DCE RPC hosts which do not
       speak DCOM.  We use the value "-2" in the table below to mark
       them as "not interesting". */

    static const signature_t signatures[] = {
      { {0x00, 0x04, 0x00, 0x08}, 92,  0, },
      { {0x00, 0x05, 0x00, 0x07}, 92,  0, },
      { {0x00, 0x54, 0x01, 0x04}, 92,  1, }, /* also Windows Me *sigh* */
      { {0x00, 0x00, 0x20, 0x00}, 32, -2, }, /* unknown, but not Microsoft */
      { {0x00, 0x00, 0x00, 0x00}, 28, -2, }, /* unknown, but not Microsoft */
    };

    status = -1;
    for (j = 0; j < (sizeof (signatures) / sizeof (signature_t)); j++) {
      if ((signatures[j].size == size) && (memcmp (p, signatures[j].data, 4) == 0)) {
        status = signatures[j].vulnerable;
        break;
      }
    }

    switch (status) {
    case -2:
      if (!show_all) {
        break;
      }
      /* FALLTHROUGH */

    case -1:
      len = sprintf (buf, "unknown reply %02X%02X%02X%02X "
                     "(reply size %u bytes)", (unsigned)p[0], (unsigned)p[1],
                     (unsigned)p[2], (unsigned)p[3], size);

      results_add (ticks_get_cached (), s->host, 0, buf, len);
      break;

    case 0:
      if (show_all) {
        len = sprintf (buf, "Microsoft DCOM Service, not vulnerable");
        results_add (ticks_get_cached (), s->host, 0, buf, len);
      }
      break;

    case 1:
      len = sprintf (buf, "Microsoft DCOM Service, vulnerable to CAN-2003-0352");
      results_add (ticks_get_cached (), s->host, 0, buf, len);
      break;
    }
  }
}

/* arch-tag: 1e1a33bc-2fac-4e6f-8650-a3b676003c4d
 */
