#
#  dpkg-cross.pl - Package with dpkg-cross common used functions
#  Copyright (C) 2004  Nikita Youshchenko <yoush@cs.msu.su>
#  Copyright (C) 2004  Raphael Bossek <bossekr@debian.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#  $Id: dpkg-cross.pl,v 1.4 2004/06/24 17:48:58 ds Exp $

# Determine if the system wide or user defined cross-compile configuration
# have to be read.
$conffile = "/etc/dpkg/cross-compile";
$private_conffile = "$ENV{'HOME'}/.dpkg/cross-compile";
$conffile = (-e $private_conffile) ? $private_conffile : $conffile;

# Name of the calling application.
($progname = $0) =~ s,.*/,,;

# List of variables which can be overriden by the cross-compile
# definitions.
@intern_vars = qw( crossbase crossprefix crossdir crossbin crosslib crossinc
		   crossinfo maintainer default_arch removedeps keepdeps );
# Avoid warnings about unused @intern_vars variables.
foreach my $var_ ( @intern_vars ) {
	eval( "\$$var_ = ''" );
}

# Version of dpkg-cross environment.
$DPKGCROSSVERSION = "1.15.1";

# Convertion table for Debian GNU/Linux architecture name (``$arch'') to GNU
# type.
%archtable = (
	'i386' => 'i386-linux',
	'sparc' => 'sparc-linux',
	'sparc64' => 'sparc64-linux',
	'alpha' => 'alpha-linux',
	'm68k' => 'm68k-linux',
	'arm' => 'arm-linux',
	'powerpc' => 'powerpc-linux',
	'mips' => 'mips-linux',
	'mipsel' => 'mipsel-linux',
	'sh3' => 'sh3-linux',
	'sh4' => 'sh4-linux',
	'sh3eb' => 'sh3eb-linux',
	'sh4eb' => 'sh4eb-linux',
	'hppa' => 'hppa-linux',
	'hurd-i386' => 'i386-gnu',
	's390' => 's390-linux',
	's390x' => 's390x-linux',
	'ia64' => 'ia64-linux',
	'openbsd-i386' => 'i386-openbsd',
	'freebsd-i386' => 'i386-freebsd',
	'darwin-powerpc' => 'powerpc-darwin',
	'darwin-i386' => 'i386-darwin',
	'win32-i386' => 'i386-cygwin');

# All environment variables for GCC with their correspondig application name
# which will be modified for cross-compiling.
%std_compiler = (
	CC => "gcc",
	GCC => "gcc",
	CXX => "g++",
	CPP => "cpp",
	IMAKECPP => "cpp");

# All environment variables for binutils with their correspondig application
# name which will be modified for cross-compiling.
%std_binutils = (
	LD => "ld",
	AS => "as",
	AR => "ar",
	RANLIB => "ranlib",
	STRIP => "strip");

# Contains variable definitions from ``$conffile'' distinguished by their
# scope ("makeflags" or "environment") of definition.
# Scope is the first key of ``%pkgvars''. Variable definitions are stored
# as a second hash table. A seperate hash table for each scope. Those hash
# tables contains for their own key,value pairs corresponds to variable
# name and variable value definitions in the specific scope.
%pkgvars = (
	"makeflags" => (),
	"environment" => ());

# Contains ``$crossroot'' definitions by ``$arch'' readed from configuration.
# ``$crossroot'' is set by setup() if ``$arch'' is known from this hash table.
%allcrossroots = ();

################################################################################
# read_config():
#     Read ``$conffile'' and save the definition in global variables for those
#     which are listed in ``@intern_vars'' until first package, mode or scope
#     definition is found.
#
#     ``$crossroot'' will be set by setup(). Until setup() is called all
#     "crossroot-<arch>" settings are stored within ``%allcrossroots''.
#
#     Package's variables are stored within ``%pkgvars''. Only variables from
#     packages "all" and "$1" are set. Those are distinguished between
#     "makeflags" and "environment" scope.
#
#     All other variable are skipped.
# $1: Package name to read configuration for (including special package "all").
#     If empty the special package name "default" is assumed.
# $2: Mode as specified with `dpkg-buildpackage -M<mode>` to read configuration
#     for (including the special mode "all").
# return: none
sub read_config {
	my ($current_package_, $current_mode_) = @_;	# ok to be undefined
	my ($package_, $mode_, $scope_, $var_, $val_) = ("", "", "", "", "");
	
	$current_package_ ||= "default";
	
	open (F, "<$conffile") || return;
	while (<F>) {
		chomp;
		s/^\s+//; s/(\s*#.*)?\s*$//;
		next if /^$/;
		
		if (/([\w\d_-]+)\s*=\s*(.*)$/) {
			($var_, $val_) = ($1, $2);
			# dpkg-cross come before any mode, package or scope declarations.
			if (!$package_ && !$mode_ && !$scope_) {
				if ($var_ =~ /^crossroot-(\S+)$/) {
					# We set ``$crossroot'' after we know which ``$arch'' to use.
					# Until then we save the architecture and value for use by setup().
					$allcrossroots{$1} = $val_;
				}
				# Set only variables which are listed in ``@intern_vars''.
				elsif (grep $_ eq $var_, @intern_vars) {
					eval "\$$var_ = '$val_'";
				}
				else {
					warn "$progname: Definition of unknown variable ".
						 "$var_ in $conffile, line $..\n";
				}
				next;
			}

			$package_ ||= "all";
			$mode_ ||= "all";
			$scope_ ||= "makeflags";

			# Store only data for current package and current mode, and
			# for package "all" and mode "all".
			if (($package_ eq "all" || $package_ eq $current_package_) && \
				($mode_ eq "all" || $mode_ eq $current_mode_)) {
				$pkgvars{$scope_}{$var_} = $val_;
			}
		}
		elsif (/^(package)?\s*?(\S+)\s*:$/i) {
			$package_ = $2;
		}
		elsif (/^mode\s*(\S+)\s*:$/i) {
			$mode_ = $1;
		}
		elsif (/^scope\s*(makeflags|environment)\s*:$/) {
			$scope_ = $1;
		}
		else {
			warn "$progname: Unrecognized line in $conffile, line $..\n";
		}
	}
	close (F);
}

################################################################################
# setup():
#     Set global variables ``$arch'', ``$arch_cpu'', ``$arch_os'',
#     ``$crossbase'', ``$crossbin'', ``$crossdir'', ``$crossinc'',
#     ``$crossinfo'', ``$crosslib'', ``$crossprefix'', ``$deb_build_arch'' and
#     ``$deb_build_gnu_type'' to defaults and substitude them with variables from
#     ``@intern_vars'' and ``$arch''.
#
#     All variable definitions from ``%pkgvars'' are also substituded with
#     ``@intern_vars'' and ``$arch''.
# return: none
sub setup {
	my ($var_, $os_, $scope_);
	my @vars_ = ("arch", @intern_vars);
	
	# Set ``$arch'' to defaults if not already specified.
	$arch ||= $ENV{'ARCH'} || $default_arch;
	
	# Determine OS and CPU from ``$arch''.
	$deb_build_arch = $arch || die "$progname: Architecture is not specified.\n";
	$deb_build_gnu_type = $archtable{$arch};
	
	$arch_cpu = $deb_build_gnu_type;
	$arch_cpu =~ s/-.*$//;
	$arch_os = $deb_build_gnu_type;
	$arch_os =~ s/^.*-//;
	
	# Finalize, no subst possible crossbase.
	$crossbase ||= "/usr/local";
	
	# Set defaults for internal vars, if not set ...
	$crossprefix   ||= $ENV{'CROSSPREFIX'} || "${arch_cpu}-${arch_os}-";
	$crossdir      ||= "\$(CROSSBASE)/${arch_cpu}-${arch_os}";
	$crossbin      ||= "\$(CROSSDIR)/bin";
	
	if (exists $allcrossroots{$arch}) {
		$crosslib  ||= "\$(CROSSROOT)/lib";
		$crossinc  ||= "\$(CROSSROOT)/usr/include";
		$crossroot = $allcrossroots{$arch};
		push (@vars_, "crossroot");
	}
	else {
		$crosslib  ||= "\$(CROSSDIR)/lib";
		$crossinc  ||= "\$(CROSSDIR)/include";
		$crossinfo ||= "\$(CROSSLIB)/dpkg-cross-info";
	}
	# ... and substitute references in them.
	foreach $var_ (@intern_vars) {
		next if $var_ eq "crossbase" || $var_ eq "maintainer";
		subst (eval "\\\$$var_", $var_, @vars_);
	}
	
	# Substitute variable references in package variable definitions.
	foreach $scope_ (keys %pkgvars) {
		foreach $var_ (keys %{$pkgvars{$scope_}}) {
			subst (\{$pkgvars{$scope_}{$var_}}, $var_, @vars_);
		}
	}
}

################################################################################
# subst():
#     Substitute strings in form $(NAME) by same named global Perl variable
#     (only if variable's name is listed in $3) or by the correspondig
#     envrionment variable. A variable name to substitude which is not defined
#     will by replaced by an empty string.
# $1: Reference to the string should be substituted in place.
# $2: Name of the Perl variable for which the content should be substituded. Is
#     used for warning notifications in case of $(NAME) can not be substitued.
# $3: List of variable names which are allowed to be substituded.
# return: none
sub subst {
	my $valref_ = shift (@_);
	my $varname_ = shift (@_);
	my @defined_vars_ = @_;
	my ($name_, $newval_);

	while ($$valref_ =~ /\$\((\w+)\)/) {
		$name_ = $1;
		if (grep "\U$_\E" eq $name_, @defined_vars_) {
			$newval_ = eval "\"\$\L$name_\E\"";
		}
		elsif (exists $ENV{$name_}) {
			$newval_ = $ENV{$name_};
		}
		else {
			warn "$progname: Cannot substitute \$($name_) in definition ".
				 "of $varname_.\n";
			$newval_ = "";
		}
		$$valref_ =~ s/\$\($name_\)/$newval_/;
	}
}

################################################################################
# find_cross_app():
#     Looking for application which is appropriete for ``$crossprefix'' by
#     searching in PATH or within ``$crossbin''. If not, $1 is returned in
#     hope it support multiple architectures.
# $1: Application name to search for.
# return: Name of the cross application to use.
sub find_cross_app {
	my $app_ = shift (@_);
	my ($dir_, $p_);
	
	if ($crossprefix) {
		# First try with ``$crossprefix'' in PATH.
		foreach $dir_ (split (':', $ENV{'PATH'})) {
			$p_ = "$dir_/${crossprefix}$app_";
			if (-x $p_) {
				return $p_;
			}
		}
	}
	
	if ($crossbin) {
		# Next guess it is in ``$crossbin''.
		$p_ = "$crossbin/$app_";
		if (-x $p) {
			return $p;
		}
	}
	
	return $app_;
}

################################################################################
# find_objdump():
#     Looking for `objdump' which is appropriete for ``$arch''.
# return: Name of the "objdump" application to use.
sub find_objdump {
	return find_cross_app ("objdump");
}

################################################################################
# find_strip():
#    Looking for `strip' which is appropriete for ``$arch''.
# return: Name of the "strip" application to use.
sub find_strip {
	return find_cross_app ("strip");
}

################################################################################
# setup_cross_env():
#     Set the environment variables MAKEFLAGS, PATH, PKG_CONFIG_LIBDIR and all
#     variables from ``%pkgvars'' which are makred for scope "environment".
# return: none
sub setup_cross_env {
	my ($var_, $tmp_, $pkghashref, %makeflags_, $setmakeflags_, $native_cpu_, $pkg_config_path_);
	
	# Read and process config file.
	read_config ($package, $mode);
	setup();
	
	# Put ``$arch'' into environment through MAKEFLAGS.
	$makeflags_{'ARCH'} = $arch;
	
	# Also set new `dpkg-architecture' environment veriables.
	chomp ($tmp_ = `dpkg-architecture -a$arch -qDEB_HOST_ARCH 2>/dev/null`);
	$makeflags_{'DEB_HOST_ARCH'} = $tmp_;
	
	chomp ($tmp_ = `dpkg-architecture -a$arch -qDEB_HOST_GNU_CPU 2>/dev/null`);
	$makeflags_{'DEB_HOST_GNU_CPU'} = $tmp_;
	
	chomp ($tmp_ = `dpkg-architecture -a$arch -qDEB_HOST_GNU_SYSTEM 2>/dev/null`);
	$makeflags_{'DEB_HOST_GNU_SYSTEM'} = $tmp_;
	
	chomp ($tmp_ = `dpkg-architecture -a$arch -qDEB_HOST_GNU_TYPE 2>/dev/null`);
	$makeflags_{'DEB_HOST_GNU_TYPE'} = $tmp_;
	
	# Append /usr/local/bin to PATH, some binaries may be in there;
	# also append $crossbin, so that cross binaries can be found, but
	# native stuff still has precedence (if a package wants to compile
	# with 'gcc' a build tool that will be executed, for example).
	$ENV{'PATH'} = "$ENV{PATH}:/usr/local/bin:$crossbin";
	
	# Set USRLIBDIR to $(CROSSLIB), for imake-generated Makefiles..
	$makeflags_{'USRLIBDIR'} = $crosslib;
	
	# Set CONFIG_SITE to /etc/dpkg/cross-config.``$arch'', for
	# packages using GNU autoconf configure scripts.
	$makeflags_{'CONFIG_SITE'} = "/etc/dpkg/cross-config.$arch";
	
	# Set standard variables for compilers.
	foreach $var_ ( keys %std_compiler ) {
		$makeflags_{$var_} = $crossprefix . $std_compiler{$var_};
	}

	# Allow to use $crossprefix-gcc -E as preprocessor if $crossprefix-cpp
	# is not available
	if (system("which " . $makeflags_{"CPP"} . " >/dev/null") != 0 &&
	    system("which " . $makeflags_{"GCC"} . " >/dev/null") == 0) {
		$makeflags_{"CPP"} = $makeflags_{"IMAKECPP"} = $makeflags_{"GCC"} . "\\ -E";
		print "OK\n";
	} else {
		print "FAIL\n";
		exit 1;
	}
	
	# Set standard variables for binutils (if different CPU).
	chomp ($native_cpu_ = `dpkg --print-installation-architecture`);
	if ($arch ne $native_cpu_) {
		foreach $var_ ( keys %std_binutils ) {
			$makeflags_{$var_} = $crossprefix . $std_binutils{$var_};
		}
	}
	
	# Set additional variables specified in "scope makeflags:" for the current
	# package.
	foreach $var_ ( keys %{ $pkgvars{'makeflags'} } ) {
		$makeflags_{$var_} = $pkgvars{'makeflags'}{$var_};
	}
	
	if (exists $ENV{'MAKEFLAGS'}) {
		$setmakeflags_ = $ENV{'MAKEFLAGS'};
		$setmakeflags_ .= " -- " if $setmakeflags_ !~ / -- /;
	} else {
		$setmakeflags_ = "w -- ";
	}
	foreach (keys %makeflags_) {
		$setmakeflags_ .= " $_=$makeflags_{$_}";
	}
	$ENV{'MAKEFLAGS'} = $setmakeflags_;
	
	# Add PKG_CONFIG_LIBDIR to enironment, to make `pkg-config' use
	# our directory instead of /usr/lib/pkgconfig.
	$ENV{'PKG_CONFIG_LIBDIR'} = $crosslib . "/pkgconfig";
	
	# Set additional environment variabled specified in "mode environment:".
	foreach $var_ (keys %{ $pkgvars{'environment'} }) {
		$ENV{$var_} = $pkgvars{'environment'}{$var_};
	}
}

################################################################################
# dpkgcross_application():
#     If not called (indirectly) from 'dpkg-buildpackage -a<arch>', then exec
#     the original.
#     This function also initialise ``$arch'' as set by `dpkg-buildpackage -a'.
# return: none
sub dpkgcross_application {
	exec "$0.orig", @ARGV
		unless (($arch = $ENV{'ARCH'}) && $ENV{'MAKEFLAGS'} =~ /\bCC=/) ||
		$arch =~ /^hurd-/;
}

1;
