/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <gnome.h>
#include <string.h>
#include <glib.h>
#include <curl/curl.h>

#include "drivel.h"
#include "login.h"
#include "journal.h"
#include "requests.h"
#include "network.h"

#define BUFFER_SIZE 1

extern GMutex *network_mutex;
gchar *post_data = NULL;
gint dialog_count = 0;
DrivelRequestType global_type;
gboolean network_cancel = FALSE; 

gchar *
lj_format_string (const gchar *string)
{
	if (!string)
		return (g_strdup (""));
	else
		return (curl_escape (string, 0));
}

gboolean
lj_http_reslv_name (DrivelClient *dc)
{
	gchar *server = NULL, *proxy = NULL;
	
	if (!dc->server)
		dc->server = g_strdup ("www.livejournal.com");
	
	g_free (dc->server_proper_format);
	g_free (dc->proxy_proper_format);
	
	if (strstr (dc->server, "http://"))
	{
		server = g_new0 (gchar, strlen (dc->server));
		memcpy (server, dc->server + 7, strlen (dc->server) - 7);
	}
	else
		server = g_strdup (dc->server);
	
	if (server [strlen (server) - 1] == '/')
		server [strlen (server) - 1] = '\0';

	if (dc->proxy_url)
	{
		if (strstr (dc->proxy_url, "http://"))
		{
			proxy = g_new0 (gchar, strlen (dc->proxy_url));
			memcpy (proxy, dc->proxy_url + 7, strlen (dc->proxy_url) - 7);
		}
		else
			proxy = g_strdup (dc->proxy_url);
	
		if (proxy [strlen (proxy) - 1] == '/')
			proxy [strlen (proxy) - 1] = '\0';
	}
	
	dc->server_proper_format = server;
	dc->proxy_proper_format = proxy;

	return TRUE;
}

void
cancel_transaction (GtkWidget *dialog, gint arg, gpointer data)
{	
	network_cancel = TRUE;
	
	return;
}

void
dialog_open (DrivelClient *dc, DrivelRequestType type)
{
	GtkWidget *label;
	GtkWidget *vbox;
	
	if (type == REQUEST_CHECKFRIENDS)
		return;
	
	dialog_count++;
	
	journal_enable (dc, FALSE);
	
	if (!dc->message_dialog)
	{
		dc->message_dialog = gtk_dialog_new_with_buttons (
				_("Network Progress"),
				GTK_WINDOW (dc->current_window),
				GTK_DIALOG_MODAL,
				GTK_STOCK_CANCEL,
				GTK_RESPONSE_REJECT,
				NULL);
		gtk_dialog_set_has_separator (GTK_DIALOG (dc->message_dialog), TRUE);
		
		vbox = gtk_vbox_new (FALSE, 12);
		gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dc->message_dialog)->vbox),
				vbox, TRUE, TRUE, 0);
		
		label = gtk_label_new (_("Sending / Receiving..."));
		gtk_box_pack_start (GTK_BOX (vbox), label, 
				TRUE, FALSE, 0);
		
		dc->network_progress = gtk_progress_bar_new ();
		gtk_progress_bar_set_pulse_step (GTK_PROGRESS_BAR (dc->network_progress),
				0.2);
		gtk_box_pack_start (GTK_BOX (vbox), dc->network_progress, 
				TRUE, FALSE, 0);
		
		g_signal_connect (G_OBJECT (dc->message_dialog), "response",
				G_CALLBACK (cancel_transaction), dc);
					
		gtk_widget_show_all (dc->message_dialog);
	}
	
	return;
}

void
dialog_close (DrivelClient *dc)
{
	if (global_type == REQUEST_CHECKFRIENDS)
		return;
	
	dialog_count--;
	
	if (!dialog_count && dc->message_dialog)
	{
		gtk_widget_destroy (dc->message_dialog);
		dc->message_dialog = NULL;
		dc->network_progress = NULL;
		
		journal_enable (dc, TRUE);
	}
	
	return;
}

gint
dialog_progress (gpointer *data, double t, double d, double ultotal, double ulnow)
{
	ThreadData *td = (ThreadData *) data;
	gchar *queue_data;
	gboolean retval;
	
	queue_data = g_strdup ("pulse");
	g_async_queue_push (td->dc->progress_queue, queue_data);
	
	if (network_cancel)
	{
		retval = TRUE;
		network_cancel = FALSE;
	}
	else
		retval = FALSE;
	
	return retval;
}

gchar *
get_line (const gchar *data, gint *offset)
{
	gchar *line;
	guint i;

	for (i = 0; i < strlen (data); i++)
	{
		if (data [i] == '\r' && data [i + 1] == '\n')
		{
			*offset = 2;
			break;
		}
		else if (data [i] == '\r' || data [i] == '\n')
		{
			*offset = 1;
			break;
		}
	}
	
	line = g_new0 (gchar, i + 1);
	memcpy (line, data, i);
	line [i] = '\0'; // trash the newline

	return line;
}

void
parse_read_data (DrivelClient *dc, const gchar *string, gint len, DrivelRequestType request)
{
	guint offset, line_offset;
	gchar *key, *value;

	offset = 0;
    
	if (dc->vars != NULL)
		hash_table_clear (dc->vars);
	else
		dc->vars = g_hash_table_new_full (g_str_hash, g_str_equal,
				hash_table_item_free, hash_table_item_free);

	while (offset < len)
	{
		key = get_line (string + offset, &line_offset);
		offset += strlen (key) + line_offset;
		if (line_offset > 1)
		{
			// this is the http header, discard it
			g_free (key);
			continue;
		}		

		value = get_line (string + offset, &line_offset);
		offset += strlen (value) + line_offset;

		g_hash_table_insert (dc->vars, key, value);
	}
	
	request_process (dc, request);
	
	return;
}

size_t
write_cb (void *ptr, size_t size, size_t nmemb, void *data)
{
	gint realsize = size * nmemb;
	HTTPMem *memory = data;

	memory->data = (gchar *) g_realloc(memory->data, memory->len + realsize + 1);

	memcpy (&(memory->data [memory->len]), ptr, realsize);
	memory->len += realsize;
	memory->data [memory->len] = '\0';

	return realsize;
}

gpointer
send_request (gpointer data)
{
	CURL *session;
	HTTPMem *memory;
	gchar *url, *error, *queue_data, *cookie, *user, *pass, *userpwd;
	glong http_code;
	ThreadData *td;
	
	g_mutex_lock (network_mutex);

	td = (ThreadData *) data;
	
	global_type = td->type;
	
	memory = g_new (HTTPMem, 1);
	memory->len = 0;
	memory->data = NULL;
	
	url = g_strdup_printf ("http://%s/interface/flat", td->dc->server_proper_format);
	user = pass = userpwd = error = NULL;
	
	session = curl_easy_init ();
	
	cookie = g_strdup_printf ("ljfastserver=%d", td->dc->use_fast_servers);
	
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_POSTFIELDS, td->post_data);
	curl_easy_setopt (session, CURLOPT_ERRORBUFFER, error);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, dialog_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, td);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookie);
	
	// Proxy stuff
	if (gconf_client_get_bool (td->dc->client, td->dc->gconf->proxy, NULL))
	{
		user = gconf_client_get_string (td->dc->client, td->dc->gconf->proxy_user, NULL);
		pass = gconf_client_get_string (td->dc->client, td->dc->gconf->proxy_pass, NULL);
		if (user && user [0] != '\0' && pass && pass [0] != '\0')
		{
			userpwd = g_strdup_printf ("%s:%s", user, pass);	
			curl_easy_setopt (session, CURLOPT_PROXYUSERPWD, userpwd);
		}
		curl_easy_setopt (session, CURLOPT_PROXY,
				gconf_client_get_string (td->dc->client, td->dc->gconf->proxy_url, NULL));
		curl_easy_setopt (session, CURLOPT_PROXYPORT,
				gconf_client_get_int (td->dc->client, td->dc->gconf->proxy_port, NULL));		
	}
	
	curl_easy_perform (session);
	
	http_code = 0;
	if ((curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code) == CURLE_OK))
	{
//		g_print ("http_code: %d (%s) (%s)\n", (gint)http_code, error, url);
	}
	else
	{
//		g_print ("http_error\n");
	}

	if (http_code != 200 || !memory->data)
	{
		queue_data = g_strdup ("error");
		g_async_queue_push (td->dc->data_queue, queue_data);
	}
	else
	{
		queue_data = g_strdup (memory->data);
		g_async_queue_push (td->dc->data_queue, queue_data);
	}
	
	curl_easy_cleanup (session);
	
	g_free (cookie);
	g_free (url);
	g_free (user);
	g_free (pass);
	g_free (userpwd);
	g_free (memory->data);
	g_free (memory);
	g_free (td->post_data);
	g_free (td);
	
	return NULL;
}

void
lj_send_request (DrivelClient *dc, const gchar *data, DrivelRequestType type)
{
	ThreadData *td;
	
	g_return_if_fail (dc != NULL);
	
	td = g_new0 (ThreadData, 1);
	td->dc = dc;
	td->post_data = g_strdup (data);
	td->type = type;
	
	dialog_open (td->dc, td->type);
	
	g_thread_create (send_request, td, FALSE, NULL);
	
	return;
}

gboolean
lj_network_loop (gpointer data)
{
	gchar *queue_data;
	DrivelClient *dc = (DrivelClient *) data;
	
	if ((queue_data = g_async_queue_try_pop (dc->data_queue)))
	{
		if (!strcmp (queue_data, "error"))
		{
			if (global_type == REQUEST_CHECKFRIENDS)
				dc->time_since_checkfriends = 300;
			else
				display_error_dialog (dc, NULL);
		}
		else	
			parse_read_data (dc, queue_data, strlen (queue_data), global_type);
		
		dialog_close (dc);
		
		g_mutex_unlock (network_mutex);
		
		g_free (queue_data);
	}
	else if ((queue_data = g_async_queue_try_pop (dc->progress_queue)))
	{
		if (dc->network_progress)
			gtk_progress_bar_pulse (GTK_PROGRESS_BAR (dc->network_progress));
		
		g_free (queue_data);
	}
	
	return TRUE;
}
