/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <string.h>
#include <curl/curl.h>

#include <config.h>

#include "tray.h"
#include "drivel.h"
#include "journal.h"
#include "login.h"
#include "dialogs.h"
#include "network.h"
#include "requests.h"

gchar *
convert_text (const gchar *lj_text)
{
	gchar *text, hex[5];
	gint i, j, len, character;
	
	if (!lj_text)
		return NULL;
	
	len = strlen (lj_text);
	text = g_new0 (gchar, len + 1);
	hex[0] = '0';
	hex[1] = 'X';
	hex[4] = '\0';
	
	for (i = j = 0; i < len; i++)
	{
		if (lj_text [i] == '%')
		{
			memcpy (hex + 2, lj_text + i + 1, 2);
			character = (gint) g_ascii_strtod (hex, NULL);

			text [j] = (gchar) character;
			i += 2;
		}
		else if (lj_text [i] == '+')
			text [j] = ' ';
		else
			text [j] = lj_text [i];
		
		j++;
	}
	
	text [j] = '\0';
	
	return text;
}

void
request_process_editfriends (DrivelClient *dc)
{
	gchar *mesg, *post_data, *username;
	
	mesg = g_hash_table_lookup (dc->vars, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = g_hash_table_lookup (dc->vars, "errmsg");
		display_error_dialog (dc, mesg);
		
//		journal_enable (dc, TRUE);
		
		return;
	}
	
	username = lj_format_string (dc->username);
	
	post_data = g_strdup_printf (
			"mode=getfriends&ver=1&user=%s&"
			"hpassword=%s&includefriendof=1",
			username, dc->password);
	
	lj_send_request (dc, post_data, REQUEST_GETFRIENDS);
	
	curl_free (username);
	
	return;
}

void
request_process_getevents (DrivelClient *dc)
{
	gchar *mesg, **new_event, key [32];
	gchar **itemid, **old_event, **security, **allowmask, **subject;
	gchar **eventtime, **comments = NULL, **autoformat = NULL;
	gchar **mood = NULL, **music = NULL, **picture = NULL;
	gint prop_count, events_count, i, j;
	
	mesg = g_hash_table_lookup (dc->vars, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = g_hash_table_lookup (dc->vars, "errmsg");
		display_error_dialog (dc, mesg);
		
//		journal_enable (dc, TRUE);
		
		return;
	}

	mesg = g_hash_table_lookup (dc->vars, "events_count");
	if (mesg)
		events_count = (gint) g_ascii_strtod (mesg, NULL);
	else
		events_count = 0;

	itemid = g_new (gchar *, events_count + 1);
	old_event = g_new (gchar *, events_count + 1);
	new_event = g_new (gchar *, events_count + 1);
	security = g_new (gchar *, events_count + 1);
	allowmask = g_new (gchar *, events_count + 1);
	subject = g_new (gchar *, events_count + 1);
	eventtime = g_new (gchar *, events_count + 1);
	picture = g_new (gchar *, events_count + 1);
	mood = g_new (gchar *, events_count + 1);
	music = g_new (gchar *, events_count + 1);
	comments = g_new (gchar *, events_count + 1);
	autoformat = g_new (gchar *, events_count + 1);
	
	for (i = 0; i < events_count; i++)
	{
		picture [0] = NULL;
		mood [0] = NULL;
		music [0] = NULL;
		comments [0] = NULL;
		autoformat [0] = NULL;
	}
	
	for (i = 1; i < events_count + 1; i++)
	{
		g_snprintf (key, 32, "events_%d_itemid", i);
		itemid [i - 1] = g_hash_table_lookup (dc->vars, key);
		
		g_snprintf (key, 32, "events_%d_event", i);
		old_event [i - 1] = g_hash_table_lookup (dc->vars, key);
		new_event [i - 1] = convert_text (old_event [i - 1]);
		
		g_snprintf (key, 32, "events_%d_security", i);
		security [i - 1] = g_hash_table_lookup (dc->vars, key);
		
		g_snprintf (key, 32, "events_%d_allowmask", i);
		allowmask [i - 1] = g_hash_table_lookup (dc->vars, key);
		
		g_snprintf (key, 32, "events_%d_subject", i);
		subject [i - 1] = g_hash_table_lookup (dc->vars, key);
		
		g_snprintf (key, 32, "events_%d_eventtime", i);
		eventtime [i - 1] = g_hash_table_lookup (dc->vars, key);
		
		mesg = g_hash_table_lookup (dc->vars, "prop_count");
		if (mesg)
			prop_count = (gint) g_ascii_strtod (mesg, NULL);
		else
			prop_count = 0;
		
		for (j = 1; j < prop_count + 1; j++)
		{
			g_snprintf (key, 32, "prop_%d_name", j);
			mesg = g_hash_table_lookup (dc->vars, key);

			g_snprintf (key, 32, "prop_%d_value", j);

			if (!strcmp (mesg, "picture_keyword"))
				picture [i - 1] = g_hash_table_lookup (dc->vars, key);
			else if (!strcmp (mesg, "current_mood"))
				mood [i - 1] = g_hash_table_lookup (dc->vars, key);
			else if (!strcmp (mesg, "current_music"))
				music [i - 1] = g_hash_table_lookup (dc->vars, key);
			else if (!strcmp (mesg, "opt_nocomments"))
				comments [i - 1] = g_hash_table_lookup (dc->vars, key);
			else if (!strcmp (mesg, "opt_preformatted"))
				autoformat [i - 1] = g_hash_table_lookup (dc->vars, key);
		}
	}
	
//	journal_enable (dc, TRUE);
	
	if (dc->edit_entry)
	{	
		journal_edit_entry (dc, itemid [0], new_event [0], security [0], allowmask [0], subject [0],
				mood [0], music [0], picture [0], eventtime [0], comments [0], autoformat [0]);
	}
	else
		update_history_list (dc, itemid, new_event, eventtime, events_count);
	
	g_free (new_event);
	g_free (old_event);
	g_free (security);
	g_free (allowmask);
	g_free (subject);
	g_free (eventtime);
	g_free (picture);
	g_free (mood);
	g_free (music);
	g_free (comments);
	g_free (autoformat);
	
	return;
}

void
request_process_getfriends (DrivelClient *dc)
{
	gchar *mesg, key [32];
	gint friends, friends_of, i;
	LJFriend *friend;
	GSList *list_item;
	
	mesg = g_hash_table_lookup (dc->vars, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = g_hash_table_lookup (dc->vars, "errmsg");
		display_error_dialog (dc, mesg);
		
//		journal_enable (dc, TRUE);
		
		return;
	}
	
	if (dc->friends_list)
	{
		g_slist_foreach (dc->friends_list, friends_list_free_item, NULL);
		g_slist_free (dc->friends_list);
		dc->friends_list = NULL;
	}
	
	mesg = g_hash_table_lookup (dc->vars, "friend_count");
	friends = (gint) g_ascii_strtod (mesg, NULL);
	for (i = 1; i < friends + 1; i++)
	{
		friend = g_new0 (LJFriend, 1);
		
		g_snprintf (key, 32, "friend_%d_user", i);
		friend->username = g_strdup (g_hash_table_lookup (dc->vars, key));
		g_snprintf (key, 32, "friend_%d_name", i);
		friend->name = g_strdup (g_hash_table_lookup (dc->vars, key));
		g_snprintf (key, 32, "friend_%d_bg", i);
		friend->bg = g_strdup (g_hash_table_lookup (dc->vars, key));
		g_snprintf (key, 32, "friend_%d_fg", i);
		friend->fg = g_strdup (g_hash_table_lookup (dc->vars, key));
		g_snprintf (key, 32, "friend_%d_groupmask", i);
		mesg = g_hash_table_lookup (dc->vars, key);
		if (mesg)
			friend->groupmask = (gint) g_ascii_strtod (mesg, NULL);
		else
			friend->groupmask = 1;
		g_snprintf (key, 32, "friend_%d_type", i);
		mesg = g_strdup (g_hash_table_lookup (dc->vars, key));
		if (mesg)
			friend->type = TRUE;
		else
			friend->type = FALSE;
		friend->friend = TRUE;
		friend->friend_of = FALSE;
		
		dc->friends_list = g_slist_prepend (dc->friends_list, friend);
	}
	
	mesg = g_hash_table_lookup (dc->vars, "friendof_count");
	friends_of = (gint) g_ascii_strtod (mesg, NULL);
	for (i = 1; i < friends_of + 1; i++)
	{
		g_snprintf (key, 32, "friendof_%d_user", i);
		mesg = g_hash_table_lookup (dc->vars, key);
		list_item = g_slist_find_custom (dc->friends_list, mesg, compare_usernames);
		if (list_item)
		{
			friend = list_item->data;
			friend->friend_of = TRUE;
		}
		else
		{
			friend = g_new0 (LJFriend, 1);
		
			g_snprintf (key, 32, "friendof_%d_user", i);
			friend->username = g_strdup (g_hash_table_lookup (dc->vars, key));
			g_snprintf (key, 32, "friendof_%d_name", i);
			friend->name = g_strdup (g_hash_table_lookup (dc->vars, key));
			g_snprintf (key, 32, "friendof_%d_bg", i);
			friend->bg = g_strdup (g_hash_table_lookup (dc->vars, key));
			g_snprintf (key, 32, "friendof_%d_fg", i);
			friend->fg = g_strdup (g_hash_table_lookup (dc->vars, key));
			friend->groupmask = 1;
			g_snprintf (key, 32, "friendof_%d_type", i);
			mesg = g_strdup (g_hash_table_lookup (dc->vars, key));
			if (mesg)
				friend->type = TRUE;
			else
				friend->type = FALSE;
			friend->friend = FALSE;
			friend->friend_of = TRUE;
		
			dc->friends_list = g_slist_prepend (dc->friends_list, friend);
		}
	}
	
	display_edit_friends_dialog (dc);
	
//	journal_enable (dc, TRUE);
	
	return;
}

void
request_process_getdaycounts (DrivelClient *dc)
{
	gchar date [12], *mesg;
	gint8 days [31];
	guint month, day, year, i, count;
	
	mesg = g_hash_table_lookup (dc->vars, "success");
	if (mesg == NULL || !strcmp (mesg, "FAIL"))
	{
		mesg = g_hash_table_lookup (dc->vars, "errmsg");
		display_error_dialog (dc, mesg);
		
//		journal_enable (dc, TRUE);
		
		return;
	}
	
	gtk_calendar_get_date (GTK_CALENDAR (dc->history_calendar),
			&year, &month, &day);
	
	for (i = 0; i < 31; i++)
	{
		g_snprintf (date, 11, "%04d-%02d-%02d", year, month + 1, i + 1);

		mesg = g_hash_table_lookup (dc->vars, date);
		if (mesg)
			count = g_ascii_strtod (mesg, NULL);
		else
			count = 0;
		
		days [i] = count;
	}
	
	update_history_marks (dc, days);
	
//	journal_enable (dc, TRUE);
	
	return;
}

void
request_process_login (DrivelClient *dc)
{
	gchar *login, *password, *mesg, *key, *value;
	gchar mood_id [32], mood_name [32], static_key [64];
	gboolean save_password, autologin;
	GSList *users;
	gint pictures, new_moods, i, j, offset, menus;
	GtkWidget *dialog;
	LJMenuItem *menu_item;
		
	login = gtk_editable_get_chars (GTK_EDITABLE (GTK_COMBO (dc->login_name)->entry), 0, -1);
	password = gtk_editable_get_chars (GTK_EDITABLE (dc->login_password), 0, -1);
	save_password = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->save_password));
	autologin = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->autologin));
	
	if (!save_password)
	{
		g_free (password);
		password = g_strdup ("");
	}
	
	users = gconf_client_get_list (dc->client, "/apps/drivel/global_settings/user_list", GCONF_VALUE_STRING, NULL);
	
	if (users != NULL && g_slist_find_custom (users, login, string_compare))
	{
	}
	else
	{
		users = g_slist_prepend (users, login);
		gconf_client_set_list (dc->client, "/apps/drivel/global_settings/user_list", GCONF_VALUE_STRING, users, NULL);
	}
	
	drivel_gconf_data_fill (dc->gconf, login, dc->client, &dc->id, dc);
	
	gconf_client_set_string (dc->client, "/apps/drivel/global_settings/current_user", login, NULL);
	gconf_client_set_string (dc->client, dc->gconf->password, password, NULL);
	gconf_client_set_bool (dc->client, dc->gconf->save_password, save_password, NULL);
	gconf_client_set_bool (dc->client, dc->gconf->autologin, autologin, NULL);

	g_free (password);
	g_free (login);

	mesg = g_hash_table_lookup (dc->vars, "success");
	if (mesg == NULL || !strcmp (mesg, "FAIL"))
	{
		mesg = g_hash_table_lookup (dc->vars, "errmsg");
		display_error_dialog (dc, mesg);
		
//		journal_enable (dc, TRUE);
		
		return;
	}
	
	/* put our login name into the list of journals we can post to */
	dc->journal_list = g_slist_append (dc->journal_list, g_strdup (dc->username));
	
	mesg = g_hash_table_lookup (dc->vars, "message");
	if (mesg)
	{
		dialog = gtk_message_dialog_new (
					GTK_WINDOW (dc->current_window),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_INFO,
					GTK_BUTTONS_CLOSE,
					mesg);
		
		gtk_dialog_run (GTK_DIALOG (dialog));
		
		gtk_widget_destroy (dialog);
	}
	
	mesg = g_hash_table_lookup (dc->vars, "name");
	if (mesg)
		dc->name = g_strdup (mesg);
	else
		dc->name = g_strdup ("unknown");
	
	mesg = g_hash_table_lookup (dc->vars, "fastserver");
	if (mesg && !strcmp (mesg, "1"))
		dc->use_fast_servers = TRUE;
	else
		dc->use_fast_servers = FALSE;
	
	if (dc->picture_keywords != NULL)
		hash_table_clear (dc->picture_keywords);
	else
		dc->picture_keywords = g_hash_table_new_full (g_str_hash, g_str_equal,
				hash_table_item_free, hash_table_item_free);
	
	g_hash_table_insert (dc->picture_keywords, g_strdup ("pickw_0"),
				g_strdup ("(default)"));
	
	mesg = g_hash_table_lookup (dc->vars, "pickw_count");
	if (mesg)
	{
		pictures = (gint) g_ascii_strtod (mesg, NULL);
		dc->pictures = pictures;
		
		for (i = 1; i < pictures + 1; i++)
		{
			mesg = g_strdup_printf ("pickw_%d", i);

			g_hash_table_insert (dc->picture_keywords, mesg, 
					g_strdup (g_hash_table_lookup (dc->vars, mesg)));
		}
	}
	else
		dc->pictures = 0;
	
	mesg = g_hash_table_lookup (dc->vars, "mood_count");
	if (mesg)
	{
		new_moods = (gint) g_ascii_strtod (mesg, NULL);

		offset = 0;
		value = NULL;
		
		for (i = 1; i < new_moods + 1; i++)
		{			
			g_snprintf (mood_id, 32, "mood_%d_id", i);
			g_snprintf (mood_name, 32, "mood_%d_name", i);

			value = g_hash_table_lookup (dc->vars, mood_id);
			key = g_hash_table_lookup (dc->vars, mood_name);
			
			g_snprintf (static_key, 64, "/apps/drivel/moods/mood_%s", value);
			
			g_hash_table_insert (dc->mood_icons, g_strdup (key), g_strdup (value));
			gconf_client_set_string (dc->client, static_key, key, NULL);
			dc->mood_list = g_slist_append (dc->mood_list, g_strdup (key));
		}
		
		if (value)
			dc->moods = (gint) g_ascii_strtod (value, NULL);

		dc->mood_list = g_slist_sort (dc->mood_list, string_compare);
		
		gconf_client_set_int (dc->client, "/apps/drivel/moods/moods", dc->moods, NULL);
		gconf_client_set_list (dc->client, "/apps/drivel/moods/mood_list", GCONF_VALUE_STRING, dc->mood_list, NULL);
	}
	
	mesg = g_hash_table_lookup (dc->vars, "access_count");
	if (mesg)
	{
		dc->journals = (gint) g_ascii_strtod (mesg, NULL);
		
		for (i = 0; i < dc->journals; i++)
		{
			g_snprintf (static_key, 64, "access_%d", i + 1);
			mesg = g_hash_table_lookup (dc->vars, static_key);
			if (strncmp (mesg, dc->username, strlen (mesg)))
				dc->journal_list = g_slist_append (dc->journal_list, g_strdup (mesg));
		}
	}
	else
	{
		dc->journals = 1;
		dc->journal_list = g_slist_append (dc->journal_list, g_strdup (dc->username));
	}
	
	for (i = 0; ; i++)
	{
		g_snprintf (static_key, 64, "menu_%d_count", i);
		mesg = g_hash_table_lookup (dc->vars, static_key);

		if (mesg)
		{			
			menus = (gint) g_ascii_strtod (mesg, NULL);
			for (j = 1; j < menus + 1; j++)
			{
				menu_item = g_new0 (LJMenuItem, 1);
				
				g_snprintf (static_key, 64, "menu_%d_%d_text", i, j);
				mesg = g_hash_table_lookup (dc->vars, static_key);
				menu_item->label =g_strdup (mesg);
				
				g_snprintf (static_key, 64, "menu_%d_%d_url", i, j);
				mesg = g_hash_table_lookup (dc->vars, static_key);
				menu_item->url =g_strdup (mesg);
				
				g_snprintf (static_key, 64, "menu_%d_%d_sub", i, j);
				mesg = g_hash_table_lookup (dc->vars, static_key);
				if (mesg)
					menu_item->sub_menu = (gint) g_ascii_strtod (mesg, NULL);
				
				menu_item->menu_index = i;
				menu_item->item_index = j;
				
				dc->menu_list = g_slist_append (dc->menu_list, menu_item);
			}
		}
		else
			break;
	}
	
	gtk_widget_hide (dc->login_window);

	journal_window_build (dc);
	
	return;
}

void
request_process_postevent (DrivelClient *dc)
{
	gchar *mesg;
	
//	journal_enable (dc, TRUE);
	
	mesg = g_hash_table_lookup (dc->vars, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
	{
		mesg = g_hash_table_lookup (dc->vars, "errmsg");
		display_error_dialog (dc, mesg);
		
//		journal_enable (dc, TRUE);
		
		return;
	}
	
	journal_edit_entry_finished (dc);
	
	remove_autosave (dc);
	
	if (gconf_client_get_bool (dc->client, dc->gconf->min_post, NULL))
		gtk_window_iconify (GTK_WINDOW (dc->journal_window));
	
	return;
}

void
request_process_checkfriends (DrivelClient *dc)
{
	gchar *mesg;
	
	dc->checking_friends = FALSE;
	
	mesg = g_hash_table_lookup (dc->vars, "success");
	if (!mesg || !strcmp (mesg, "FAIL"))
		return;
	
	mesg = g_hash_table_lookup (dc->vars, "lastupdate");
	if (mesg)
	{
		g_free (dc->lastupdate);
		dc->lastupdate = g_strdup (mesg);
	}
	
	mesg = g_hash_table_lookup (dc->vars, "new");
	if (mesg)
	{
		if ((gint) g_strtod (mesg, NULL) == 1)
		{
			dc->friends_update = TRUE;
			tray_icon_update (TRAY_UPDATED_FRIENDS);
		}
		else
			dc->friends_update = FALSE;
	}
	
	mesg = g_hash_table_lookup (dc->vars, "interval");
	if (mesg)
		dc->time_since_checkfriends = (gint) g_strtod (mesg, NULL);
		
	return;
}

void
request_process (DrivelClient *dc, DrivelRequestType request)
{
	switch (request)
	{
		case REQUEST_EDITFRIENDS:
		{
			request_process_editfriends (dc);
			break;
		}
		case REQUEST_GETEVENTS:
		{
			request_process_getevents (dc);
			break;
		}
		case REQUEST_GETFRIENDS:
		{
			request_process_getfriends (dc);
			break;
		}
		case REQUEST_GETDAYCOUNTS:
		{
			request_process_getdaycounts (dc);
			break;
		}
		case REQUEST_LOGIN:
		{
			request_process_login (dc);
			break;
		}
		case REQUEST_POSTEVENT:
		{
			request_process_postevent (dc);
			break;
		}
		case REQUEST_CHECKFRIENDS:
		{
			request_process_checkfriends (dc);
			break;
		}
		default :
			g_assert_not_reached ();
	}

	return;
}
