/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2005 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <gnome.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs-cancellation.h>
#include <curl/curl.h>

#include "drivel.h"
#include "drivel_request.h"
#include "msg_queue.h"
#include "login.h"
#include "journal.h"
#include "xmlrpc.h"
#include "network.h"

#define USER_AGENT "GNOME-Drivel/" VERSION

extern GMutex *net_mutex;
extern gboolean verbose;
static GAsyncQueue *net_queue;

static void
send_error (DrivelClient *dc, const gchar *header, const gchar *msg)
{
	MsgInfo *info;
	
	info = msg_info_new ();
	info->type = MSG_TYPE_ERROR;
	info->msg = g_strdup (msg);
	info->header = g_strdup (header);
	info->dc = (gpointer)dc;
	g_async_queue_push (dc->net->msg_queue, info);
	
	return;
}

/* returns the next line of text in 'data' */

static size_t
write_cb (void *ptr, size_t size, size_t nmemb, void *data)
{
	gint realsize = size * nmemb;
	gint bufsize = realsize;
	DrivelRequestData *memory = data;

	if (memory->len == 0)
	{
		while(bufsize > 0 && (((gchar*)ptr)[0]=='\n' || ((gchar*)ptr)[0]==' ' || ((gchar*)ptr)[0] == '\t'))
		{
			ptr++;
			bufsize--;
		}
	}
	if (bufsize > 0)
	{
		memory->data = (gchar *) g_realloc (memory->data, memory->len + bufsize + 1);

		memcpy (&(memory->data [memory->len]), ptr, bufsize);
		memory->len += bufsize;
		memory->data [memory->len] = '\0';
	}
	return realsize;
}

static size_t
read_cb (gchar *buffer, size_t size, size_t nmemb, void *data)
{
	gchar *xml_copied, *remaining_data;
	gint bytes_copied = 0, maxsize = size *nmemb;
	DrivelRequestData *memory = data;
	
	if (memory->data)
	{
		xml_copied = strncpy (buffer, memory->data, maxsize);
		bytes_copied = strlen (xml_copied);
		remaining_data = g_strdup (memory->data + bytes_copied);
		g_free (memory->data);
		memory->data = remaining_data;
	}

	return bytes_copied;
}

static gchar*
get_line (const gchar *data, gint *offset)
{
	gchar *line;
	guint i, len;

	len = strlen (data);
	
	for (i = 0; i < len; i++)
	{
		if (data [i] == '\r' && data [i + 1] == '\n')
		{
			*offset = 2;
			break;
		}
		else if (data [i] == '\r' || data [i] == '\n')
		{
			*offset = 1;
			break;
		}
	}
	
	line = g_new0 (gchar, i + 1);
	memcpy (line, data, i);
	line [i] = '\0'; /* trash the newline */

	return line;
}

/* store the key/value pairs in 'string' to a hash table for later processing */

static void
parse_to_hash_table (DrivelClient *dc, DrivelRequest *dr, const gchar *string, gint len)
{
	guint offset, line_offset;
	gchar *key, *value;

	offset = 0;
	
	if (!g_utf8_validate (string, -1, NULL))
		g_warning ("It looks like this string is not valid UTF8!");
    
	drivel_request_clear_values (dr);
	
	while (offset < len)
	{
		key = get_line (string + offset, (gint *)&line_offset);
		offset += strlen (key) + line_offset;
		if (line_offset > 1)
		{
			/* this is the http header, discard it */
			g_free (key);
			continue;
		}

		value = get_line (string + offset, (gint *)&line_offset);
		offset += strlen (value) + line_offset;

		drivel_request_value_insert (dr, key, value);
		
		g_free (key);
		g_free (value);
	}
	
	return;
}

/* use the msg-queue to update the progress dialog */

static gint
request_progress (gpointer *data, double dltotal, double dlnow, double ultotal, 
		double ulnow)
{
	gboolean retval;
	gdouble percent;
	MsgInfo *info;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (dltotal)
		percent = dlnow / dltotal;
	else if (ultotal)
		percent = ulnow / ultotal;
	else
		percent = 0.0;
	
	info = msg_info_new ();
	info->type = MSG_TYPE_UPDATE_PROGRESS_PERCENT;
	info->progress = percent;
	g_async_queue_push (dc->net->msg_queue, info);
	
	/* cancel this request and anything which is already queued */
	retval = gnome_vfs_cancellation_check (dc->net->cancel);
	if (retval)
	{
		DrivelRequest *dr;
		
		do
		{
			dr = g_async_queue_try_pop (net_queue);
			if (dr)
				drivel_request_free (dr);
		} while (dr);
		
		gnome_vfs_cancellation_ack (dc->net->cancel);
	}
	
	return retval;
}

static void
setup_proxies (DrivelClient *dc, CURL *session, gchar **proxy_userpwd, gchar **proxy_url)
{
	gchar *user, *pass, *url, *userpwd;
	gboolean use_proxy, use_auth;
	gint port;
	
	g_mutex_lock (net_mutex);
	use_proxy = dc->proxy;
	if (dc->proxy_user)
		user = g_strdup (dc->proxy_user);
	else
		user = NULL;
	if (dc->proxy_pass)
		pass = g_strdup (dc->proxy_pass);
	else
		pass = NULL;
	userpwd = NULL;
	if (dc->proxy_url)
	{
		gchar *colon1, *colon2;
		
		/* Check for an IPv6 URL by searching for at least two colons (:) */
		colon1 = strstr (dc->proxy_url, ":");
		if (colon1)
			colon2 = strstr (colon1 + 1, ":");
		else
			colon2 = NULL;
		if (colon2)
			url = g_strdup_printf ("[%s]", dc->proxy_url);
		else
			url = g_strdup (dc->proxy_url);
	}
	else
		url = NULL;
	port = dc->proxy_port;
	use_auth = dc->proxy_auth;
	g_mutex_unlock (net_mutex);

	/* Proxy stuff */
	if (use_proxy)
	{
		gchar *msg;
		
		/* Newer releases of GNOME use the "use_authentication" GConf key
		 * to enable/disable authentication, older releases just cleared the
		 * username */
		if (use_auth && user && user [0] != '\0' && pass && pass [0] != '\0')
		{
			userpwd = g_strdup_printf ("%s:%s", user, pass);	
			curl_easy_setopt (session, CURLOPT_PROXYUSERPWD, userpwd);
		}
		/* We listen to GConf for changes to the proxy URL and port, so these
		   variables should always be up to date */
		msg = g_strdup_printf (
				"Proxy enabled:\nURL: %s\nPort: %d\nAuthentication: %s\n",
				url, port, userpwd);
		debug (msg);
		g_free (msg);
		
		curl_easy_setopt (session, CURLOPT_PROXY, url);
		curl_easy_setopt (session, CURLOPT_PROXYPORT, port);
	}
	
	g_free (user);
	g_free (pass);
	
	if (userpwd)
		*proxy_userpwd = userpwd;
	else
		*proxy_userpwd = NULL;
	if (url)
		*proxy_url = url;
	else
		*proxy_url = NULL;

	return;
}

/* build the authentication response token for a livejournal server */

static gchar*
get_response_lj (const gchar *pass, const gchar *challenge)
{
	gchar *response, *hash;
	
	response = g_strdup_printf ("%s%s", challenge, pass);
	hash = md5_hash (response);
	g_free (response);
	
	return g_strdup (hash);
}

/* get a authentication challenge token from a livejournal server */
static gint
get_challenge_lj (DrivelClient *dc, DrivelRequest *dr, const gchar *url, 
		gint fast_servers, gchar **challenge)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *memory;
	gchar *cookie, *proxy_userpwd, *proxy_url;
	glong http_code;
	gint retval;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	proxy_userpwd = proxy_url = NULL;
	
	cookie = g_strdup_printf ("ljfastserver=%d", fast_servers);
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt (session, CURLOPT_USERAGENT, USER_AGENT);
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_POSTFIELDS, "&mode=getchallenge");
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookie);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code == 200 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		parse_to_hash_table (dc, dr, memory->data, memory->len);
		*challenge = g_strdup (drivel_request_value_lookup (dr, "challenge"));
		retval = 0;
	}
	else
	{
		*challenge = NULL;
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = 1;
		else
		{
			g_warning ("Could not get a challenge token from the server.");
			retval = -1;
		}
	}
	
	g_free (proxy_userpwd);
	g_free (proxy_url);
	
	return retval;
}

/* concat the items of 'dr' together in HTTP/POST format */

static gchar*
build_post_request (DrivelRequest *dr)
{
	gchar *string;
	gboolean valid;
	
	string = g_strdup ("");
	valid = drivel_request_start (dr);
	while (valid)
	{
		DrivelRequestItem *item;
		gchar *temp;

		item = drivel_request_get_current_item (dr);
		temp = g_strdup (string);
		g_free (string);
		string = g_strdup_printf ("%s&%s=%s", temp, item->key, item->value);
				
		valid = drivel_request_next (dr);
	}
	
	return string;
}

/* perform the HTTP/POST operation */
static gint
perform_post (DrivelClient *dc, DrivelRequest *dr, const gchar *url, 
		const gchar *request_string, const gchar *cookies)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *memory;
	gchar *proxy_userpwd, *proxy_url;
	glong http_code;
	gint retval;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	proxy_userpwd = proxy_url = NULL;
	
	debug ("post url:");
	debug (url);
	debug ("post field:");
	debug (request_string);
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt (session, CURLOPT_USERAGENT, USER_AGENT);
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_POSTFIELDS, request_string);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookies);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	if (!curl_code)
		curl_code = curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code >= 200 && http_code < 300 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		debug ("post result:");
		debug (memory->data);
		
		parse_to_hash_table (dc, dr, memory->data, memory->len);
		retval = 0;
	}
	else
	{
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = 1;
		else
		{
			retval = -1;
			g_warning ("HTTP/POST request failed (HTTP code %d: %s).", 
				   (gint)http_code, curl_easy_strerror (curl_code));
		}
	}
	
	g_free (memory->data);
	g_free (memory);
	g_free (proxy_userpwd);
	g_free (proxy_url);
	
	return retval;
}

/* perform the HTTP/GET operation */
static gint
perform_get (DrivelClient *dc, DrivelRequest *dr, const gchar *url, 
		const gchar *cookies, struct curl_slist *headers)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *memory;
	gchar *proxy_userpwd, *proxy_url, *userpass;
	const gchar *http_type;
	glong http_code;
	gint retval;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	proxy_userpwd = proxy_url = userpass = NULL;
	
	debug ("get url:");
	debug (url);
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt (session, CURLOPT_USERAGENT, USER_AGENT);
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookies);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	curl_easy_setopt (session, CURLOPT_HTTPHEADER, headers);
	if (drivel_request_item_lookup (dr, "http_basic"))
	{
		const gchar *username, *password;
		
		username = drivel_request_item_lookup (dr, "username");
		password = drivel_request_item_lookup (dr, "password");
		userpass = g_strdup_printf ("%s:%s", username, password);
		
		curl_easy_setopt (session, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
		curl_easy_setopt (session, CURLOPT_USERPWD, userpass);
	}
	http_type = drivel_request_item_lookup (dr, "http_type");
	if (http_type && !strcmp (http_type, "delete"))
		curl_easy_setopt (session, CURLOPT_CUSTOMREQUEST, "DELETE");
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	if (!curl_code)
		curl_code = curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if ((http_code == 200 || http_code == 204) && 
		curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		debug ("get result:");
		debug (memory->data);
		
		drivel_request_set_data (dr, memory);
		retval = 0;
	}
	else
	{
		g_free (memory->data);
		g_free (memory);
		
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = 1;
		else
		{
			retval = -1;
			g_warning ("HTTP/GET request failed (HTTP code %d: %s).", 
				   (gint)http_code, curl_easy_strerror (curl_code));
		}
	}
	
	g_free (proxy_userpwd);
	g_free (proxy_url);
	g_free (userpass);
	
	return retval;
}

/* perform the HTTP/POST operation using XML-RPC */
static gint
perform_xmlrpc_post (DrivelClient *dc, DrivelRequest *dr, const gchar *url)
{
	CURL *session;
	CURLcode curl_code;
	struct curl_slist *headers;
	DrivelRequestData *memory;
	gchar *proxy_userpwd, *proxy_url;
	const gchar *post_field;
	glong http_code;
	gint retval;
	GHashTable *table;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	headers = NULL;
	headers = curl_slist_append (headers, "Content-Type: text/xml");
	headers = curl_slist_append (headers, "Pragma:");
	headers = curl_slist_append (headers, "Accept:");
	proxy_userpwd = proxy_url = NULL;
	post_field = drivel_request_item_lookup (dr, "xml");
	
	debug ("xmlrpc url:");
	debug (url);
	debug ("xmlrpc post field:");
	debug (post_field);
	
	if (!post_field)
		g_warning ("post_field is NULL");
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt (session, CURLOPT_USERAGENT, USER_AGENT);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_POSTFIELDS, post_field);
	curl_easy_setopt (session, CURLOPT_HTTPHEADER, headers);
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	if (!curl_code)
		curl_code = curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code >= 200 && http_code < 300 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		debug ("xmlrpc result:");
		debug (memory->data);
		
		table = xmlrpc_parse_packet (memory->data, memory->len);
		drivel_request_set_values (dr, table);
		retval = 0;
	}
	else
	{
		g_free (memory->data);
		g_free (memory);
		
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = 1;
		else
		{
			retval = -1;
			g_warning ("HTTP/POST XML-RPC request failed (HTTP code %d: %s).", 
				   (gint)http_code, curl_easy_strerror (curl_code));
		}
	}
	
	curl_slist_free_all (headers);
	g_free (proxy_userpwd);
	g_free (proxy_url);
	
	return retval;
}

/* Perform an HTTP/POST operation with an Atom entry */
static gint
perform_atom_post (DrivelClient *dc, DrivelRequest *dr, 
		struct curl_slist *headers)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *wmemory, *rmemory;
	gchar *proxy_userpwd, *proxy_url, *userpass;
	const gchar *post_field;
	glong http_code;
	gint retval;
	
	wmemory = g_new (DrivelRequestData, 1);
	wmemory->len = 0;
	wmemory->data = NULL;
	rmemory = g_new (DrivelRequestData, 1);
	rmemory->len = 0;
	rmemory->data = NULL;
	
	proxy_userpwd = proxy_url = userpass = NULL;
	post_field = drivel_request_item_lookup (dr, "xml");
	
	debug ("atom post field:");
	debug (post_field);
	
	if (!post_field)
		g_warning ("post_field is NULL");
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt (session, CURLOPT_USERAGENT, USER_AGENT);
	curl_easy_setopt (session, CURLOPT_URL, drivel_request_get_uri (dr));
	curl_easy_setopt (session, CURLOPT_HTTPHEADER, headers);
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, wmemory);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	if (drivel_request_item_lookup (dr, "http_basic"))
	{
		const gchar *username, *password;
		
		username = drivel_request_item_lookup (dr, "username");
		password = drivel_request_item_lookup (dr, "password");
		userpass = g_strdup_printf ("%s:%s", username, password);
		
		curl_easy_setopt (session, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
		curl_easy_setopt (session, CURLOPT_USERPWD, userpass);
	}
	if (drivel_request_item_lookup (dr, "http_put"))
	{
		rmemory->data = g_strdup (post_field);
		curl_easy_setopt (session, CURLOPT_UPLOAD, TRUE);
		curl_easy_setopt (session, CURLOPT_READFUNCTION, read_cb);
		curl_easy_setopt (session, CURLOPT_READDATA, rmemory);
	}
	else
		curl_easy_setopt (session, CURLOPT_POSTFIELDS, post_field);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	if (!curl_code)
		curl_code = curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code >= 200 && http_code < 300 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		debug ("atom result:");
		debug (wmemory->data);
		
		/* FIXME: Parse the result */
		retval = 0;
	}
	else
	{
		debug ("atom failed result:");
		debug (wmemory->data);
		
		g_free (wmemory->data);
		g_free (wmemory);
		
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = 1;
		else
		{
			retval = -1;
			g_warning ("HTTP/POST Atom request failed (HTTP code %d: %s).",
				   (gint)http_code, curl_easy_strerror (curl_code));
		}
	}
	
	g_free (proxy_userpwd);
	g_free (proxy_url);
	g_free (userpass);
	g_free (rmemory->data);
	g_free (rmemory);
	
	return retval;
}

/* Update the network progress dialog to explain to the user what is happening */

static void
update_progress_msg (GAsyncQueue *queue, DrivelRequestType type)
{
	MsgInfo *info;
	gchar *msg, *header;
	
	switch (type)
	{
		case REQUEST_TYPE_LOGIN:
		{
			msg = g_strdup (_("Retrieving user information"));
			break;
		}
		case REQUEST_TYPE_GETPICTURE:
		{
			msg = g_strdup (_("Downloading user pictures"));
			break;
		}
		case REQUEST_TYPE_POSTEVENT:
		{
			msg = g_strdup (_("Posting journal entry"));
			break;
		}
		case REQUEST_TYPE_EDITEVENT:
		{
			msg = g_strdup (_("Updating journal entry"));
			break;
		}
		case REQUEST_TYPE_GETEVENTS:
		{
			msg = g_strdup (_("Retrieving journal entries"));
			break;
		}
		case REQUEST_TYPE_GETDAYCOUNTS:
		{
			msg = g_strdup (_("Retrieving journal history"));
			break;
		}
		case REQUEST_TYPE_EDITFRIENDS:
		{
			msg = g_strdup (_("Updating Friends list"));
			break;
		}
		case REQUEST_TYPE_GETFRIENDS:
		{
			msg = g_strdup (_("Retrieving Friends list"));
			break;
		}
		case REQUEST_TYPE_GETCATEGORIES:
		case REQUEST_TYPE_GETPOSTCATEGORIES:
		{
			msg = g_strdup (_("Retrieving categories"));
			break;
		}
		case REQUEST_TYPE_SETPOSTCATEGORIES:
		{
			msg = g_strdup (_("Setting categories"));
			break;
		}
		case REQUEST_TYPE_PUBLISH:
		{
			msg = g_strdup (_("Publishing journal entry"));
			break;
		}
		case REQUEST_TYPE_DELETEEVENT:
		{
			msg = g_strdup (_("Deleting journal entry"));
			break;
		}
		case REQUEST_TYPE_PING:
		{
			msg = g_strdup (_("Notifying Technorati"));
			break;
		}
		case REQUEST_TYPE_GETFRIENDGROUPS:
		{
			msg = g_strdup (_("Retrieving security groups"));
			break;
		}
		case REQUEST_TYPE_SETFRIENDGROUPS:
		{
			msg = g_strdup (_("Updating security groups"));
			break;
		}
		default:
		{
			msg = g_strdup (_("We're doing something, but I'm not sure what"));
			break;
		}
	}
	
	info = msg_info_new ();
	info->type = MSG_TYPE_UPDATE_PROGRESS_LABEL;
	header = g_strdup (_("Sending / Receiving"));
	info->msg = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">"
			"%s</span>\n\n"
			"%s...", header, msg);
	g_async_queue_push (queue, info);
	
	g_free (header);
	g_free (msg);
	
	return;
}

/* this thread never dies, it just loops, waiting for input from 
   net_enqueue_request() */

static gpointer
net_thread (DrivelClient *dc)
{
	while (TRUE)
	{
		DrivelRequest *dr;
		DrivelRequestType type;
		MsgInfo *info;
		gint retval = 0;
		gchar *request_string = NULL, *cookies = NULL;

		dr = g_async_queue_pop (net_queue);

		type = drivel_request_get_type (dr);
		/* for everything except CHECKFRIENDS, we re-parent the dialog to
		   ensure it is transient for the top window, and then update its
		   message. */
		if (type != REQUEST_TYPE_CHECKFRIENDS)
		{
			info = msg_info_new ();
			info->type = MSG_TYPE_REPARENT_DIALOG;
			info->dc = (gpointer) dc;
			g_async_queue_push (dc->net->msg_queue, info);
			update_progress_msg (dc->net->msg_queue, type);
		}
		
		switch (drivel_request_get_protocol (dr))
		{
			case REQUEST_PROTOCOL_POST:
			{
				gchar *url = NULL;
				
				switch (drivel_request_get_api (dr))
				{
					case BLOG_API_LJ:
					{
						gint fast_servers;
						gchar *challenge, *response, *password;
						
						g_mutex_lock (net_mutex);
						fast_servers = dc->net->fast_servers;
						password = md5_hash (dc->user->password);
						g_mutex_unlock (net_mutex);
						
						url = g_strdup_printf ("%s/interface/flat", 
								drivel_request_get_uri (dr));
						cookies = g_strdup_printf ("ljfastserver=%d", fast_servers);
						retval = get_challenge_lj (dc, dr, url, fast_servers, &challenge);
						response = get_response_lj (password, challenge);
						drivel_request_add_items (dr,
								g_strdup ("auth_method"), g_strdup ("challenge"),
								g_strdup ("auth_challenge"), challenge,
								g_strdup ("auth_response"), response,
								NULL);
						
						g_free (password);

						break;
					}
					default:
					{
						g_warning ("net_thread: Unknown journal API.");
						break;
					}
				}
				
				request_string = build_post_request (dr);
				if (!retval)
				{
					retval = perform_post (dc, dr, url, request_string, cookies);
				}
				if (retval < 0)
				{
					send_error (dc, _("Communication Error"),
							_("There was a problem sending information to the "
							"server.  Please try again later."));
				}
				
				g_free (url);
				
				break;
			}
			case REQUEST_PROTOCOL_GET:
			{
				const gchar *url;
				gchar *wsse = NULL;
				struct curl_slist *headers = NULL;
					
				switch (drivel_request_get_api (dr))
				{
					case BLOG_API_LJ:
					{
						gint fast_servers;
						
						if (drivel_request_item_lookup (dr, "fastservers"))
							fast_servers = 1;
						else
							fast_servers = 0;
						
						cookies = g_strdup_printf ("ljfastserver=%d", fast_servers);
						
						break;
					}
					case BLOG_API_ATOM:
					case BLOG_API_MT:
						/* nothing special to do for these */
						break;
					default:
					{
						g_warning ("net_thread: Unknown journal API.");
						break;
					}
				}
				
				url = drivel_request_get_uri (dr);
				retval = perform_get (dc, dr, url, cookies, headers);
				if (retval < 0)
				{
					send_error (dc, _("Communication Error"),
							_("There was a problem receiving information from "
							"the server.  Please try again later."));
				}
				
				g_free (wsse);
				curl_slist_free_all (headers);
				
				break;
			}
			case REQUEST_PROTOCOL_XMLRPC:
			{
				const gchar *url = NULL;
				
				switch (drivel_request_get_api (dr))
				{
					case BLOG_API_ADVOGATO:
					{
						url = g_strdup_printf ("%s/XMLRPC", 
								drivel_request_get_uri (dr));
						break;
					}
					case BLOG_API_BLOGGER:
					case BLOG_API_MT:
					case BLOG_API_GENERIC:
					{
						url = drivel_request_get_uri (dr);
						break;
					}
					default:
					{
						g_warning ("net_thread: Unknown journal API.");
						break;
					}
				}
				
				retval = perform_xmlrpc_post (dc, dr, url);
				if (retval < 0)
				{
					send_error (dc, _("Communication Error"),
							_("There was a problem sending information to the "
							"server.  Please try again later."));
				}
				
				break;
			}
			case REQUEST_PROTOCOL_ATOM:
			{
				struct curl_slist *headers = NULL;
				
				headers = curl_slist_append (headers, "Content-Type: text/xml");
						
				retval = perform_atom_post (dc, dr, headers);
				if (retval < 0)
				{
					send_error (dc, _("Communication Error"),
							_("There was a problem sending information to the "
							"server.  Please try again later."));
				}
				
				curl_slist_free_all (headers);
				
				break;
			}
			case REQUEST_PROTOCOL_NONE:
			{
				/* we don't do anything, just pass the DrivelRequest on */
				break;
			}
		}
		
		g_free (request_string);
		g_free (cookies);
		
		if (!retval)
		{
			info = msg_info_new ();
			info->type = MSG_TYPE_PROCESS_NET_REQUEST;
			info->dr = (gpointer) dr;
			info->dc = (gpointer) dc;
			g_async_queue_push (dc->net->msg_queue, info);
		}
		
		if (type != REQUEST_TYPE_CHECKFRIENDS)
		{
			g_usleep (500000);
			info = msg_info_new ();
			info->type = MSG_TYPE_DONE;
			g_async_queue_push (dc->net->msg_queue, info);
		}
	}
	
	return NULL;
}

/* add a network request to the queue for processing */
void
net_enqueue_request (DrivelClient *dc, DrivelRequest *dr)
{
	MsgInfo *info;
	DrivelRequestType type;
	
	if (!dr)
		return;
	
	/* first build the progress dialog */
	type = drivel_request_get_type (dr);
	if (type != REQUEST_TYPE_CHECKFRIENDS)
	{
		info = msg_info_new ();
		info->type = MSG_TYPE_BUILD_PROGRESS;
		info->msg = g_strdup ("<span weight=\"bold\" size=\"larger\">"
				"Sending / Receiving</span>\n\n"
				"Begining network transaction...");
		info->widget = GTK_WINDOW (dc->current_window);
		info->dc = (gpointer)dc;
		g_async_queue_push (dc->net->msg_queue, info);
	}
	
	/* then queue the network request */
	g_async_queue_push (net_queue, dr);
	
	return;
}

/* start the network thread */
gint
net_start_thread (DrivelClient *dc)
{
	static gboolean inited = FALSE;
	gint retval;
	
	/* we can only start one network thread */
	if (inited)
		return -1;
	
	net_queue = g_async_queue_new ();
	
	if (g_thread_create ((GThreadFunc) net_thread, dc, FALSE, NULL))
		retval = 0;
	else
		retval = -1;
	
	return retval;
}

/* send a ping to technorati */
void
net_ping_technorati (DrivelClient *dc)
{
	DrivelRequest *dr;
	gchar *packet;
	const gchar *name;
	
	debug ("net_ping_technorati ()");
	
	if (!gconf_client_get_bool (dc->client, dc->gconf->technorati, NULL))
		return;
	
	/* Use the LiveJournal description as the Technorati title for LJ users */
	if ((dc->user->api == BLOG_API_LJ) && 
		(dc->active_journal->type == JOURNAL_TYPE_USER))
		name = dc->active_journal->description;
	else
		name = dc->active_journal->name;
	
	packet = xmlrpc_build_packet ("weblogUpdates.ping", 
			XMLRPC_TYPE_STRING, name,
			XMLRPC_TYPE_STRING, dc->active_journal->uri_view,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_PING,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_GENERIC, 
			"http://rpc.technorati.com/rpc/ping",
			g_strdup ("xml"), packet,
			NULL);
	
	net_enqueue_request (dc, dr);
	
	return;
}
