/*  dvdisaster: Additional error correction for optical media.
 *  Copyright (C) 2004,2005 Carsten Gnoerlich.
 *  Project home page: http://www.dvdisaster.com
 *  Email: carsten@dvdisaster.com  -or-  cgnoerlich@fsfe.org
 *
 *  The Reed-Solomon error correction draws a lot of inspiration - and even code -
 *  from Phil Karn's excellent Reed-Solomon library: http://www.ka9q.net/code/fec/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 *  or direct your browser at http://www.gnu.org.
 */

#include "dvdisaster.h"

#include "rs01-window.h"

/***
 *** Method registration
 ***/

static void create(Method*);
static void fix(Method*);
static void destroy(Method*);

void register_rs01(void)
{  Method *method = g_malloc0(sizeof(Method));

   /*** Standard infomation and methods */ 

   strncpy(method->name, "RS01", 4);
   method->description = _("Classic Reed-Solomon method based on polynomial arithmetic"); 
   method->create = create;
   method->fix    = fix;

   /*** Linkage to rs01-window.c */

   method->createCreateWindow = CreateRS01EWindow;
   method->createFixWindow    = CreateRS01FWindow;

   method->resetCreateWindow = ResetRS01EncodeWindow;
   method->resetFixWindow    = ResetRS01FixWindow;

   method->createPrefsPage   = CreateRS01PrefsPage;

   method->destroy = destroy;

   /*** Register ourself */

   RegisterMethod(method);
}

static void destroy(Method *method)
{  RS01Widgets *wl = (RS01Widgets*)method->widgetList;

   if(wl)
   {  if(wl->fixCurve) FreeCurve(wl->fixCurve);
      g_free(method->widgetList);
   }
}

/***
 *** Galois field arithmetic.
 *** 
 * Calculations are done over the extension field GF(2**n).
 * Be careful not to overgeneralize these arithmetics;
 * they only work for the case of GF(p**n) with p being prime.
 */


#define SYMBOLSIZE 8
#define FIELDSIZE (1<<SYMBOLSIZE)
#define FIELDMAX (FIELDSIZE-1)
#define ALPHA0 FIELDMAX

#define GENERATOR_POLY 0x187    /* 1 + X + X**2 + X**7 + X**8 */
#define FIRST_ROOT 112          /* same choices as in CCSDS */                 
#define PRIM_ELEM 11                  
#define PRIMTH_ROOT 116         /* prim-th root of 1 */           

#define	min(a,b) ((a) < (b) ? (a) : (b))

static inline int mod_fieldmax(int x)
{
  while (x >= FIELDMAX) 
  {
    x -= FIELDMAX;
    x = (x >> SYMBOLSIZE) + (x & FIELDMAX);
  }

  return x;
}

/*
 * Interpret our redundancy settings
 *
 * a) "normal" -> nroots=32; "high" -> nroots=64
 * b) "n"      -> nroots = n
 * c) "n%"     -> use closest nroots for given redundancy in percent
 * d) "nm"     -> choose redundancy so that .ecc file does not exceed n megabytes
 */

static gint64 ecc_file_size(gint64 sectors, int nr)
{  int nd = FIELDMAX - nr;
   gint64 bytesize; 

   bytesize = 4096 + 4*sectors + 2048*nr*((sectors+nd-1)/nd);

   return (bytesize+0xfffff)/0x100000;   /* size in MB */
}		 

static int calculate_redundancy(char *image_name)
{  struct stat mystat; 
   int nr = 0;
   char last = 0; 
   double p;
   int ignore;
   gint64 fs,sectors;

   if(Closure->redundancy) /* get last char of redundancy parameter */
   {  int len = strlen(Closure->redundancy);

      if(len) last = Closure->redundancy[len-1];
   }

   switch(last)
   {  case '%' : p = atof(Closure->redundancy);
                 if(p<3.2 || p>64.5) Stop(_("Redundancy %4.1f%% out of useful range [3.2%%..64.5%%]"),p);
		 nr = (int)round((FIELDMAX*p) / (100.0+p));
	         break;

      case 'm' : if(stat(image_name, &mystat) == -1)
  	         {  nr = 32;   /* If the image file is not present, simply return 32. */
		    break;     /* Later stages will fail anyways, but can report the error */
	         }             /* in a more meaningful context. */

	         CalcSectors(mystat.st_size, &sectors, &ignore);

	         fs = strtoll(Closure->redundancy, NULL, 10);
	         if(fs < ecc_file_size(sectors, 8) || fs > ecc_file_size(sectors, 100))
		   Stop(_("Ecc file size %lldm out of useful range [%lld .. %lld]"),
			fs, ecc_file_size(sectors, 8), ecc_file_size(sectors, 100));
		 for(nr=100; nr>8; nr--)
		   if(fs >= ecc_file_size(sectors, nr))
		     break;
	         break;

      default:
	if(!Closure->redundancy || !strcmp(Closure->redundancy, "normal")) nr = 32; 
	else if(!strcmp(Closure->redundancy, "high")) nr = 64;
	else nr = atoi(Closure->redundancy);
	break;
   }

   if(nr < 8 || nr > 100)
     Stop(_("Redundancy %d out of useful range [8..100]."),nr);

   return nr;
}

/* Initialize the Galois field tables */

static int nroots;        /* degree of generator polynomial */
static int ndata;         /* data bytes per ecc block */

static int *gf_index_of;  /* log */
static int *gf_alpha_to;  /* inverse log */
static int *enc_alpha_to; /* inverse log for encoder */
static int *gf_gpoly;     /* generator polynomial */

static void init_tables(int nroots_in)
{  int i,j,b,log,root;

   nroots = nroots_in;
   ndata  = FIELDMAX - nroots;

   /* Allocate the tables.
      The encoder uses a special version of alpha_to which has the mod_fieldmax()
      folded into the table. */

   gf_index_of  = g_malloc(FIELDSIZE * sizeof(int));
   gf_alpha_to  = g_malloc(FIELDSIZE * sizeof(int));
   enc_alpha_to = g_malloc(2*FIELDSIZE * sizeof(int));
   gf_gpoly     = g_malloc((nroots+1) * sizeof(int));
   
   /* create the log/ilog values */

   for(b=1, log=0; log<FIELDMAX; log++)
   {  gf_index_of[b]   = log;
      gf_alpha_to[log] = b;
      b = b << 1;
      if(b & FIELDSIZE)
	b = b ^ GENERATOR_POLY;
   }

   if(b!=1) Stop(_("Failed to create the log tables!\n"));

   /* we're even closed using infinity (makes things easier) */

   gf_index_of[0] = ALPHA0;    /* log(0) = inf */
   gf_alpha_to[ALPHA0] = 0;   /* and the other way around */

   for(b=0; b<2*FIELDSIZE; b++)
     enc_alpha_to[b] = gf_alpha_to[mod_fieldmax(b)];

   /* Create the RS code generator polynomial */

   gf_gpoly[0] = 1;

   for(i=0, root=FIRST_ROOT*PRIM_ELEM; i<nroots; i++, root+=PRIM_ELEM)
   {  gf_gpoly[i+1] = 1;

     /* Multiply gf_gpoly  by  alpha**(root+x) */

     for(j=i; j>0; j--)
     {
       if(gf_gpoly[j] != 0)
         gf_gpoly[j] = gf_gpoly[j-1] ^ gf_alpha_to[mod_fieldmax(gf_index_of[gf_gpoly[j]] + root)];
       else
	 gf_gpoly[j] = gf_gpoly[j-1];
     }

     gf_gpoly[0] = gf_alpha_to[mod_fieldmax(gf_index_of[gf_gpoly[0]] + root)];
   }

   /* Store the polynomials index for faster encoding */ 

   for(i=0; i<=nroots; i++)
     gf_gpoly[i] = gf_index_of[gf_gpoly[i]];

#if 0
   /* for the precalculated unrolled loops only */

   for(i=nroots-1; i>0; i--)
     PrintCLI(
	    "                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + %3d];\n",
	    gf_gpoly[i]);

   PrintCLI("                  par_idx[sp] = enc_alpha_to[feedback + %3d];\n",
	  gf_gpoly[0]);
#endif

}

static void free_tables(void)
{
  if(gf_index_of)  g_free(gf_index_of);
  if(gf_alpha_to)  g_free(gf_alpha_to);
  if(enc_alpha_to) g_free(enc_alpha_to);
  if(gf_gpoly)     g_free(gf_gpoly);

  gf_index_of  = NULL;
  gf_alpha_to  = NULL;
  enc_alpha_to = NULL;
  gf_gpoly     = NULL;
}

/*
 * Special debugging options which are only useful during
 * development and back testing. Having them on in production
 * releases is not recommended.
 */

/*
 * Read an image sector from the .img file.
 *
 * Two special cases here:
 * - Missing sectors (beyond the range recorded in eh->sectors) will be padded with zeros,
 *   since we need a multiple of ndata sectors for the parity generation. 
 * - Missing sectors beyond the range recorded in ii->sectors, but before the real end
 *   as defined above are treated as "dead sectors".
 */

static void read_sector(ImageInfo *ii, EccHeader *eh, unsigned char *buf, gint64 s)
{ gint64 eh_sectors = uchar_to_gint64(eh->sectors);

  if(s >= ii->sectors && s < eh_sectors)
  {
     memcpy(buf, Closure->deadSector, 2048);   /* truncated image / dead sector */
  }
  else if(s >= eh_sectors)      
  {
     memset(buf, 0, 2048);            /* zero padding for reads past the image */
  }
  else                                /* else normal read within the image */
  {  int n,expected;
	
     if(!LargeSeek(ii->file, (gint64)(2048*s)))
       Stop(_("lseek failed for %s, sector %lld"),Closure->imageName,s);

     /* Prepare for short reads at the last image sector.
	Doesn't happen for CD and DVD media, but perhaps for future media? */

     if(s < ii->sectors-1) expected = 2048;
     else  
     {  memset(buf, 0, 2048);
        expected = ii->inLast;
     }

     /* Finally, read the sector */

     n = LargeRead(ii->file, buf, expected);
     if(n != expected)
       Stop(_("Could not read image sector %lld:\n%s"),s,strerror(errno));
  }
}

/*
 * Read crc values from the .ecc file.
 */

static void read_crc(FILE *ecc, guint32 *buf, int first_sector, int n_sectors)
{  int n;
  
   if(fseeko(ecc, (gint64)(sizeof(EccHeader) + first_sector*sizeof(guint32)), SEEK_SET) == -1)
     Stop(_("fseek failed reading crc: %s"), strerror(errno));

   n = fread(buf, sizeof(guint32), n_sectors, ecc);
	
   if(n != n_sectors)
     Stop(_("problem reading crc data: %s"),strerror(errno));
}

/***
 *** Fix the medium sectors.
 ***
 * Note that if DEBUG_ECC is defined, we check the syndromes of each ecc block
 * regardless whether its CRC sum is correct (slow!), but do not actually
 * correct anything in the image file. 
 * This mode is just for testing the integrity of the RS codec and works
 * best with a complete/undamaged image file.
 */

/*
 * Local data package used during fixing
 */

typedef struct
{  RS01Widgets *wl;
   int earlyTermination;
   char *msg;
   ImageInfo *ii;
   EccInfo *ei;
   unsigned char *imgBlock[256];
   guint32 *crcBuf[256];
} fix_closure;

static void fix_cleanup(gpointer data)
{  fix_closure *fc = (fix_closure*)data;
   int i;

   Closure->cleanupProc = NULL;

   if(Closure->guiMode)
   {  if(fc->earlyTermination)
         SwitchAndSetFootline(fc->wl->fixNotebook, 1,
			      fc->wl->fixFootline,
			      _("<span color=\"red\">Aborted by unrecoverable error.</span>")); 
      AllowActions(TRUE);
   }

   /** Clean up */

   free_tables();

   if(fc->msg) g_free(fc->msg);
   if(fc->ii) FreeImageInfo(fc->ii);
   if(fc->ei) FreeEccInfo(fc->ei);

   for(i=0; i<ndata; i++)
   {  g_free(fc->imgBlock[i]);
      g_free(fc->crcBuf[i]);
   }

   g_free(fc);

   g_thread_exit(0);
}

/*
 * Try to repair the image 
 */

static void fix(Method *method)
{  RS01Widgets *wl = (RS01Widgets*)method->widgetList;
   fix_closure *fc = g_malloc0(sizeof(fix_closure)); 
   ImageInfo *ii = NULL;
   EccInfo   *ei = NULL;
   EccHeader *eh = NULL;
   unsigned char parity[256];
   int erasure_count,erasure_list[256],erasure_map[256];
   int unexpected_failure;
   gint64 block_idx[256];
   gint64 s,si;
   int i,j,k,n;
   gint64 corrected, uncorrected;
   gint64 last_corrected, last_uncorrected;
   gint64 parity_block = 0;
   int worst_ecc,damaged_ecc,damaged_sec,percent,last_percent = -1;
   int cache_size,cache_sector,cache_offset = 0;
   int local_plot_max;
   char *t = NULL;

   /*** Register the cleanup procedure for GUI mode */

   fc->wl = wl;
   fc->earlyTermination = TRUE;
   RegisterCleanup(_("Repairing of image aborted"), fix_cleanup, fc);

   /*** Open the image and ecc files */

   if(Closure->guiMode)
     SetLabelText(GTK_LABEL(wl->fixHeadline),
		  _("<big>Repairing the image.</big>\n<i>%s</i>"),
		  _("Opening files..."));

   OpenImageAndEcc(&fc->ii, &fc->ei, WRITEABLE_IMAGE | READABLE_ECC);
   ii = fc->ii;
   ei = fc->ei;
   eh = ei->eh;

   /*** Announce what we are going to do. */

   fc->msg = g_strdup_printf(_("Error correction file using Method RS01, %d roots, %4.1f%% redundancy."),
			     eh->eccBytes, 
			     ((double)eh->eccBytes*100.0)/(double)eh->dataBytes);

   if(Closure->guiMode)
   {  SetLabelText(GTK_LABEL(wl->fixHeadline),
		  _("<big>Repairing the image.</big>\n<i>%s</i>"),fc->msg);
      SetFixMaxValues(wl, eh->dataBytes, eh->eccBytes, ii->sectors);
   }    

   PrintLog(_("\nFix mode: Repairable sectors will be fixed in the image.\n"));

   /*** Do some trivial comparisons between the .ecc file and the image file */

   if(ii->sectors > ei->sectors)
   { gint64 diff = ii->sectors - ei->sectors;
     char *trans = _("The image file is %lld sectors longer as noted in the\n"
		     "ecc file. This might simply be zero padding, especially\n"
		     "on dual layer DVD media, but could also mean that\n"
		     "the image and ecc files do not belong together.\n\n%s");

     if(diff>0 && diff<=2)
     {  int answer = ModalDialog(GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, NULL,
				 _("Image file is %lld sectors longer than expected.\n"
				   "Assuming this is a TAO mode medium.\n"
				   "%lld sectors will be removed from the image end.\n"),
				 diff, diff);

        if(!answer)
        {  SwitchAndSetFootline(fc->wl->fixNotebook, 1,
				fc->wl->fixFootline,
				_("<span color=\"red\">Aborted by user request!</span>")); 
	   fc->earlyTermination = FALSE;  /* suppress respective error message */
	   goto terminate;
	}

        ii->sectors -= diff;

        if(!LargeTruncate(ii->file, (gint64)(2048*ii->sectors)))
	  Stop(_("Could not truncate %s: %s\n"),Closure->imageName,strerror(errno));
     }
     
     if(diff>2 && Closure->guiMode)
     {  int answer = ModalDialog(GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, NULL,
				 trans,
				 diff, 
				 _("Is it okay to remove the superfluous sectors?"));

       if(!answer)
       {  SwitchAndSetFootline(fc->wl->fixNotebook, 1,
			       fc->wl->fixFootline,
			       _("<span color=\"red\">Aborted by user request!</span>")); 
	  fc->earlyTermination = FALSE;  /* suppress respective error message */
	  goto terminate;
       }

       ii->sectors -= diff;

       if(!LargeTruncate(ii->file, (gint64)(2048*ii->sectors)))
	 Stop(_("Could not truncate %s: %s\n"),Closure->imageName,strerror(errno));

       PrintLog(_("Image has been truncated by %lld sectors.\n"), diff);
     }

     if(diff>2 && !Closure->guiMode)
     {  if(!Closure->truncate)
	   Stop(trans, 
		diff,
		_("Add the --truncate option to the program call\n"
		  "to have the superfluous sectors removed."));

         ii->sectors -= diff;

	 if(!LargeTruncate(ii->file, (gint64)(2048*ii->sectors)))
	   Stop(_("Could not truncate %s: %s\n"),Closure->imageName,strerror(errno));

	 PrintLog(_("Image has been truncated by %lld sectors.\n"), diff);
     }
   }

   if(ii->sectors < ei->sectors)
   {  int answer;

      answer = ModalWarning(GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, NULL,
			    _("Image file appears to be truncated.\n"
			      "Consider completing it with another reading pass before going on.\n"));
      if(!answer)
      {  SwitchAndSetFootline(fc->wl->fixNotebook, 1,
			      fc->wl->fixFootline,
			      _("<span color=\"red\">Aborted by user request!</span>")); 
	 fc->earlyTermination = FALSE;  /* suppress respective error message */
	 goto terminate;
      }
   }

   if(!memcmp(ii->mediumFP, "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000", 16))
   {  int answer;

      answer = ModalWarning(GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, NULL,
			    _("Sector %d is missing. Can not compare image and ecc fingerprints.\n"
			      "Double check that image and ecc file belong together.\n"),
			    eh->fpSector);
      if(!answer)
      {  SwitchAndSetFootline(fc->wl->fixNotebook, 1,
			      fc->wl->fixFootline,
			      _("<span color=\"red\">Aborted by user request!</span>")); 
	 fc->earlyTermination = FALSE;  /* suppress respective error message */
	 goto terminate;
      }
   }
   else if(memcmp(ii->mediumFP, eh->mediumFP, 16))
          Stop(_("Footprints of image and ecc file do not match.\n"
	         "Image and ecc file do not belong together.\n"));

   /*** Set up the Galois field arithmetic */

   init_tables(eh->eccBytes);

   /*** Prepare buffers for ecc code processing.
	Our ecc blocks are built from ndata medium sectors spread over the full medium size.
        We read cache_size * ndata medium sectors ahead. */

   cache_size = 2*Closure->cacheMB;  /* ndata medium sectors are approx. 0.5MB */

   for(i=0; i<ndata; i++)
   {  fc->imgBlock[i] = g_malloc(cache_size*2048);
      fc->crcBuf[i]   = g_malloc(sizeof(int) * cache_size);
   }

   /*** Setup the block counters for mapping medium sectors to
	ecc blocks */

   s = (ei->sectors+ndata-1)/ndata;

   for(si=0, i=0; i<ndata; si+=s, i++)
     block_idx[i] = si;

   cache_sector = cache_size;  /* forces instant reload of cache */

   /*** Verify ecc information for the medium image. */ 

   corrected = uncorrected = 0;
   last_corrected = last_uncorrected = 0;
   worst_ecc = damaged_ecc = damaged_sec = local_plot_max = 0;

   for(si=0; si<s; si++)
   { 
     if(Closure->stopActions) /* User hit the Stop button */
     {   SwitchAndSetFootline(fc->wl->fixNotebook, 1,
			      fc->wl->fixFootline,
			      _("<span color=\"red\">Aborted by user request!</span>")); 
         fc->earlyTermination = FALSE;  /* suppress respective error message */
	 goto terminate;
     }

     /* Read the next batch of (cache_size * ndata) medium sectors
        if the cache ran empty. */

     if(cache_sector >= cache_size)
     {  
        if(s-si < cache_size)
           cache_size = s-si;
        for(i=0; i<ndata; i++)
        {  int offset = 0;
	   for(j=0; j<cache_size; j++) 
	   {  read_sector(ii, eh, fc->imgBlock[i]+offset, block_idx[i]+j);
	      offset += 2048;
	   }
	   read_crc(ei->file, fc->crcBuf[i], block_idx[i], cache_size);
	}
        cache_sector = cache_offset = 0;
     }

     /* Determine erasures based on the "dead sector" marker */

     erasure_count = 0;
     unexpected_failure = 0;

     for(i=0; i<ndata; i++)
     {  guint32 crc = Crc32(fc->imgBlock[i]+cache_offset, 2048);

        erasure_map[i] = 0;

        if(block_idx[i] < ei->sectors)  /* ignore the padding sectors! */
	{
	  if(!memcmp(fc->imgBlock[i]+cache_offset, Closure->deadSector, 2048))
	  {  erasure_map[i] = 1;
	     erasure_list[erasure_count++] = i;
          }
	  else if(crc != fc->crcBuf[i][cache_sector])
	  {  erasure_map[i] = 3;
	     erasure_list[erasure_count++] = i;
	     PrintCLI(_("CRC error in sector %lld\n"),block_idx[i]);
          }
	}
     }

     if(!erasure_count)  /* Skip completely read blocks */
     {  parity_block+=2048;
        goto skip;
     }
     else
     {  damaged_ecc++;
        damaged_sec+=erasure_count;
     }

     if(erasure_count>worst_ecc)
       worst_ecc = erasure_count;

     if(erasure_count>local_plot_max)
       local_plot_max = erasure_count;

     /* Turn the ndata medium sectors into 2048 ecc blocks
        and try to correct them. */

     if(erasure_count>nroots)   /* uncorrectable */
     {  if(!Closure->guiMode)
	{  PrintCLI(_("* %3d unrepairable sectors: "), erasure_count);

	   for(i=0; i<erasure_count; i++)
	     PrintCLI("%lld ", block_idx[erasure_list[i]]);

	   PrintCLI("\n");
	}

	uncorrected += erasure_count;
	parity_block+=2048;

	/* For truncated images, make sure we leave no "zero holes" in the image
	   by writing the sector(s) with our "dead sector" markers. */

	for(i=0; i<erasure_count; i++)
	{  gint64 idx = block_idx[erasure_list[i]];

	   if(idx < ii->sectors)
	     continue;  /* It's (already) dead, Jim ;-) */

	   if(!LargeSeek(ii->file, (gint64)(2048*idx)))
	     Stop(_("lseek/dead sector failed recovering %s"),Closure->imageName);

	   n = LargeWrite(ii->file, Closure->deadSector, 2048);
	   if(n != 2048)
	     Stop(_("could not write dead sector marker for sector %lld:\n%s"),idx,strerror(errno));
	}
     }
     else  /* try to correct them */
     {  int bi;

        for(bi=0; bi<2048; bi++)
        {  int offset = cache_offset+bi;
	   int r, deg_lambda, el, deg_omega;
	   int u,q,tmp,num1,num2,den,discr_r;
	   int lambda[nroots+1], s[nroots]; /* Err+Eras Locator poly * and syndrome poly */
	   int b[nroots+1], t[nroots+1], omega[nroots+1];
	   int root[nroots], reg[nroots+1], loc[nroots];
	   int syn_error, count;

	   /* Read the parity bytes */

	   if(fseeko(ei->file, (gint64)(sizeof(EccHeader) + ei->sectors*sizeof(guint32) + nroots*parity_block), SEEK_SET) == -1)
	      Stop(_("fseek/sector failed for %s"), Closure->eccName);

	   n = fread(parity, 1, nroots, ei->file);
	   if(n != nroots)
	     Stop(_("Can't read ecc file:\n%s"),strerror(errno));
	   parity_block++;

	   /* Form the syndromes; i.e., evaluate data(x) at roots of g(x) */

	   for(i=0; i<nroots; i++)
	     s[i] = fc->imgBlock[0][offset];

	   for(j=1; j<FIELDMAX; j++)
	   {  int data = j>=ndata ? parity[j-ndata] : fc->imgBlock[j][offset];

	      for(i=0;i<nroots;i++)
              {  if(s[i] == 0) s[i] = data;
	         else s[i] = data ^ gf_alpha_to[mod_fieldmax(gf_index_of[s[i]] + (FIRST_ROOT+i)*PRIM_ELEM)];
	      }
	   }

	   /* Convert syndromes to index form, check for nonzero condition */

	   syn_error = 0;
	   for(i=0; i<nroots; i++)
           {  syn_error |= s[i];
	      s[i] = gf_index_of[s[i]];
	   }

	   /* If it is already correct by coincidence,
	      we have nothing to do any further */

	   if(!syn_error) continue;

	   /* NOTE: Since we already know all our erasure positions, 
	      we could do away simpler than by using the Berlekamp and Chien
	      algorithms.I've left them in this release to have a reference
	      implementation Phil's library code which can be compared
	      against later optimized versions. */

	   /* Init lambda to be the erasure locator polynomial */

	   memset(lambda+1, 0, nroots*sizeof(lambda[0]));
	   lambda[0] = 1;

	   lambda[1] = gf_alpha_to[mod_fieldmax(PRIM_ELEM*(FIELDMAX-1-erasure_list[0]))];
	   for(i=1; i<erasure_count; i++) 
	   {  u = mod_fieldmax(PRIM_ELEM*(FIELDMAX-1-erasure_list[i]));
	      for(j=i+1; j>0; j--) 
	      {  tmp = gf_index_of[lambda[j-1]];
	         if(tmp != ALPHA0)
		   lambda[j] ^= gf_alpha_to[mod_fieldmax(u + tmp)];
	      }
	   }
	
	   for(i=0; i<nroots+1; i++)
	     b[i] = gf_index_of[lambda[i]];
  
	   /* Begin Berlekamp-Massey algorithm to determine error+erasure locator polynomial */

	   r = erasure_count;   /* r is the step number */
	   el = erasure_count;
	   while(++r <= nroots) /* Compute discrepancy at the r-th step in poly-form */
	   {  
	     discr_r = 0;
	     for(i=0; i<r; i++)
	       if((lambda[i] != 0) && (s[r-i-1] != ALPHA0))
		 discr_r ^= gf_alpha_to[mod_fieldmax(gf_index_of[lambda[i]] + s[r-i-1])];

	     discr_r = gf_index_of[discr_r];	/* Index form */

	     if(discr_r == ALPHA0) 
	     {
	        /* B(x) = x*B(x) */
	        memmove(b+1, b, nroots*sizeof(b[0]));
	        b[0] = ALPHA0;
	     } 
	     else 
	     {  /* T(x) = lambda(x) - discr_r*x*b(x) */
	        t[0] = lambda[0];
	        for(i=0; i<nroots; i++) 
		{  if(b[i] != ALPHA0)
		        t[i+1] = lambda[i+1] ^ gf_alpha_to[mod_fieldmax(discr_r + b[i])];
		   else t[i+1] = lambda[i+1];
		}

		if(2*el <= r+erasure_count-1) 
		{  el = r + erasure_count - el;

		   /* B(x) <-- inv(discr_r) * lambda(x) */
		   for(i=0; i<=nroots; i++)
		     b[i] = (lambda[i] == 0) ? ALPHA0 : mod_fieldmax(gf_index_of[lambda[i]] - discr_r + FIELDMAX);
		} 
		else 
		{  /* 2 lines below: B(x) <-- x*B(x) */
		   memmove(b+1, b, nroots*sizeof(b[0]));
		   b[0] = ALPHA0;
		}

		memcpy(lambda,t,(nroots+1)*sizeof(t[0]));
	     }
	   }

	   /* Convert lambda to index form and compute deg(lambda(x)) */
	   deg_lambda = 0;
	   for(i=0; i<nroots+1; i++)
	   {  lambda[i] = gf_index_of[lambda[i]];
	      if(lambda[i] != ALPHA0)
		deg_lambda = i;
	   }

	   /* Find roots of the error+erasure locator polynomial by Chien search */
	   memcpy(reg+1, lambda+1, nroots*sizeof(reg[0]));
	   count = 0;		/* Number of roots of lambda(x) */

	   for(i=1, k=PRIMTH_ROOT-1; i<=FIELDMAX; i++, k=mod_fieldmax(k+PRIMTH_ROOT))
	   {  q=1; /* lambda[0] is always 0 */

	      for(j=deg_lambda; j>0; j--)
	      {  if(reg[j] != ALPHA0) 
		 {  reg[j] = mod_fieldmax(reg[j] + j);
		    q ^= gf_alpha_to[reg[j]];
		 }
	      }

	      if(q != 0) continue; /* Not a root */

	      /* store root (index-form) and error location number */

	      root[count] = i;
	      loc[count] = k;

	      /* If we've already found max possible roots, abort the search to save time */

	      if(++count == deg_lambda) break;
	      //if(++count >= deg_lambda) break;
	   }

	   /* deg(lambda) unequal to number of roots => uncorrectable error detected */

	   if(deg_lambda != count)
	   {  PrintLog(_("Decoder problem (%d != %d) for %d sectors: "), deg_lambda, count, erasure_count);

	      for(i=0; i<erasure_count; i++)
	      {  gint64 idx = block_idx[erasure_list[i]];
	       
                 PrintLog("%lld ", idx);
	      }
	      PrintLog("\n");
	      break;
	   }

	   /* Compute err+eras evaluator poly omega(x) = s(x)*lambda(x) 
	      (modulo x**nroots). in index form. Also find deg(omega). */

	   deg_omega = deg_lambda-1;

	   for(i=0; i<=deg_omega; i++)
	   {  tmp = 0;
	      for(j=i; j>=0; j--)
	      {  if((s[i - j] != ALPHA0) && (lambda[j] != ALPHA0))
		   tmp ^= gf_alpha_to[mod_fieldmax(s[i - j] + lambda[j])];
	      }

	      omega[i] = gf_index_of[tmp];
	   }

	   /* Compute error values in poly-form. 
	      num1 = omega(inv(X(l))), 
              num2 = inv(X(l))**(FIRST_ROOT-1) and 
	      den  = lambda_pr(inv(X(l))) all in poly-form. */

	   for(j=count-1; j>=0; j--)
	   {  num1 = 0;

	      for(i=deg_omega; i>=0; i--) 
	      {  if(omega[i] != ALPHA0)
		    num1 ^= gf_alpha_to[mod_fieldmax(omega[i] + i * root[j])];
	      }

	      num2 = gf_alpha_to[mod_fieldmax(root[j] * (FIRST_ROOT - 1) + FIELDMAX)];
	      den = 0;
    
	      /* lambda[i+1] for i even is the formal derivative lambda_pr of lambda[i] */

	      for(i=min(deg_lambda, nroots-1) & ~1; i>=0; i-=2) 
	      {  if(lambda[i+1] != ALPHA0)
		   den ^= gf_alpha_to[mod_fieldmax(lambda[i+1] + i * root[j])];
	      }

	      /* Apply error to data */

	      if(num1 != 0)
	      {  int location = loc[j];
		
		 if(location >= 0 && location < ndata)
		 {  if(erasure_map[location] == 3)
		    {  int old = fc->imgBlock[location][offset];
		       int new = old ^ gf_alpha_to[mod_fieldmax(gf_index_of[num1] + gf_index_of[num2] + FIELDMAX - gf_index_of[den])];

		       PrintCLI(_("-> Error located in sector %lld at byte %4d (value %2x '%c', expected %2x '%c')\n"),
				block_idx[location], bi, 
				old, isprint(old) ? old : '.',
				new, isprint(new) ? new : '.');
		    }

		    if(!erasure_map[location])
		      PrintLog(_("Unexpected byte error in sector %lld, byte %d\n"),
			       block_idx[location], bi);

		    fc->imgBlock[location][offset] ^= gf_alpha_to[mod_fieldmax(gf_index_of[num1] + gf_index_of[num2] + FIELDMAX - gf_index_of[den])];
		 }
		 else
		   PrintLog(_("Bad error location %d; corrupted .ecc file?\n"), location);
	      }
	   }
	}
     }

     /*** Report if any sectors could be recovered.
	  Write the recovered sectors to the image file .*/

     if(erasure_count && erasure_count<=nroots)
     {  PrintCLI(_("  %3d repaired sectors: "), erasure_count);

        for(i=0; i<erasure_count; i++)
	{  gint64 idx = block_idx[erasure_list[i]];
	   int length;

	   PrintCLI("%lld ", idx);

	   /* Write the recovered sector */

	   if(!LargeSeek(ii->file, (gint64)(2048*idx)))
	     Stop(_("lseek/sector failed recovering %s"),Closure->imageName);

	   if(idx < ei->sectors-1) length = 2048;
	   else length = ii->inLast;  /* error: use inLast calculated from eh->sectors */

	   n = LargeWrite(ii->file, cache_offset+fc->imgBlock[erasure_list[i]], length);
	   if(n != length)
	     Stop(_("could not write medium sector %lld:\n%s"),idx,strerror(errno));
	}

	PrintCLI("\n");
	corrected += erasure_count;
     }

skip:
     /* Advance the cache pointers */

     cache_sector++;
     cache_offset += 2048;

     /* Report progress */

     percent = (1000*si)/s;

     if(last_percent != percent) 
     {  if(Closure->guiMode)
	{  
	   AddFixValues(wl, percent, local_plot_max);
	   local_plot_max = 0;

	   //if(last_corrected != corrected || last_uncorrected != uncorrected) 
	   UpdateFixResults(wl, corrected, uncorrected);
	}
        else PrintProgress(_("Ecc progress: %3d.%1d%%"),percent/10,percent%10);
        last_percent = percent;
     }

     /* Increment the block indices */

     for(i=0; i<ndata; i++)
	block_idx[i]++;
   }

   /*** Print results */

   PrintProgress(_("Ecc progress: 100.0%%\n"));
   if(corrected > 0) PrintLog(_("Repaired sectors: %lld     \n"),corrected);
   if(uncorrected > 0) 
   {  PrintLog(_("Unrepaired sectors: %lld\n"), uncorrected);      
      if(Closure->guiMode)
        SwitchAndSetFootline(wl->fixNotebook, 1, wl->fixFootline,
			     _("Image sectors could not be fully restored "
			       "(%lld repaired; <span color=\"red\">%lld unrepaired</span>)"),
			     corrected, uncorrected);
   }
   else
   {  if(!corrected)
      {    t=_("Good! All sectors are already present.");
           PrintLog("%s\n", t);
      }
      else 
      {    t=_("Good! All sectors are repaired.");
	   PrintLog("%s\n", t);
      }
   }
   if(corrected > 0 || uncorrected > 0)
     PrintLog(_("Erasure counts per ecc block:  avg =  %.1f; worst = %d.\n"),
	     (double)damaged_sec/(double)damaged_ecc,worst_ecc);

   if(Closure->guiMode && t)
     SwitchAndSetFootline(wl->fixNotebook, 1, wl->fixFootline,
			  "%s %s", _("Repair results:"), t);


   /*** Clean up */

   fc->earlyTermination = FALSE;

terminate:
   fix_cleanup((gpointer)fc);
}

/***
 *** Create parity information for the medium sectors.
 ***/

/*
 * Local data package used during encoding
 */

typedef struct
{  Method *self;
   RS01Widgets *wl;
   int earlyTermination;
   unsigned char *data;
   unsigned char *parity;
   char *msg;
   ImageInfo *ii;
   EccInfo *ei;
} ecc_closure;

static void ecc_cleanup(gpointer data)
{  ecc_closure *ec = (ecc_closure*)data;

   Closure->cleanupProc = NULL;

   if(Closure->guiMode)
   {  if(ec->earlyTermination)
        SetLabelText(GTK_LABEL(ec->wl->encFootline),
		     _("<span color=\"red\">Aborted by unrecoverable error.</span>")); 
      AllowActions(TRUE);
   }

   /** Clean up */

   free_tables();

   if(ec->data) g_free(ec->data);
   if(ec->parity) g_free(ec->parity);

   if(ec->msg) g_free(ec->msg);
   if(ec->ii) FreeImageInfo(ec->ii);
   if(ec->ei) FreeEccInfo(ec->ei);

   if(Closure->enableCurveSwitch)
   {  Closure->enableCurveSwitch = FALSE;
      RS01ShowCurveButton(ec->self);
   }

   g_free(ec);

   g_thread_exit(0);
}

/*
 * Create the parity file.
 */

enum { NORMAL, HIGH, GENERIC };

static void create(Method *method)
{  RS01Widgets *wl = (RS01Widgets*)method->widgetList;
   ecc_closure *ec = g_malloc0(sizeof(ecc_closure));
   ImageInfo *ii = NULL;
   EccInfo   *ei = NULL;
   gint64 block_idx[256];  /* must be >= ndata */
   gint64 s,si;
   int i,n;
   int percent = 0,max_percent,progress = 0, last_percent = -1;
   int n_parity_blocks,n_layer_sectors;
   int n_parity_bytes,n_layer_bytes;
   int layer,chunk;
   int loop_type = GENERIC;

   /*** Register the cleanup procedure for GUI mode */

   ec->self = method;
   ec->wl = wl;
   ec->earlyTermination = TRUE;
   RegisterCleanup(_("Error correction file creation aborted"), ecc_cleanup, ec);

   /*** Set up the Galois field arithmetic */

   /* Calculate number of roots (= max. number of erasures)
      and number of data bytes from redundancy setting */

   if(!Closure->redundancy || !strcmp(Closure->redundancy, "normal")) 
	                                          loop_type = NORMAL;
   else if(!strcmp(Closure->redundancy, "high"))  loop_type = HIGH;

   n = calculate_redundancy(Closure->imageName);
   init_tables(n);

   /*** Announce what we are going to do */

   ec->msg = g_strdup_printf(_("Encoding with Method RS01: %d roots, %4.1f%% redundancy."),
			     nroots,
			     ((double)nroots*100.0)/(double)ndata);

   if(Closure->guiMode)
     SetLabelText(GTK_LABEL(wl->encHeadline),
		  _("<big>Creating the error correction file.</big>\n<i>%s</i>"), ec->msg);

   /*** Test the image file and create the CRC sums */

   OpenImageAndEcc(&ec->ii, &ec->ei, READABLE_IMAGE | WRITEABLE_ECC);
   ii = ec->ii;
   ei = ec->ei;
   ScanImage(ii, ei, CREATE_CRC, Closure->guiMode ? wl->encPBar1 : NULL);

   if(ii->sectorsMissing)
   {  fclose(ei->file);  /* Will be deleted anyways; no need to test for errors */
      ei->file = NULL;

      unlink(Closure->eccName);  /* Do not leave a CRC-only .ecc file behind */

      if(Closure->stopActions)   
      {
	SetLabelText(GTK_LABEL(wl->encFootline), 
		     _("<span color=\"red\">Aborted by user request!</span> (partial error correction file removed)")); 
	ec->earlyTermination = FALSE;  /* suppress respective error message */
        goto terminate;
      }
      else 
      {  if(Closure->guiMode)
	  SetProgress(wl->encPBar1, 100, 100);
        
	 Stop(_("%lld sectors unread or missing due to errors.\n"), ii->sectorsMissing);
      }
   }

   if(Closure->guiMode)
   {  SetProgress(wl->encPBar1, 100, 100);
      ShowWidget(wl->encPBar2);
      ShowWidget(wl->encLabel2);
   }

   if(!Closure->guiMode)
     PrintLog("%s\n",ec->msg);

   /*** Prepare Ecc file header.
        The .eccSum will be filled in after all ecc blocks have been created. */

   //   memset(&ei->eh,0,sizeof(EccHeader));
   //   strncpy(ei->eh->cookie, "*dvdisaster*", 12);
   //   strncpy(ei->eh->method, "RS01", 4);
   memcpy(ei->eh->cookie, "*dvdisaster*", 12);
   memcpy(ei->eh->method, "RS01", 4);
   ei->eh->method_flags[0] = 1;
   gint64_to_uchar(ei->eh->sectors, ii->sectors);
   ei->eh->dataBytes       = ndata;
   ei->eh->eccBytes        = nroots;

   ei->eh->creatorVersion  = Closure->version;
   ei->eh->neededVersion   = 5500;
   ei->eh->fpSector        = FOOTPRINT_SECTOR;

   memcpy(ei->eh->mediumFP, ii->mediumFP, 16);
   memcpy(ei->eh->mediumSum, ii->mediumSum, 16);

   if(fseeko(ei->file, (gint64)sizeof(EccHeader) + ii->sectors*sizeof(guint32), SEEK_SET) == -1)
	Stop(_("fseek failed skipping ecc+crc header: %s"),strerror(errno));

   /*** Allocate buffers for the parity calculation and image data caching. 

        The algorithm builds the parity file consecutively in chunks of n_parity_blocks.
        We use all the amount of memory allowed by cacheMB for caching the parity blocks. */

   n_parity_blocks = (Closure->cacheMB<<20) / nroots;
   n_parity_blocks &= ~0x7ff;                   /* round down to multiple of 2048 */
   n_parity_bytes  = nroots * n_parity_blocks;

   /* Each chunk of parity blocks is built iteratively by processing the data in layers
      (first all bytes at pos 0, then pos 1, until ndata layers have been processed).
      So one buffer of n_layer_bytes = n_parity_blocks needs to be buffered.
      For practical reasons we require that the layer size is a multiple of the
      medium sector size of 2048 bytes. */

   n_layer_bytes   = n_parity_blocks;
   n_layer_sectors = n_parity_blocks/2048;

   if(n_layer_sectors*2048 != n_parity_blocks)
     Stop(_("Internal error: parity blocks are not a multiple of sector size.\n"));

   ec->parity = g_malloc(n_parity_bytes);
   ec->data   = g_malloc(n_layer_bytes);

   /*** Setup the block counters for mapping medium sectors to ecc blocks 
        The image is divided into ndata sections;
        with each section spanning s sectors. */

   s = (ii->sectors+ndata-1)/ndata;

   for(si=0, i=0; i<ndata; si+=s, i++)
     block_idx[i] = si;

   /*** Create ecc information for the medium image. */ 

   max_percent = ndata * ((s / n_layer_sectors) + 1);

   /* Process the image.
      From each section a chunk of n_layer_sectors is read in at once.
      So after (s/n_layer_sectors)+1 iterations the whole image has been processed. */

   for(chunk=0; chunk<s; chunk+=n_layer_sectors) 
   {  int actual_layer_bytes,actual_layer_sectors;

      /* Prepare the parity data for the next chunk. */

      memset(ec->parity, 0, n_parity_bytes);

      /* The last chunk may contain fewer sectors. */

      if(chunk+n_layer_sectors < s)
           actual_layer_sectors = n_layer_sectors;
      else actual_layer_sectors = s-chunk;

      actual_layer_bytes   = 2048*actual_layer_sectors;

      /* Work each of the ndata data layers 
	 into the parity data of the current chunk. */

      switch(loop_type)
      { case NORMAL:  /* Inner loop unrolled for nroots = 32. */
	{int sp=1;    /* sp==1 makes sure sp==0 after ndata bytes [since (223+1) mod 32 = 0]*/
  
         for(layer=0; layer<ndata; layer++)
	 {  int offset = 0;
            unsigned char *par_idx = ec->parity;

	    if(Closure->stopActions) /* User hit the Stop button */
	    {  SetLabelText(GTK_LABEL(wl->encFootline), 
			    _("<span color=\"red\">Aborted by user request!</span> (partial error correction file removed)")); 
	       ec->earlyTermination = FALSE;  /* suppress respective error message */
	       fclose(ei->file);
	       ei->file = NULL;
	       unlink(Closure->eccName);  /* Do not leave partial .ecc file behind */
	       goto terminate;
	    }

	    /* Read the next data sectors of this layer. */

	    for(i=0; i<actual_layer_sectors; i++)
	    {  read_sector(ii, ei->eh, ec->data+offset, block_idx[layer]);
	       block_idx[layer]++;
	       offset += 2048;
	    }

	    /* Now process the data bytes of the current layer. */

	    for(i=0; i<actual_layer_bytes; i++)
	    {  register int feedback;

	       feedback = gf_index_of[ec->data[i] ^ par_idx[sp]];

	       if(feedback != ALPHA0) /* non-zero feedback term */
	       {  register int spk = sp;

                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 249];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  59];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  66];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +   4];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  43];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 126];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 251];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  97];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  30];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +   3];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 213];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  50];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  66];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 170];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +   5];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  24];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +   5];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 170];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  66];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  50];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 213];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +   3];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  30];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  97];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 251];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 126];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  43];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +   4];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  66];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback +  59];
                  par_idx[((++spk)&31)] ^= enc_alpha_to[feedback + 249];

		  par_idx[sp] = enc_alpha_to[feedback];  /* feedback + 0 */
	       }
	       else                   /* zero feedback term */
		 par_idx[sp] = 0;

	       par_idx += nroots;
	    }

	    sp = (sp+1) & 31;         /* shift */

	    /* Report progress */

	    progress++;
	    percent = (1000*progress)/max_percent;
	    if(last_percent != percent) 
	    {  if(Closure->guiMode)
	          SetProgress(wl->encPBar2, percent, 1000);
	       else
	          PrintProgress(_("Ecc generation: %3d.%1d%%"), percent/10, percent%10);
	       last_percent = percent;
	    }
	 }
	}
	break;

        case HIGH: /* Inner loop is unrolled for nroots = 64. */
	{int sp=1; /* sp==1 makes sure sp==0 after ndata bytes [since (191+1) mod 64 = 0] */
         for(layer=0; layer<ndata; layer++)
	 {  int offset = 0;
	    unsigned char *par_idx = ec->parity;

	    if(Closure->stopActions) /* User hit the Stop button */
	    {  SetLabelText(GTK_LABEL(wl->encFootline), 
			    _("<span color=\"red\">Aborted by user request!</span> (partial error correction file removed)")); 
	       ec->earlyTermination = FALSE;  /* suppress respective error message */
	       fclose(ei->file);
	       ei->file = NULL;
	       unlink(Closure->eccName);  /* Do not leave partial .ecc file behind */
	       goto terminate;
	    }

	    /* Read the next data sectors of this layer. */

	    for(i=0; i<actual_layer_sectors; i++)
	    {  read_sector(ii, ei->eh, ec->data+offset, block_idx[layer]);
	       block_idx[layer]++;
	       offset += 2048;
	    }

	    /* Now process the data bytes of the current layer. */

	    for(i=0; i<actual_layer_bytes; i++)
	    {  register int feedback;

	       feedback = gf_index_of[ec->data[i] ^ par_idx[sp]];

	       if(feedback != ALPHA0) /* non-zero feedback term */
	       {  register int spk = sp;

	          par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[63]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[62]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[61]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[60]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[59]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[58]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[57]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[56]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[55]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[54]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[53]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[52]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[51]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[50]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[49]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[48]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[47]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[46]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[45]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[44]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[43]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[42]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[41]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[40]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[39]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[38]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[37]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[36]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[35]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[34]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[33]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[32]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[31]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[30]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[29]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[28]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[27]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[26]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[25]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[24]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[23]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[22]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[21]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[20]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[19]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[18]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[17]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[16]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[15]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[14]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[13]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[12]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[11]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[10]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 9]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 8]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 7]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 6]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 5]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 4]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 3]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 2]];
		  par_idx[((++spk)&63)] ^= enc_alpha_to[feedback + gf_gpoly[ 1]];

		  par_idx[sp] = enc_alpha_to[feedback + gf_gpoly[0]];
	       }
	       else                   /* zero feedback term */
		 par_idx[sp] = 0;

	       par_idx += nroots;
	    }

	    sp = (sp+1) & 63;         /* shift */

	    /* Report progress */

	    progress++;
	    percent = (1000*progress)/max_percent;
	    if(last_percent != percent) 
	    {  if(Closure->guiMode)
	          SetProgress(wl->encPBar2, percent, 1000);
	       else
	          PrintLog(_("Ecc generation: %3d.%1d%%"), percent/10, percent%10);
	       last_percent = percent;
	    }
	 }
	}
	break;

        default:   /* general case for nroots other than 32 or 64 */
	{int sp = nroots - ndata % nroots; /* => (ndata + sp) mod nroots = 0 so that parity */
	                                   /* is aligned at sp=0 after ndata iterations */
      	 if(sp==nroots) sp=0;

	 for(layer=0; layer<ndata; layer++)
	 {  int offset = 0;
            unsigned char *par_idx = ec->parity;

	    if(Closure->stopActions) /* User hit the Stop button */
	    {  SetLabelText(GTK_LABEL(wl->encFootline), 
			    _("<span color=\"red\">Aborted by user request!</span>")); 
	       ec->earlyTermination = FALSE;  /* suppress respective error message */
	       fclose(ei->file);
	       ei->file = NULL;
	       unlink(Closure->eccName);  /* Do not leave partial .ecc file behind */
	       goto terminate;
	    }

            /* Read the next data sectors of this layer. */

   	    for(i=0; i<actual_layer_sectors; i++)
	    {  read_sector(ii, ei->eh, ec->data+offset, block_idx[layer]);
	       block_idx[layer]++;
	       offset += 2048;
	    }

	    /* Now process the data bytes of the current layer. */

	    for(i=0; i<actual_layer_bytes; i++)
	    {  register int feedback;

	       feedback = gf_index_of[ec->data[i] ^ par_idx[sp]];

	       if(feedback != ALPHA0) /* non-zero feedback term */
	       {  register int spk = sp+1;
		  register int *gpoly = gf_gpoly + nroots;

		  switch(nroots-spk)
		  {  
		    case 110: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 109: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 108: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 107: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 106: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 105: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 104: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 103: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 102: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 101: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 100: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 99: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 98: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 97: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 96: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 95: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 94: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 93: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 92: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 91: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 90: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 89: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 88: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 87: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 86: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 85: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 84: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 83: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 82: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 81: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 80: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 79: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 78: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 77: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 76: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 75: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 74: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 73: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 72: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 71: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 70: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 69: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 68: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 67: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 66: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 65: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 64: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 63: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 62: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 61: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 60: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 59: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 58: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 57: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 56: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 55: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 54: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 53: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 52: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 51: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 50: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 49: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 48: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 47: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 46: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 45: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 44: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 43: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 42: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 41: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 40: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 39: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 38: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 37: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 36: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 35: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 34: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 33: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 32: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 31: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 30: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 29: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 28: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 27: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 26: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 25: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 24: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 23: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 22: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 21: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 20: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 19: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 18: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 17: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 16: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 15: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 14: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 13: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 12: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 11: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 10: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  9: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  8: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  7: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  6: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  5: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  4: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  3: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  2: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  1: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		  }

		  spk = 0;
		  
		  switch(sp)
		  {
                    case 110: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 109: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 108: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 107: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 106: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 105: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 104: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 103: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 102: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 101: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		    case 100: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 99: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 98: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 97: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 96: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 95: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 94: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 93: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 92: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 91: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 90: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 89: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 88: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 87: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 86: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 85: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 84: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 83: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 82: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 81: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 80: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 79: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 78: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 77: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 76: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 75: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 74: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 73: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 72: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 71: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 70: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 69: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 68: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 67: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 66: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 65: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 64: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 63: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 62: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 61: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 60: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 59: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 58: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 57: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 56: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 55: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 54: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 53: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 52: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 51: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 50: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 49: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 48: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 47: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 46: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 45: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 44: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 43: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 42: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 41: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 40: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 39: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 38: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 37: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 36: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 35: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 34: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 33: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 32: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 31: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 30: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 29: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 28: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 27: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 26: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 25: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 24: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 23: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 22: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 21: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 20: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 19: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 18: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 17: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 16: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 15: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 14: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 13: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 12: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 11: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case 10: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  9: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  8: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  7: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  6: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  5: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  4: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  3: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  2: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		     case  1: par_idx[spk++] ^= enc_alpha_to[feedback + *--gpoly];
		  }

		  par_idx[sp] = enc_alpha_to[feedback + gf_gpoly[0]];
	       }
	       else                   /* zero feedback term */
		 par_idx[sp] = 0;

	       par_idx += nroots;
	    }

	    if(++sp>=nroots) sp=0;   /* shift */

	    /* Report progress */

	    progress++;
	    percent = (1000*progress)/max_percent;
	    if(last_percent != percent) 
	    {  if(Closure->guiMode)
	          SetProgress(wl->encPBar2, percent, 1000);
	       else
	          PrintLog(_("Ecc generation: %3d.%1d%%"), percent/10, percent%10);
	       last_percent = percent;
	    }
	 }
	}
	break;
      }

      /* Write the nroots bytes of parity information */

      n = fwrite(ec->parity, nroots, actual_layer_bytes, ei->file);

      if(n != actual_layer_bytes)
        Stop(_("could not write to ecc file \"%s\":\n%s"),Closure->eccName,strerror(errno));

      MD5Update(&ei->md5Ctxt, ec->parity, nroots*actual_layer_bytes);
   }

   /*** Complete the ecc header and write it out */

   MD5Final(ei->eh->eccSum, &ei->md5Ctxt);

   rewind(ei->file);
   n = fwrite(ei->eh, sizeof(EccHeader), 1, ei->file);
   if(n != 1)
     Stop(_("Can't write ecc header:\n%s"),strerror(errno));

   if(fclose(ei->file))
     Stop(_("Error closing error correction file:\n%s"), strerror(errno));
   ei->file = NULL;

   PrintProgress(_("Ecc generation: 100.0%%\n"));
   PrintLog(_("Error correction file \"%s\" created.\n"
	       "Make sure to keep this file on a reliable medium.\n"),
	     Closure->eccName);
   
   if(Closure->guiMode)
   {  SetProgress(wl->encPBar2, 100, 100);

      SetLabelText(GTK_LABEL(wl->encFootline), 
		   _("The error correction file has been successfully created.\n"
		     "Make sure to keep this file on a reliable medium."),
		   Closure->eccName); 
   }

   /*** If the --unlink option or respective GUI switch is set, 
	unlink the image.
	Windows can not unlink until all file handles are closed. Duh. */

   if(Closure->unlinkImage)
   {  if(ec->ii) FreeImageInfo(ec->ii);
      ec->ii = NULL;
      UnlinkImage(Closure->guiMode ? wl->encFootline2 : NULL);
   }

   /*** Clean up */

   ec->earlyTermination = FALSE;

terminate:
   ecc_cleanup((gpointer)ec);
}
