/////////////////////////////////////////////////////////////////////////////
// Name:        Cache.h
// Purpose:     Manage cache of transcoded files
// Author:      Alex Thuering
// Created:     29.08.2008
// RCS-ID:      $Id: Cache.h,v 1.6 2014/09/20 06:04:42 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#ifndef DVDSTYLER_CACHE_H
#define DVDSTYLER_CACHE_H

#include "DVD.h"
#include "mediaenc.h"
#include <wx/hashmap.h>
#include <wx/window.h>

class CacheEntry;
WX_DECLARE_STRING_HASH_MAP(CacheEntry*, CacheMap);

/**
 * Manage cache of transcoded files
 */
class Cache {
public:
	/**
	 * Constructor
	 */
	Cache();
	
	/**
	 * Destructor
	 */
	~Cache();
	
	/**
	 * Returns true if the cache is already initialized
	 */
	bool IsInitialized();
	
	/**
	 * Sets the temporally directory for cache files
	 */
	bool SetTempDir(wxString tempDir);
	
	/**
	 * Returns temporally directory
	 */
	wxString GetTempDir();
	
	/**
	 * Return count of entries in cache
	 */
	int GetCount();
	
	/**
	 * Return size of files in cache (in KB)
	 */
	long GetSize();
	
	/**
	 * Checks if there is a file for given VOB in the cache
	 * @return The file name in cache or empty string if no match entry in cache
	 */
	wxString Find(Vob* vob, DVD* dvd, bool removeOld = true);
	
	/**
	 * Adds an entry for given VOB to the cache
	 * @return The file name in cache
	 */
	wxString Add(Vob* vob, DVD* dvd);
	
	/**
	 * Removes an entry for given VOB from the cache
	 */
	void Remove(Vob* vob);
	
	/**
	 * Removes all files from cache
	 */
	void Clear();
	
	/**
	 * Begins checking for unused entries
	 */
	void BeginClean();
	
	/**
	 * Removes all unused entries
	 */
	void EndClean();
	
	/**
	 * Saves list of cache entries
	 */
	void Save();
	
	/**
	 * Shows prompt dialog and clears or saves cache 
	 */
	void ShowClearPrompt(wxWindow* parent);

private:
	wxString m_dir;
	CacheMap m_cacheMap;
	
	/**
	 * @return The file name of file in cache with given index
	 */
	wxString GetFileName(int index);
	
	/**
	 * @return The index for new cache entry
	 */
	int GenerateIndex();
	
	/**
	 * Loads list of cache entries
	 */
	bool Open();
};

#endif //DVDSTYLER_CACHE_H
