/*  $Header: /cvsroot/dvipdfmx/src/psspecial.c,v 1.7 2004/03/16 09:33:51 hirata Exp $
    
    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#include <stdlib.h>
#include "system.h"
#include "mem.h"
#include "error.h"
#include "mfileio.h"

#include "psspecial.h"
#include "pdfparse.h"
#include "pdfspecial.h"

#include "pdfdoc.h"

#include "psimage.h"
#include "mpost.h"

#include "pdfximage.h"

#define HOFFSET 1
#define VOFFSET 2
#define HSIZE   3
#define VSIZE   4
#define HSCALE  5
#define VSCALE  6
#define ANGLE   7
#define CLIP    8
#define LLX     9
#define LLY    10
#define URX    11
#define URY    12
#define RWI    13
#define RHI    14

static struct keys
{
  const char *key;
  int id;
} keys[] = {
  {"hoffset", HOFFSET},
  {"voffset", VOFFSET},
  {"hsize",   HSIZE},
  {"vsize",   VSIZE},
  {"hscale",  HSCALE},
  {"vscale",  VSCALE},
  {"angle",   ANGLE},
  {"clip",    CLIP},
  {"llx",     LLX},
  {"lly",     LLY},
  {"urx",     URX},
  {"ury",     URY},
  {"rwi",     RWI},
  {"rhi",     RHI},
};

static int
guess_key (const char *key)
{
  int i;

  for (i = 0; i < sizeof(keys)/sizeof(keys[0]); i++) {
    if (!strcmp (key, keys[i].key))
      break;
  }

  if (i == sizeof(keys)/sizeof(keys[0]))
    ERROR("Unknown key in special: %s", key);

  return keys[i].id;
}

static int
parse_psfile (char **start, char *end, double x_user, double y_user) 
{
  int   xobj_id;
  char *key, *val;
  char *filename = NULL;
  double hoffset = 0.0, voffset = 0.0;
  double hsize   = 0.0, vsize   = 0.0;
  transform_info p;

  skip_white(start, end);
  parse_key_val(start, end, &key, &filename);
  if (!key || !filename) {
    WARN("PSfile special has no filename.");
    if (key)
      RELEASE(key);
    if (filename)
      RELEASE(filename);
    return -1;
  }
  RELEASE(key);

  xobj_id = pdf_ximage_findresource(filename);
  RELEASE(filename);
  if (xobj_id < 0) {
    WARN("Failed to open image file: %s", filename);
    return -1;
  }

  transform_info_clear(&p);
  skip_white(start, end);
  while (*start < end) {
    int key_id;

    parse_key_val(start, end, &key, &val);
    if (!key) {
      WARN("Invalid keyword in PSfile special.");
      if (val)
	RELEASE(val);
      return -1;
    }

    key_id = guess_key(key);
    RELEASE(key);

    if (!val) {
      switch (key_id) {
      case CLIP:
	p.clip = 1;
	break;
      default:
	WARN("Unknown PSfile token \"%s\".", key);
	return -1;
      }
      continue;
    }
    if (!is_a_number(val)) {
      WARN("PSfile key \"%s\" assigned nonnumeric value.", key);
      RELEASE(val);
      return -1;
    }
    switch (key_id) {
      case HOFFSET: hoffset = atof(val); break;
      case VOFFSET: voffset = atof(val); break;
      case HSIZE:   hsize = atof(val);   break;
      case VSIZE:   vsize = atof(val);   break;
      case HSCALE:
	p.xscale = atof(val) / 100.0;
	break;
      case VSCALE:
	p.yscale = atof(val) / 100.0;
	break;
      case ANGLE:
	p.rotate = atof(val) * M_PI / 180.0;
	break;
      case LLX:
	p.user_bbox = 1;
	p.u_llx = atof(val);
	break;
      case LLY:
	p.user_bbox = 1;
	p.u_lly = atof(val);
	break;
      case URX:
	p.user_bbox = 1;
	p.u_urx = atof(val);
	break;
      case URY:
	p.user_bbox = 1;
	p.u_ury = atof(val);
	break;
      case RWI:
	p.width  = atof(val)/10.0;
	break;
      case RHI:
	p.height = atof(val)/10.0;
	break;
      default:
	ERROR("PSfile key \"%s=%s\" not recognized.", key, val);
      }
    RELEASE(val);
    skip_white(start, end);
  }

  if (*start < end) {
    WARN("Unparsed tokens in ps special.");
    return -1;
  }
  pdf_ximage_put_image(xobj_id, &p, x_user, y_user);

  return 0;
}

static void
do_texfig (char **start, char *end, double ref_x, double ref_y)
{
  int   xobj_id;
  char *filename;
  transform_info p;

  skip_white(start, end);
  filename = parse_val_ident(start, end);
  if (!filename) {
    WARN("Expecting filename here:");
    dump(*start, end);
    return;
  }

  xobj_id = pdf_ximage_findresource(filename);
  if (xobj_id < 0)
    WARN("Could not find image \"%s\".", filename);
  else {
    transform_info_clear(&p);
    p.yscale = -1.0; p.xscale = 1.0;
    pdf_ximage_put_image(xobj_id, &p, 0, 0);
  }
  RELEASE(filename);
}

static int
do_postscriptbox (char **start, char *end, double x_user, double y_user)
{
  int  xobj_id;
  transform_info p;
  char filename[256];
  double width, height;

  *start += 13;
  if (sscanf(*start, "{%lfpt}{%lfpt}{%256[^}]}",
	     &width, &height, filename) != 3) {
    WARN("Unrecogzied postscriptbox special.");
    return 0;
  }

  transform_info_clear(&p);

  p.width  = width  * 72.0 /72.27;
  p.height = height * 72.0 /72.27;
  p.user_bbox = 0;

  xobj_id = pdf_ximage_findresource(filename);
  pdf_ximage_put_image(xobj_id, &p, x_user, y_user);

  return 1;
}

int
ps_parse_special (char *buffer, UNSIGNED_QUAD size, double x_user,
		  double y_user)
{
  char *start = buffer, *end;
  static int    block_pending = 0;
  static double pending_x = 0.0, pending_y = 0.0;
  int    result = 0;

  end = buffer + size;

  skip_white (&start, end);
  if (!strncmp(start, "PSfile", strlen("PSfile")) ||
      !strncmp(start, "psfile", strlen("PSfile"))) {
    result = 1; /* This means it is a PSfile special, not that it was
		   successful */
    parse_psfile(&start, end, x_user, y_user);
  } else if (!strncmp (start, "ps::[begin]", strlen("ps::[begin]"))) {
    start += strlen("ps::[begin]");
    block_pending = 1;
    pending_x = x_user;
    pending_y = y_user;
    result = 1; /* Likewise */
    do_raw_ps_special (&start, end, 1, x_user, y_user);
  } else if (!strncmp (start, "ps::[end]", strlen("ps::[end]"))) {
    if (!block_pending)
      WARN("ps::[end] without ps::[begin] ignored.");
    else {
      start += strlen("ps::[end]");
      do_raw_ps_special(&start, end, 1, pending_x, pending_y);
      block_pending = 0;
    }
    result = 1; /* Likewise */
  } else if (!strncmp(start, "ps: plotfile", strlen("ps: plotfile"))) {
    /* This is a bizarre, ugly  special case.. Not really postscript code */
    start += strlen ("ps: plotfile");
    result = 1;
    do_texfig(&start, end,
	      block_pending ? pending_x : x_user, block_pending ? pending_y : y_user);
  } else if (!strncmp (start, "ps::", strlen("ps::")) ||
	     !strncmp (start, "PS::", strlen("PS::"))) {
    /* dvipdfm doesn't distinguish between ps:: and ps: */
    start += 4;
    result = 1; /* Likewise */
    do_raw_ps_special(&start, end, 1, 
		      block_pending ? pending_x : x_user, block_pending ? pending_y : y_user);
  } else if (!strncmp (start, "ps:", strlen("ps:")) ||
	     !strncmp (start, "PS:", strlen("PS:"))) {
    start += 3;
    result = 1; /* Likewise */
    do_raw_ps_special(&start, end, 1,
		      block_pending ? pending_x: x_user, block_pending ? pending_y : y_user);
  } else if (!strncmp(start, "postscriptbox", strlen("postscriptbox"))) {
    result = do_postscriptbox(&start, end, x_user, y_user);
  }

  return result;
}

